<?php

namespace ZeroV99\ShipmentTracking\helpers;

if (! defined('ABSPATH')) {
    exit;
}

use WP_Error;

/**
 * Handles validation for shipment tracking form fields.
 *
 * This class provides static methods to validate various fields such as dates, tracking numbers,
 * carrier names, carrier urls and customer emails. It ensures that the data meets specific requirements
 * and returns appropriate error messages if validation fails.
 */
class ZeroV99_Shipment_Tracking_Form_Validator
{

    /**
     * Validates a date to ensure it is not empty, in a valid format, and not in the past.
     *
     * @param string $date The date to validate.
     * @return true|WP_Error Returns `true` if the date is valid, or a `WP_Error` object if validation fails.
     */
    public static function validate_date($date)
    {
        if (empty($date)) {
            return new WP_Error('zerov99_empty_date', __('The date field is required.', 'zerov99-shipment-tracking'));
        }

        $timestamp = strtotime($date);
        if ($timestamp === false) {
            return new WP_Error('zerov99_invalid_date_format', __('Invalid date format.', 'zerov99-shipment-tracking'));
        }

        return true;
    }

    /**
     * Validates a tracking number to ensure it is not empty and matches the required format.
     *
     * @param string $tracking_number The tracking number to validate.
     * @return true|WP_Error Returns `true` if the tracking number is valid, or a `WP_Error` object if validation fails.
     */
    public static function validate_tracking_number($tracking_number)
    {
        if (empty($tracking_number)) {
            return new WP_Error('zerov99_empty_tracking', __('Tracking number is required.', 'zerov99-shipment-tracking'));
        }
        if (! preg_match('/^[a-zA-Z0-9_-]{8,30}$/', $tracking_number)) {
            return new WP_Error('zerov99_invalid_tracking', __('Invalid tracking number. It must be 8-30 characters long and contain only letters, numbers, underscore and hyphens.', 'zerov99-shipment-tracking'));
        }
        return true;
    }

    /**
     * Validates a carrier name to ensure it is not empty and is one of the allowed carriers.
     *
     * @param string $carrier_name The carrier name to validate.
     * @return true|WP_Error Returns `true` if the carrier name is valid, or a `WP_Error` object if validation fails.
     */
    public static function validate_carrier_name($carrier_name)
    {
        if (empty($carrier_name)) {
            return new WP_Error('zerov99_empty_carrier', __('Carrier name is required.', 'zerov99-shipment-tracking'));
        }

        $allowed_carriers = array_column(get_option(ZeroV99_Shipment_Tracking_Config::OPTION_SHIPPING_PROVIDERS, []), 'name');

        if (! in_array($carrier_name, $allowed_carriers)) {
            return new WP_Error('zerov99_invalid_carrier', __('Invalid carrier selected.', 'zerov99-shipment-tracking'));
        }

        return true;
    }

    /**
     * Validates the customer email if the "Send Email" option is enabled.
     *
     * @param string $email      The customer email to validate.
     * @param string $send_email The value of the "Send Email" checkbox (e.g., "yes" if enabled).
     * @return true|WP_Error Returns `true` if the email is valid or if email sending is disabled, or a `WP_Error` object if validation fails.
     */
    public static function validate_email_send($email, $send_email)
    {
        if ($send_email !== 'yes') {

            return true;
        }
        if (empty($email) || ! is_email($email)) {
            return new WP_Error('zerov99_invalid_email', __('Order not updated. Email not sent. Invalid customer email. Please ensure the customer email address is provided.', 'zerov99-shipment-tracking'));
        }
        return true;
    }

    /**
     * Validates the provider name.
     *
     * @param string $name The provider name to validate.
     * @return string|WP_Error The cleaned name or a WP_Error object if validation fails.
     */
    public static function validate_settings_provider_name($name)
    {
        $name = trim($name);

        if (empty($name)) {
            return new WP_Error('zerov99_empty_provider_name', __('Provider name is required.', 'zerov99-shipment-tracking'));
        }

        // Combined check: ensure the name is at least 2 characters long and contains only alphanumeric characters.
        if (!preg_match('/^(?=.{2,}$)[a-zA-Z0-9]+$/', $name)) {
            return new WP_Error(
                'zerov99_invalid_provider_name',
                __('Provider name must be at least 2 characters long and contain only alphanumeric characters.', 'zerov99-shipment-tracking')
            );
        }

        return $name;
    }

    /**
     * Validates a URL that can either be a plain URL (Example 2) or a URL with a query 
     * that may include a placeholder (Example 1). In the plain URL, no "?" or curly braces "{}" 
     * are allowed. In the URL with a query, the path (before "?") must not contain "?" or "{}",
     * while the query string is validated parameter by parameter.
     *
     * Valid examples:
     * - Plain URL: 
     *    https://www.dhl.com/us-en/home/tracking.html
     * - URL with query placeholder (placeholder embedded anywhere in the parameter value):
     *    https://www.mydhli.com/global-en/home/tracking.html?tracking-id={tracking_code}/mis-envios
     *    https://www.dhl.com/us-en/home/tracking.html?tracking-id=prefix{tracking_code}suffix&lang=en
     *
     * @param string $url The URL to validate.
     * @return string|WP_Error The cleaned URL or a WP_Error object if validation fails.
     */
    public static function validate_settings_provider_url($url)
    {
        $url = trim($url);

        if (empty($url)) {
            return new WP_Error('zerov99_empty_url', __('Provider URL is required.', 'zerov99-shipment-tracking'));
        }

        // Case 1: URL without query string: must not include "?" or "{}" anywhere.
        if (strpos($url, '?') === false) {
            $patternPlain = '/^(https?:\/\/)'               // Protocol (http:// or https://)
                . '(([\w\-]+\.)+[\w\-]+)'           // Domain (with subdomains allowed)
                . '(\/[^\s?{}]*)?$/';               // Optional path with no "?" or "{}"
            if (!preg_match($patternPlain, $url)) {
                return new WP_Error('zerov99_invalid_provider_url', __('Invalid URL. Please review the tooltip help available next to the Tracking URL column title.', 'zerov99-shipment-tracking'));
            }
        } else {
            // Case 2: URL with query string.
            // Validate that the non-query part (protocol, domain, and path) does not contain "?" or "{}".
            // Then, validate the query string parameters.
            // The pattern breakdown:
            //   - ^(https?:\/\/)                         => Protocol.
            //   - (([\w\-]+\.)+[\w\-]+)                   => Domain.
            //   - (\/[^\s?{}]*)?                         => Optional path without "?" or curly braces.
            //   - \?                                     => Query string marker.
            //   - Then one or more query parameters, separated by "&", where each parameter is:
            //         [A-Za-z0-9\-_]+                    => Parameter name.
            //         =                                  => Equal sign.
            //         Then the value, which can be either:
            //              a) A string without curly braces: [A-Za-z0-9\-_\/]+
            //              b) Or a string that contains a placeholder:
            //                 Any allowed characters, then a placeholder \{[A-Za-z0-9\-_]+\} and then any allowed characters.
            //         Allowed characters in values here are alphanumerics, hyphen, underscore, and forward slash.
            //
            // Note: This regex allows only one pair of curly braces per parameter value if used.
            $patternWithQuery = '/^(https?:\/\/)'                            // Protocol
                . '(([\w\-]+\.)+[\w\-]+)'                      // Domain
                . '(\/[^\s?{}]*)?'                             // Optional path (no "?" or "{}")
                . '\?'                                        // Query string marker
                . '(?:'                                       // Start first query parameter
                .     '[A-Za-z0-9\-_]+'                        // Parameter name
                .     '='
                .     '(?:'                                   // Parameter value alternatives:
                .         '(?:[A-Za-z0-9\-_\/]*\{[A-Za-z0-9\-_]+\}[A-Za-z0-9\-_\/]*)' // value with placeholder
                .         '|'
                .         '(?:[A-Za-z0-9\-_\/]+)'            // or value without any curly braces
                .     ')'
                . ')'
                . '(?:&'                                      // Additional query parameters:
                .     '[A-Za-z0-9\-_]+='
                .     '(?:'
                .         '(?:[A-Za-z0-9\-_\/]*\{[A-Za-z0-9\-_]+\}[A-Za-z0-9\-_\/]*)'
                .         '|'
                .         '(?:[A-Za-z0-9\-_\/]+)'
                .     ')'
                . ')*'
                . '$/';

            if (!preg_match($patternWithQuery, $url)) {
                return new WP_Error('zerov99_invalid_provider_url', __('Invalid URL. Please review the tooltip help available next to the Tracking URL column title.', 'zerov99-shipment-tracking'));
            }
        }

        return $url;
    }
}
