<?php
namespace ZeroV99\ShipmentTracking\helpers;

if (! defined('ABSPATH')) {
    exit;
}

/**
 * Handles cleanup of shipment tracking data when order status changes.
 *
 * This class ensures that shipment tracking data is cleared or reset when an order's status
 * changes to a status other than "shipped" or "completed". It also preserves tracking data
 * for orders that were previously marked as "shipped" before the plugin was deactivated.
 */
class ZeroV99_Shipment_Tracking_Data_Cleanup
{

    /**
     * Constructor.
     *
     * Hooks into the WooCommerce order status change event to trigger data cleanup.
     */
    public function __construct()
    {

        add_action('woocommerce_order_status_changed', [$this, 'delete_tracking_data_on_status_change'], 10, 3);
    }

    /**
     * Cleans up shipment tracking data when the order status changes.
     *
     * This function clears or resets shipment tracking data if the new order status is neither
     * "shipped" nor "completed". It preserves tracking data for orders that were previously
     * marked as "shipped" before the plugin was deactivated.
     *
     * @param int    $order_id   The ID of the order.
     * @param string $old_status The old order status.
     * @param string $new_status The new order status.
     */
    public function delete_tracking_data_on_status_change($order_id, $old_status, $new_status)
    {

        if ($new_status === 'shipped' || $new_status === 'completed') {
            return;
        }

        $order = wc_get_order($order_id);
        if (!$order) {
            return;
        }

        $was_shipped_before_deactivation = $order->get_meta(ZeroV99_Shipment_Tracking_Config::META_WAS_SHIPPED_BEFORE_DEACTIVATION, true);

        if ($was_shipped_before_deactivation !== 'yes') {

            $meta_keys_to_update = [
                ZeroV99_Shipment_Tracking_Config::META_ORDER_PICKED_UP,
                ZeroV99_Shipment_Tracking_Config::META_TRACKING_CODE,
                ZeroV99_Shipment_Tracking_Config::META_CARRIER_NAME,
                ZeroV99_Shipment_Tracking_Config::META_PICKUP_DATE,
                ZeroV99_Shipment_Tracking_Config::META_CARRIER_WEBSITE,
                ZeroV99_Shipment_Tracking_Config::META_SEND_EMAIL,
                ZeroV99_Shipment_Tracking_Config::META_LAST_SHIPPED_DATE
            ];

            foreach ($meta_keys_to_update as $meta_key) {
                $order->update_meta_data($meta_key, '');
            }
        }

        $order->save();
    }
}


