<?php

namespace ZeroV99\ShipmentTracking\frontend;

if (!defined('ABSPATH')) {
    exit;
}

use ZeroV99\ShipmentTracking\helpers\ZeroV99_Shipment_Tracking_Config;

/**
 * Class ZeroV99_Order_Tracking_Details
 *
 * This class is responsible for displaying shipment tracking details on the frontend
 * order details page in WooCommerce. It hooks into the WooCommerce order details page
 * and renders tracking information (carrier name, tracking code, pickup date, and carrier website)
 * if the order status is "shipped" and the required tracking details are available.
 *
 * The class also ensures that the tracking information is displayed in the user's preferred language
 * by temporarily switching the locale to the user's locale before rendering the template.
 */
class ZeroV99_Order_Tracking_Details
{

    /**
     * Constructor.
     *
     * Initializes the class and hooks into the WooCommerce order details page to display
     * shipment tracking information.
     */
    public function __construct()
    {

        add_action('woocommerce_order_details_after_order_table', [$this, 'add_order_tracking_details_to_order_frontend_shop'], 10, 1);
    }

    /**
     * Displays shipment tracking details on the frontend order details page.
     *
     * This function checks if the order status is "shipped" and retrieves the tracking details
     * (carrier name, tracking code, pickup date, and carrier website) from the order meta.
     * If the required details are available, it switches to the user's preferred locale,
     * includes a template to render the tracking information in the correct language,
     * and then restores the original locale.
     *
     * @param WC_Order $order The order object.
     * @return void
     */
    public function add_order_tracking_details_to_order_frontend_shop($order)
    {
        if (!$order) return;

        if ($order->get_status() !== 'shipped') return;

        $carrier_name = $order->get_meta(ZeroV99_Shipment_Tracking_Config::META_CARRIER_NAME);
        $tracking_code = $order->get_meta(ZeroV99_Shipment_Tracking_Config::META_TRACKING_CODE);
        $pickup_date = $order->get_meta(ZeroV99_Shipment_Tracking_Config::META_PICKUP_DATE);
        $carrier_website = $order->get_meta(ZeroV99_Shipment_Tracking_Config::META_CARRIER_WEBSITE);
       
        if (!$tracking_code || !$carrier_name) return;

        // Get user locale
        $user_id = $order->get_user_id();
        $user_locale = $user_id ? get_user_locale($user_id) : get_locale();

        switch_to_locale($user_locale);

        include plugin_dir_path(dirname(__DIR__)) . '/templates/frontend/zerov99_order_tracking_details.php';

        restore_previous_locale();
    }
}
