<?php

namespace ZeroV99\ShipmentTracking\emails;

if (! defined('ABSPATH')) {
    exit;
}

use ZeroV99\ShipmentTracking\helpers\ZeroV99_Shipment_Tracking_Config;
use Exception;

/**
 * Handles the sending of shipment tracking emails to customers.
 *
 * This class provides functionality to send an email notification to customers when their order
 * has been shipped. It uses WooCommerce's email settings for styling and includes dynamic
 * content such as the carrier name, tracking code, and carrier website.
 */
class ZeroV99_Shipment_Tracking_Email
{

    /**
     * Sends an email notification to the customer when their order is shipped.
     *
     * This function constructs and sends an HTML email to the customer when their order
     * has been shipped. The email includes details such as the carrier name, tracking code,
     * and carrier website. It uses WooCommerce's email settings for styling and layout to
     * ensure consistency with other WooCommerce emails.
     *
     * The function temporarily switches the locale to the customer's preferred language (if available)
     * to ensure the email content is localized. After sending the email, it restores the original locale.
     *
     * @param WC_Order $order The order object containing the customer and shipping details.
     * @return bool Whether the email was sent successfully. Returns `true` on success, `false` on failure.
     */
    public static function send_shipped_email($order)
    {

        // Get the user's locale if available, otherwise fall back to the site's default locale
        $user_id = $order->get_user_id();
        $user_locale = $user_id ? get_user_locale($user_id) : get_locale();

        // Temporarily switch to the customer's locale for email localization
        switch_to_locale($user_locale);

        try {

            $customer_email = $order->get_billing_email();

            $customer_name = $order->get_billing_first_name();
            $greeting = sprintf(
                /* translators: %s is the customer name */
                __('Hello, %s!', 'zerov99-shipment-tracking'),
                $customer_name
            );
            $subject       = __('Your order has been shipped!', 'zerov99-shipment-tracking');

            // Retrieve WooCommerce email styling settings
            $header_image          = get_option('woocommerce_email_header_image');
            $base_color            = get_option('woocommerce_email_base_color', '#720eec');
            $body_background_color = get_option('woocommerce_email_body_background_color', '#f7f7f7');
            $body_text_color       = get_option('woocommerce_email_text_color', '#636363');
            $footer_text           = get_option('woocommerce_email_footer_text');
            $footer_text_color     = get_option('woocommerce_email_footer_text_color', '#636363');
            $footer_text           = str_replace(['{site_title}', '{WooCommerce}'], [get_bloginfo('name'), 'WooCommerce'], $footer_text);

            // Retrieve shipping details from order meta
            $carrier_name   = $order->get_meta(ZeroV99_Shipment_Tracking_Config::META_CARRIER_NAME);
            $tracking_code  = $order->get_meta(ZeroV99_Shipment_Tracking_Config::META_TRACKING_CODE);
            $carrier_web    = $order->get_meta(ZeroV99_Shipment_Tracking_Config::META_CARRIER_WEBSITE);
            $pickup_date    = $order->get_meta(ZeroV99_Shipment_Tracking_Config::META_PICKUP_DATE);
            $order_id = "#" . $order->get_id();

            // Load template
            ob_start();
            include plugin_dir_path(dirname(__DIR__)) . 'templates/emails/zerov99_shipped_email.php';
            $message = ob_get_clean();

            // Send mail
            $headers   = ['Content-Type: text/html; charset=UTF-8'];
            $mail_sent = wp_mail($customer_email, $subject, $message, $headers);

            return $mail_sent;
        } catch (Exception $e) {

            return false;
        } finally {

            restore_previous_locale();
        }
    }
}
