<?php

namespace ZeroV99\ShipmentTracking\admin\settings;

if (!defined('ABSPATH')) {
    exit;
}

use ZeroV99\ShipmentTracking\helpers\ZeroV99_Shipment_Tracking_Config;
use ZeroV99\ShipmentTracking\helpers\ZeroV99_Shipment_Tracking_Form_Validator;
use ZeroV99\ShipmentTracking\admin\ZeroV99_Shipment_Tracking_Admin_Notices;

/**
 * Handles the settings and configuration for the zerov99 Shipment Tracking plugin.
 *
 * This class manages the WooCommerce settings tab for shipment tracking, including
 * adding the tab, rendering the settings form, and saving the settings.
 */
class ZeroV99_Shipment_Tracking_Settings
{
    /**
     * Initializes the settings functionality.
     *
     * This function hooks into WooCommerce to add a custom settings tab, render the
     * settings form, and handle the saving of settings.
     */
    public static function init()
    {
        add_filter('woocommerce_settings_tabs_array', array(__CLASS__, 'add_settings_page'), 60);
        add_action('woocommerce_settings_zerov99_shipment_tracking', array(__CLASS__, 'output_form'));
        add_action('woocommerce_update_options_zerov99_shipment_tracking', array(__CLASS__, 'save_form'));
    }

    /**
     * Adds a "Shipment Tracking" tab to the WooCommerce settings page and inserts it after the "Shipping" tab.
     *
     * @param array $settings_tabs The existing WooCommerce settings tabs.
     * @return array The updated settings tabs with the new "Shipment Tracking" tab.
     */
    public static function add_settings_page($settings_tabs)
    {
        $shipping_tab_index = array_search('shipping', array_keys($settings_tabs));

        if ($shipping_tab_index !== false) {
            // Insert the "Shipment Tracking" tab after the "Shipping" tab
            $settings_tabs = array_slice($settings_tabs, 0, $shipping_tab_index + 1, true) +
                ['zerov99_shipment_tracking' => __('Shipment Tracking', 'zerov99-shipment-tracking')] +
                array_slice($settings_tabs, $shipping_tab_index + 1, null, true);
        } else {
            // If the "Shipping" tab is not found, add the "Shipment Tracking" tab at the end
            $settings_tabs['zerov99_shipment_tracking'] = __('Shipment Tracking', 'zerov99-shipment-tracking');
        }

        return $settings_tabs;
    }

    /**
     * Outputs the form fields for the "Shipment Tracking" settings tab.
     *
     * This function renders the settings fields defined in `get_settings`.
     */
    public static function output_form()
    {
        woocommerce_admin_fields(self::get_settings());
    }

    /**
     * Defines the settings fields for the "Shipment Tracking" tab.
     *
     * @return array An array of settings fields, including section titles, color pickers,
     *               checkboxes, number inputs, and custom fields for shipping providers.
     */
    public static function get_settings()
    {
        return array(
            'section_title' => array(
                'name' => __('Shipment Tracking Settings', 'zerov99-shipment-tracking'),
                'type' => 'title',
                'id'   => 'wc_settings_shipped_section_title'
            ),
            ZeroV99_Shipment_Tracking_Config::OPTION_SHIPPED_LABEL_COLOR => array(
                'name' => __('Shipped Label Color', 'zerov99-shipment-tracking'),
                'type' => 'color',
                'desc' => __('Choose the color for the "Shipped" status label.', 'zerov99-shipment-tracking'),
                'id'   => ZeroV99_Shipment_Tracking_Config::OPTION_SHIPPED_LABEL_COLOR,
                'default' => '#4CAF50',
            ),
            ZeroV99_Shipment_Tracking_Config::OPTION_SHIPPING_PROVIDERS => array(
                'name' => __('Shipping Providers', 'zerov99-shipment-tracking'),
                'type' => 'zerov99_shipping_providers_table',
                'id'   => ZeroV99_Shipment_Tracking_Config::OPTION_SHIPPING_PROVIDERS,
            ),
            ZeroV99_Shipment_Tracking_Config::OPTION_SEND_EMAIL => array(
                'name' => __('Send Email', 'zerov99-shipment-tracking'),
                'type' => 'checkbox',
                'desc' => __('Check this to enable the email option by default in the order tracking forms.', 'zerov99-shipment-tracking'),
                'id'   => ZeroV99_Shipment_Tracking_Config::OPTION_SEND_EMAIL,
                'default' => 'no',
            ),
            ZeroV99_Shipment_Tracking_Config::OPTION_ENABLE_AUTO_COMPLETE => array(
                'name' => __('Auto Complete Shipped Orders', 'zerov99-shipment-tracking'),
                'type' => 'checkbox',
                'desc' => __('Enable to automatically complete orders in "Shipped" status.', 'zerov99-shipment-tracking'),
                'id'   => ZeroV99_Shipment_Tracking_Config::OPTION_ENABLE_AUTO_COMPLETE,
                'default' => 'no',
            ),
            ZeroV99_Shipment_Tracking_Config::OPTION_AUTO_COMPLETE_DAYS => array(
                'name' => __('Days to Auto Complete Orders', 'zerov99-shipment-tracking'),
                'type' => 'number',
                'desc' => __('Number of days after which orders in "Shipped" status will be marked as "Completed".', 'zerov99-shipment-tracking'),
                'id'   => ZeroV99_Shipment_Tracking_Config::OPTION_AUTO_COMPLETE_DAYS,
                'default' => 3,
                'custom_attributes' => array(
                    'min' => 1,
                    'step' => 1,
                ),
                'class' => 'auto-complete-days',
            ),
            ZeroV99_Shipment_Tracking_Config::OPTION_DELETE_PLUGIN_DATA => array(
                'name' => __('Delete Plugin Data on Uninstall', 'zerov99-shipment-tracking'),
                'type' => 'checkbox',
                'desc' => __('If checked, all plugin data will be deleted when the plugin is uninstalled, and this action is irreversible.', 'zerov99-shipment-tracking'),
                'id'   => ZeroV99_Shipment_Tracking_Config::OPTION_DELETE_PLUGIN_DATA,
                'default' => 'no',
            ),
            'section_end' => array(
                'type' => 'sectionend',
                'id'   => 'wc_settings_shipped_section_end'
            ),
        );
    }

    /**
     * Saves the settings from the "Shipment Tracking" tab.
     *
     * This function processes and saves the shipping providers' data and updates
     * the WooCommerce settings options.
     */
    public static function save_form()
    {
        // Verify WooCommerce nonce
        if (!isset($_POST['_wpnonce']) || !wp_verify_nonce(sanitize_key(wp_unslash($_POST['_wpnonce'])), 'woocommerce-settings')) {
            wp_die(esc_html__('Security check failed', 'zerov99-shipment-tracking'));
        }

        if (
            isset($_POST['zerov99_shipping_providers_name'], $_POST['zerov99_shipping_providers_url']) &&
            !empty($_POST['zerov99_shipping_providers_name']) &&
            !empty($_POST['zerov99_shipping_providers_url'])
        ) {
            $providers = array();
            $names = array_map('sanitize_text_field', wp_unslash($_POST['zerov99_shipping_providers_name']));
            $urls = array_map('sanitize_text_field', wp_unslash($_POST['zerov99_shipping_providers_url']));

            foreach ($names as $index => $name) {
                $clean_name = sanitize_text_field($name);
                $provider_validation_name = ZeroV99_Shipment_Tracking_Form_Validator::validate_settings_provider_name($clean_name);
                if (is_wp_error($provider_validation_name)) {
                    ZeroV99_Shipment_Tracking_Admin_Notices::handle_form_error($provider_validation_name->get_error_message(), admin_url('admin.php?page=wc-settings&tab=zerov99_shipment_tracking'));
                    return;
                }

                // Sanitize the url and manage the brackets
                $temp_url = str_replace(['{', '}'], ['__OPEN_BRACE__', '__CLOSE_BRACE__'], $urls[$index]);
                $clean_url = sanitize_url($temp_url);
                $clean_url = str_replace(['__OPEN_BRACE__', '__CLOSE_BRACE__'], ['{', '}'], $clean_url);

                $provider_validation_url = ZeroV99_Shipment_Tracking_Form_Validator::validate_settings_provider_url($clean_url);
                if (is_wp_error($provider_validation_url)) {
                    ZeroV99_Shipment_Tracking_Admin_Notices::handle_form_error($provider_validation_url->get_error_message(), admin_url('admin.php?page=wc-settings&tab=zerov99_shipment_tracking'));
                    return;
                }

                if (!empty($provider_validation_name) && !empty($provider_validation_url)) {
                    $providers[] = array(
                        'name' => $provider_validation_name,
                        'url'  => $provider_validation_url,
                    );
                }
            }
            update_option(ZeroV99_Shipment_Tracking_Config::OPTION_SHIPPING_PROVIDERS, $providers);
        } else {
            update_option(ZeroV99_Shipment_Tracking_Config::OPTION_SHIPPING_PROVIDERS, []);
        }

        woocommerce_update_options(self::get_settings());
    }
}
