<?php

namespace ZeroV99\ShipmentTracking\admin;

if (!defined('ABSPATH')) {
    exit;
}

use ZeroV99\ShipmentTracking\helpers\ZeroV99_Shipment_Tracking_Config;

/**
 * Handles the custom "Shipped" order status in WooCommerce.
 *
 * This class is responsible for registering the "Shipped" status, restoring orders to the "Shipped" status
 * after plugin reactivation, and adding the "Shipped" status to the list of WooCommerce order statuses.
 * It uses constants defined in `ZeroV99_Shipment_Tracking_Config` for metadata keys to ensure consistency and maintainability.
 */
class ZeroV99_Shipment_Tracking_Status_Shipped
{
    /**
     * Registers the custom "Shipped" order status in WooCommerce.
     *
     * This function checks if the "Shipped" status is already registered. If not, it registers
     * the status with the appropriate labels and settings.
     */
    public static function register_shipped_order_status()
    {
        if (!get_post_status_object('wc-shipped')) {
            register_post_status('wc-shipped', array(
                'label'                     => __('Shipped', 'zerov99-shipment-tracking'),
                'public'                    => true,
                'exclude_from_search'       => false,
                'show_in_admin_all_list'    => true,
                'show_in_admin_status_list' => true,
                /* translators: %s is the number of shipped orders */
                'label_count'               => _n_noop('Shipped <span class="count">(%s)</span>', 'Shipped <span class="count">(%s)</span>', 'zerov99-shipment-tracking')
            ));
        }
    }

    /**
     * Restores orders to the "Shipped" status after plugin reactivation.
     *
     * This function finds orders that were previously marked as "Shipped" before the plugin was deactivated
     * and restores their status to "Shipped". It uses the constant `ZeroV99_Shipment_Tracking_Config::META_WAS_SHIPPED_BEFORE_DEACTIVATION`
     * to identify these orders and removes the temporary meta flag after restoring the status.
     */
    public static function restore_shipped_orders()
    {
        // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- One-time operation during plugin reactivation
        $orders = wc_get_orders(array(
            'status'     => 'processing',
            'limit'      => -1,
            'meta_query' => [[   
                'key' => ZeroV99_Shipment_Tracking_Config::META_WAS_SHIPPED_BEFORE_DEACTIVATION,
                'value' => 'yes'
            ]],
            'return'     => 'ids',
        ));

        foreach ($orders as $order_id) {
            $order = wc_get_order($order_id);
            if ($order) {
                $order->update_status('wc-shipped', __('Plugin reactivated: Status changed from Processing to Shipped.', 'zerov99-shipment-tracking'));
                $order->delete_meta_data(ZeroV99_Shipment_Tracking_Config::META_WAS_SHIPPED_BEFORE_DEACTIVATION);
                $order->save();
            }
        }
    }

    /**
     * Adds the "Shipped" status to the list of WooCommerce order statuses.
     *
     * This function inserts the "Shipped" status into the list of order statuses, placing it
     * immediately after the "Processing" status.
     *
     * @param array $order_statuses The existing list of order statuses.
     * @return array The updated list of order statuses with "Shipped" included.
     */
    public static function add_shipped_to_order_statuses($order_statuses)
    {
        $new_order_statuses = array();
        foreach ($order_statuses as $key => $status) {
            $new_order_statuses[$key] = $status;
            if ('wc-processing' === $key) {
                $new_order_statuses['wc-shipped'] = __('Shipped', 'zerov99-shipment-tracking');
            }
        }
        return $new_order_statuses;
    }
}
