<?php

namespace ZeroV99\ShipmentTracking\admin;

if (!defined('ABSPATH')) {
    exit;
}

use ZeroV99\ShipmentTracking\helpers\ZeroV99_Shipment_Tracking_Config;

/**
 * Handles custom order actions in the WooCommerce orders list.
 *
 * This class adds custom actions to the order actions dropdown based on the order status.
 * For example, it allows marking orders as "shipped" or returning them to "processing" status.
 * It uses constants defined in `ZeroV99_Shipment_Tracking_Config` for option names to ensure consistency and maintainability.
 */
class ZeroV99_Shipment_Tracking_Order_Actions
{
    /**
     * @var string The slug for the custom "shipped" action.
     */
    private $custom_order_action = 'zerov99-shipped'; // Slug custom action

    /**
     * @var string The color for the "shipped" status label.
     */
    private $shipped_color;

    /**
     * Constructor.
     *
     * Initializes the class and hooks into the `woocommerce_admin_order_actions` filter
     * to add custom actions to the orders list. It retrieves the shipped label color
     * from the plugin settings using the constant `ZeroV99_Shipment_Tracking_Config::OPTION_SHIPPED_LABEL_COLOR`.
     */
    public function __construct()
    {
        $this->shipped_color = get_option(ZeroV99_Shipment_Tracking_Config::OPTION_SHIPPED_LABEL_COLOR, '#4CAF50');
        add_filter('woocommerce_admin_order_actions', [$this, 'add_actions_to_orders'], 10, 2);
    }

    /**
     * Adds custom actions to the order actions dropdown in the WooCommerce orders list.
     *
     * This function modifies the actions available for orders based on their status:
     * - For orders with the "shipped" status, it adds actions to mark the order as "processing" or "completed".
     * - For orders with the "processing" status, it adds an action to mark the order as "shipped".
     *
     * @param array    $actions The existing order actions.
     * @param WC_Order $order   The order object.
     * @return array The modified order actions.
     */
    public function add_actions_to_orders($actions, $order)
    {
        // Add actions for orders with the "shipped" status
        if ($order->has_status('shipped')) {
            // Add action to return the order to "processing" status
            $actions['mark_processing'] = [
                'url'    => wp_nonce_url(admin_url('admin-ajax.php?action=woocommerce_mark_order_status&status=processing&order_id=' . $order->get_id()), 'woocommerce-mark-order-status'),
                'name'   => __('Processing', 'woocommerce'),
                'action' => 'processing',
            ];

            // Add action to mark the order as "completed"
            $actions['complete'] = [
                'url'    => wp_nonce_url(admin_url('admin-ajax.php?action=woocommerce_mark_order_status&status=completed&order_id=' . $order->get_id()), 'woocommerce-mark-order-status'),
                'name'   => __('Complete', 'woocommerce'),
                'action' => 'complete',
            ];
        }

        // Add action for orders with the "processing" status to mark them as "shipped"
        if ($order->has_status('processing')) {
            $actions = array($this->custom_order_action => [
                'url'    => wp_nonce_url(admin_url('admin-ajax.php?action=woocommerce_mark_order_status&status=shipped&order_id=' . $order->get_id()), 'woocommerce-mark-order-status'),
                'name'   => __('Shipped', 'zerov99-shipment-tracking'),
                'action' => $this->custom_order_action,
            ]) + $actions;
        }

        return $actions;
    }
}