<?php

namespace ZeroV99\ShipmentTracking\admin;

if (!defined('ABSPATH')) {
    exit;
}

use ZeroV99\ShipmentTracking\helpers\ZeroV99_Shipment_Tracking_Config;
use ZeroV99\ShipmentTracking\helpers\ZeroV99_Shipment_Tracking_Form_Validator;
use ZeroV99\ShipmentTracking\emails\ZeroV99_Shipment_Tracking_Email;

/**
 * Handles the modal for tracking details in the WooCommerce order screens.
 *
 * This class is responsible for displaying a modal to input and save shipment tracking details
 * for orders. It also processes the data submitted via the modal and updates the order metadata
 * using constants defined in `ZeroV99_Shipment_Tracking_Config`. The class handles AJAX requests for saving tracking data
 * and sending email notifications if the "Send Email" option is enabled.
 */
class ZeroV99_Shipment_Tracking_Modal
{
    /**
     * Constructor.
     *
     * Initializes the class and hooks into the `admin_footer` action to add the modal,
     * and the `wp_ajax_save_tracking_modal_data` action to handle the modal form submission.
     */
    public function __construct()
    {
        add_action('admin_footer', [$this, 'add_modal_tracking']);
        add_action('wp_ajax_save_tracking_modal_data', [$this, 'save_tracking_modal_data']);
    }

    /**
     * Adds a modal for tracking details to WooCommerce order screens.
     *
     * This function checks if the current screen is a WooCommerce order screen and includes
     * the modal template for shipment tracking. It also sets default values for email sending
     * based on plugin settings using constants from `ZeroV99_Shipment_Tracking_Config`.
     */
    public function add_modal_tracking()
    {
        $screen = get_current_screen();

        $current_date = gmdate('Y-m-d');

        // Default value to avoid undefined warnings
        $send_email = get_option(ZeroV99_Shipment_Tracking_Config::OPTION_SEND_EMAIL) === 'yes' ? 'yes' : 'no';

        if ($screen && in_array($screen->id, ['woocommerce_page_wc-orders', 'toplevel_page_woocommerce'])) {
            include plugin_dir_path(dirname(__FILE__, 2)) . 'templates/admin/zerov99_shipment_tracking_modal.php';
        }
    }

    /**
     * Saves tracking data submitted via the modal and updates the order metadata.
     *
     * This function processes and validates the tracking data submitted via AJAX, updates the order metadata
     * using constants from `ZeroV99_Shipment_Tracking_Config`, and sends an email notification if the "Send Email" option is enabled.
     * It also handles error responses and JSON success/error messages.
     */
    public function save_tracking_modal_data()
    {
        // Verify AJAX request nonce
        check_ajax_referer('zerov99_save_tracking_nonce', '_ajax_nonce');

        if (empty($_POST['order_id'])) {
            wp_send_json_error(['message' => __('Order ID is missing.', 'zerov99-shipment-tracking')]);
            return;
        }

        $order_id = intval($_POST['order_id']);
        $order = wc_get_order($order_id);

        if (!$order) {
            wp_send_json_error(['message' => __('Order not found.', 'zerov99-shipment-tracking')]);
            return;
        }

        // Update shipped date and picked_up status
        $order->update_meta_data(ZeroV99_Shipment_Tracking_Config::META_LAST_SHIPPED_DATE, current_time('mysql'));
        $order->update_meta_data(ZeroV99_Shipment_Tracking_Config::META_ORDER_PICKED_UP, 'yes');

        // Sanitization, validation, and update meta fields
        if (isset($_POST['carrier_name']) && !empty($_POST['carrier_name'])) {
            $carrier_name = sanitize_text_field(wp_unslash($_POST['carrier_name']));
            $carrier_validation = ZeroV99_Shipment_Tracking_Form_Validator::validate_carrier_name($carrier_name);
            if (is_wp_error($carrier_validation)) {
                wp_send_json_error(['message' => $carrier_validation->get_error_message()]);
                return;
            }
            $order->update_meta_data(ZeroV99_Shipment_Tracking_Config::META_CARRIER_NAME, $carrier_name);
        }

        if (isset($_POST['tracking_code']) && !empty($_POST['tracking_code'])) {
            $tracking_code = sanitize_text_field(wp_unslash($_POST['tracking_code']));
            $tracking_validation = ZeroV99_Shipment_Tracking_Form_Validator::validate_tracking_number($tracking_code);
            if (is_wp_error($tracking_validation)) {
                wp_send_json_error(['message' => $tracking_validation->get_error_message()]);
                return;
            }
            $order->update_meta_data(ZeroV99_Shipment_Tracking_Config::META_TRACKING_CODE, $tracking_code);
        }

        if (isset($_POST['pickup_date'])) {
            $pickup_date = sanitize_text_field(wp_unslash($_POST['pickup_date']));
            $date_validation = ZeroV99_Shipment_Tracking_Form_Validator::validate_date($pickup_date);
            if (is_wp_error($date_validation)) {
                wp_send_json_error(['message' => $date_validation->get_error_message()]);
                return;
            }
            $order->update_meta_data(ZeroV99_Shipment_Tracking_Config::META_PICKUP_DATE, $pickup_date);
        }

        if (isset($_POST['carrier_website'])) {
            $carrier_website = esc_url(sanitize_url(wp_unslash($_POST['carrier_website'])));
            $order->update_meta_data(ZeroV99_Shipment_Tracking_Config::META_CARRIER_WEBSITE, $carrier_website);
        }

        if (isset($_POST['send_email'])) {
            $send_email = $_POST['send_email'] === 'yes' ? 'yes' : 'no';
            $email_validation = ZeroV99_Shipment_Tracking_Form_Validator::validate_email_send($order->get_billing_email(), $send_email);
            if (is_wp_error($email_validation)) {
                wp_send_json_error(['message' => $email_validation->get_error_message()]);
                return;
            }
            $order->update_meta_data(ZeroV99_Shipment_Tracking_Config::META_SEND_EMAIL, $send_email);
        }

        // Update status and save
        $order->update_status('wc-shipped', __('Order status updated to Shipped.', 'zerov99-shipment-tracking'));
        $order->save();

        // Send email if requested
        if ($send_email === 'yes' && $order->get_status() === 'shipped') { //wc-shipped does not work
            if (!ZeroV99_Shipment_Tracking_Email::send_shipped_email($order)) {
                wp_send_json_error(['message' => __('Order updated. Email not sent. Please check your email settings.', 'zerov99-shipment-tracking')]);
                return;
            }
        }

        wp_send_json_success(__('Tracking details saved successfully!', 'zerov99-shipment-tracking'));
    }
}
