<?php

namespace ZeroV99\ShipmentTracking\admin;

if (!defined('ABSPATH')) {
    exit;
}

use ZeroV99\ShipmentTracking\helpers\ZeroV99_Shipment_Tracking_Config;
use ZeroV99\ShipmentTracking\helpers\ZeroV99_Shipment_Tracking_Form_Validator;
use ZeroV99\ShipmentTracking\admin\ZeroV99_Shipment_Tracking_Admin_Notices;
use ZeroV99\ShipmentTracking\emails\ZeroV99_Shipment_Tracking_Email;

/**
 * Handles the order tracking metabox in the WooCommerce order edit screen.
 *
 * This class is responsible for adding a metabox to the WooCommerce order edit screen,
 * displaying tracking details, and saving the data submitted from the metabox. It also
 * handles email notifications when the "Send Email" option is enabled. The class uses
 * constants defined in `ZeroV99_Shipment_Tracking_Config` for metadata keys and options to ensure consistency
 * and maintainability across the plugin.
 */
class ZeroV99_Shipment_Tracking_Metabox
{
    /**
     * Constructor.
     *
     * Initializes the class and hooks into the `add_meta_boxes` and `woocommerce_process_shop_order_meta`
     * actions to add the metabox and save its data.
     */
    public function __construct()
    {
        add_action('add_meta_boxes', [$this, 'add_order_tracking_metabox']);
        add_action('woocommerce_process_shop_order_meta', [$this, 'save_order_tracking_metabox'], 10, 2);
    }

    /**
     * Adds a metabox for order tracking details to the WooCommerce order edit screen.
     *
     * This function registers a metabox titled "Order Tracking" on the side of the WooCommerce
     * order edit screen. The metabox content is rendered by the `display_order_tracking_metabox` method.
     */
    public function add_order_tracking_metabox()
    {
        add_meta_box(
            'zerov99-order-tracking-metabox',
            __('Order Tracking', 'zerov99-shipment-tracking'),
            [$this, 'display_order_tracking_metabox'],
            wc_get_page_screen_id('shop-order'),
            'side'
        );
    }

    /**
     * Displays the content of the order tracking metabox.
     *
     * This function retrieves tracking-related metadata (e.g., tracking code, carrier name, pickup date)
     * from the order using constants defined in `ZeroV99_Shipment_Tracking_Config`. It also sets default values for pickup date
     * and email settings if they are not already defined. The metabox content is rendered using a template file.
     *
     * @param WP_Post|WC_Order $post The post or order object.
     */
    public function display_order_tracking_metabox($post)
    {
        $order = wc_get_order($post);

        // Retrieve metadata using constants from ZeroV99_Shipment_Tracking_Config
        $picked_up       = $order->get_meta(ZeroV99_Shipment_Tracking_Config::META_ORDER_PICKED_UP, true);
        $tracking_code   = $order->get_meta(ZeroV99_Shipment_Tracking_Config::META_TRACKING_CODE, true);
        $carrier_name    = $order->get_meta(ZeroV99_Shipment_Tracking_Config::META_CARRIER_NAME, true);
        $pickup_date     = $order->get_meta(ZeroV99_Shipment_Tracking_Config::META_PICKUP_DATE, true);
        $carrier_website = $order->get_meta(ZeroV99_Shipment_Tracking_Config::META_CARRIER_WEBSITE, true);
        $send_email      = $order->get_meta(ZeroV99_Shipment_Tracking_Config::META_SEND_EMAIL, true);
        $providers       = get_option(ZeroV99_Shipment_Tracking_Config::OPTION_SHIPPING_PROVIDERS, []);

        // Set default values
        $current_date = gmdate('Y-m-d');
        $pickup_date = !empty($pickup_date) ? $pickup_date : $current_date;
        $picked_up = ($order->has_status('wc-shipped')) ? 'yes' : 'no';

        $send_email = $order->get_meta(ZeroV99_Shipment_Tracking_Config::META_SEND_EMAIL, true);
        if (empty($send_email)) {
            $send_email = get_option(ZeroV99_Shipment_Tracking_Config::OPTION_SEND_EMAIL) === 'yes' ? 'yes' : 'no';
        }

        // Include the template to render the metabox content
        include plugin_dir_path(dirname(__DIR__)) . '/templates/admin/zerov99_shipment_tracking_metabox.php';
    }

    /**
     * Saves the data from the order tracking metabox and sends an email if the "Send Email" option is checked.
     *
     * This function processes and validates the data submitted from the order tracking metabox, updates the order metadata,
     * and sends an email notification if the "Send Email" option is enabled. It uses constants from `ZeroV99_Shipment_Tracking_Config` for metadata keys
     * and options. If validation fails, it handles error notifications and redirects to the order edit screen.
     *
     * @param int   $order_id The ID of the order being processed.
     * @param array $data     WooCommerce processing data (not used in this case).
     */
    public function save_order_tracking_metabox($order_id, $data)
    {
        // VerifY WooCommerce nonce
        if (!isset($_POST['_wpnonce']) || !wp_verify_nonce(sanitize_key(wp_unslash($_POST['_wpnonce'])), 'update-order_' . $order_id)) {
            wp_die(esc_html__('Security check failed', 'zerov99-shipment-tracking'));
        }

        $order = wc_get_order($order_id);
        if (!$order) {
            return;
        }

        // Save current values before any changes to compare later and do not send email if no changes were made
        $original_tracking = [
            'code' => $order->get_meta(ZeroV99_Shipment_Tracking_Config::META_TRACKING_CODE, true),
            'carrier' => $order->get_meta(ZeroV99_Shipment_Tracking_Config::META_CARRIER_NAME, true),
            'date' => $order->get_meta(ZeroV99_Shipment_Tracking_Config::META_PICKUP_DATE, true),
        ];

        // Save data only if status equals shipped
        $selected_status = isset($_POST['order_status']) ? sanitize_text_field(wp_unslash($_POST['order_status'])) : $order->get_status();
        if ($selected_status === 'wc-shipped') {
            $order->update_meta_data(ZeroV99_Shipment_Tracking_Config::META_LAST_SHIPPED_DATE, current_time('mysql'));
            $order->update_meta_data(ZeroV99_Shipment_Tracking_Config::META_ORDER_PICKED_UP, 'yes');

            $tracking_changed = false;
            $new_tracking = [];

            // Sanitization, validation, and update meta fields
            if (isset($_POST['zerov99_carrier_name']) && !empty($_POST['zerov99_carrier_name'])) {
                $carrier_name = sanitize_text_field(wp_unslash($_POST['zerov99_carrier_name']));
                $carrier_validation = ZeroV99_Shipment_Tracking_Form_Validator::validate_carrier_name($carrier_name);
                if (is_wp_error($carrier_validation)) {
                    ZeroV99_Shipment_Tracking_Admin_Notices::handle_form_error($carrier_validation->get_error_message(), admin_url("post.php?post={$order_id}&action=edit"));
                    return;
                }
                $order->update_meta_data(ZeroV99_Shipment_Tracking_Config::META_CARRIER_NAME, $carrier_name);
                $new_tracking['carrier'] = $carrier_name;
            }

            if (isset($_POST['zerov99_tracking_code']) && !empty($_POST['zerov99_tracking_code'])) {
                $tracking_code = sanitize_text_field(wp_unslash($_POST['zerov99_tracking_code']));
                $tracking_validation = ZeroV99_Shipment_Tracking_Form_Validator::validate_tracking_number($tracking_code);
                if (is_wp_error($tracking_validation)) {
                    ZeroV99_Shipment_Tracking_Admin_Notices::handle_form_error($tracking_validation->get_error_message(), admin_url("post.php?post={$order_id}&action=edit"));
                    return;
                }
                $order->update_meta_data(ZeroV99_Shipment_Tracking_Config::META_TRACKING_CODE, $tracking_code);
                $new_tracking['code'] = $tracking_code;
            }

            if (isset($_POST['zerov99_pickup_date'])) {
                $pickup_date = sanitize_text_field(wp_unslash($_POST['zerov99_pickup_date']));
                $date_validation = ZeroV99_Shipment_Tracking_Form_Validator::validate_date($pickup_date);
                if (is_wp_error($pickup_date)) {
                    ZeroV99_Shipment_Tracking_Admin_Notices::handle_form_error($date_validation->get_error_message(), admin_url("post.php?post={$order_id}&action=edit"));
                    return;
                }
                $order->update_meta_data(ZeroV99_Shipment_Tracking_Config::META_PICKUP_DATE, $pickup_date);
                $new_tracking['date'] = $pickup_date;
            }

            if (isset($_POST['zerov99_carrier_website'])) {
                $carrier_website = esc_url(sanitize_url(wp_unslash($_POST['zerov99_carrier_website'])));
                $order->update_meta_data(ZeroV99_Shipment_Tracking_Config::META_CARRIER_WEBSITE, $carrier_website);
            }

            // Determine if there were changes in the tracking fields
            foreach ($new_tracking as $key => $value) {
                if ($original_tracking[$key] !== $value) {
                    $tracking_changed = true;
                    break;
                }
            }

            $send_email = 'no';  // Default value to avoid undefined warnings
            if (isset($_POST['zerov99_send_email'])) {
                $send_email = $_POST['zerov99_send_email'] === 'yes' ? 'yes' : 'no';
                $email_validation = ZeroV99_Shipment_Tracking_Form_Validator::validate_email_send($order->get_billing_email(), $send_email);
                if (is_wp_error($email_validation)) {
                    ZeroV99_Shipment_Tracking_Admin_Notices::handle_form_error($email_validation->get_error_message(), admin_url("post.php?post={$order_id}&action=edit"));
                    return;
                }
                $order->update_meta_data(ZeroV99_Shipment_Tracking_Config::META_SEND_EMAIL, $send_email);
            } else {

                $order->update_meta_data(ZeroV99_Shipment_Tracking_Config::META_SEND_EMAIL, 'no');
                $send_email = 'no';
            }

            // Update status and save
            $order->update_status('wc-shipped', __('Order status updated to Shipped', 'zerov99-shipment-tracking'));
            $order->save();


            if ($send_email === 'yes' && $selected_status === 'wc-shipped' && $tracking_changed)  {
                if (!ZeroV99_Shipment_Tracking_Email::send_shipped_email($order)) {
                    ZeroV99_Shipment_Tracking_Admin_Notices::handle_form_error(
                        esc_html__('Order updated. Email not sent. Please check your email settings.', 'zerov99-shipment-tracking'),
                        admin_url("post.php?post={$order_id}&action=edit")
                    );

                    return;
                }
                
            }
        }
    }
}
