<?php

namespace ZeroV99\ShipmentTracking\admin;

if (!defined('ABSPATH')) {
    exit;
}

use ZeroV99\ShipmentTracking\helpers\ZeroV99_Shipment_Tracking_Config;


/**
 * Handles the display of a metabox for completed tracking details in the WooCommerce order edit screen.
 *
 * This class adds a metabox to the WooCommerce order edit screen for orders with the "completed" status.
 * The metabox displays tracking details such as the tracking code, carrier name, and pickup date.
 * It uses constants defined in the `ZeroV99_Shipment_Tracking_Config` class to retrieve the metadata keys, ensuring consistency
 * and maintainability across the plugin.
 */
class ZeroV99_Shipment_Tracking_Completed_Metabox
{
    /**
     * Constructor.
     *
     * Initializes the class and hooks into the `add_meta_boxes` action to add the completed tracking metabox.
     * The metabox is only added if the current screen is the WooCommerce order edit screen and the order
     * status is "completed".
     */
    public function __construct()
    {
        add_action('add_meta_boxes', [$this, 'add_completed_tracking_metabox'], 10, 2);
    }

    /**
     * Adds a metabox to display completed tracking details for orders with the "completed" status.
     *
     * This function checks if the current screen is the WooCommerce order edit screen and if the order
     * status is "completed". If both conditions are met, it adds a metabox to display tracking details.
     * The metabox is displayed on the side of the order edit screen.
     *
     * @param string $post_type The current post type.
     * @param WP_Post|WC_Order $post The post or order object.
     */
    public function add_completed_tracking_metabox($post_type, $post)
    {
        $screen = get_current_screen();
        if (!$screen || $screen->id !== wc_get_page_screen_id('shop-order')) return;

        $order = wc_get_order($post);

        if ($order && $order->get_status() === 'completed') {
            add_meta_box(
                'zerov99-complete-tracking-metabox',
                __('Completed Tracking Details', 'zerov99-shipment-tracking'),
                [$this, 'display_completed_tracking_metabox'],
                $screen->id,
                'side'
            );
        } else {
            return;
        }
    }

    /**
     * Displays the content of the completed tracking metabox.
     *
     * This function retrieves the tracking details (tracking code, carrier name, and pickup date)
     * from the order meta using the constants defined in `ZeroV99_Shipment_Tracking_Config`. It then includes a template file
     * to render the metabox content.
     *
     * @param WP_Post|WC_Order $post The post or order object.
     */
    public function display_completed_tracking_metabox($post)
    {
        $order = wc_get_order($post);

        if (!$order || $order->get_status() !== 'completed') {
            return;
        }

        // Retrieve tracking details using constants from ZeroV99_Shipment_Tracking_Config
        $tracking_code = $order->get_meta(ZeroV99_Shipment_Tracking_Config::META_TRACKING_CODE, true);
        $carrier_name = $order->get_meta(ZeroV99_Shipment_Tracking_Config::META_CARRIER_NAME, true);
        $pickup_date = $order->get_meta(ZeroV99_Shipment_Tracking_Config::META_PICKUP_DATE, true);
        $send_email = $order->get_meta(ZeroV99_Shipment_Tracking_Config::META_SEND_EMAIL, true);

        // Include the template to render the metabox content
        include plugin_dir_path(dirname(__DIR__)) . '/templates/admin/zerov99_completed_tracking_metabox.php';
    }
}
