jQuery(document).ready(function ($) {

  // ==========================
  // COMMON VARIABLES
  // ==========================
  const orderStatusField = $("#order_status");
  const trackingContainer = $("#zerov99-order-tracking-metabox");
  const trackingFields = $("#zerov99_tracking_fields");
  const orderPickedUpToggle = $("#zerov99_order_picked_up");

  // Providers sent via wp_localize_script
  let providers = zerov99Data.providers || [];

  // Metabox Fields
  const toggle = $("#zerov99_order_picked_up");
  const pickupDateInput = $("#zerov99_pickup_date");
  const carrierSelect = $("#zerov99_carrier_name");
  const trackingInput = $("#zerov99_tracking_code");
  const websiteInput = $("#zerov99_carrier_website");

  // Modal Variables
  let nonce = $("#zerov99_tracking_form").data("nonce");
  const modalTrackingInput = $("#zerov99_modal_tracking_code");
  const initialSendEmailState = $("#zerov99_modal_send_email").prop("checked");
 
  // ==========================
  // FUNCTIONS
  // ==========================

  // Toggle tracking fields visibility in metabox
  function toggleTrackingFields() {
    trackingFields.css("display", toggle.is(":checked") ? "block" : "none");
  }

  // Update metabox state based on order status
  function updateMetaboxState() {
    const isShipped = orderStatusField.val() === "wc-shipped";

    trackingFields.toggle(isShipped);
    carrierSelect.prop("required", isShipped);
    trackingInput.prop("required", isShipped);
    trackingInput.attr("pattern", "[a-zA-Z0-9\\-_]{8,30}");
    pickupDateInput.prop("required", isShipped);
    
    trackingFields.find("input, select").prop("disabled", !isShipped);
  }

  // Validate tracking code input
  function validateTrackingInput(input) {
    const isValid = input.checkValidity();
    $(input).css("border", isValid ? "" : "2px solid red");
  }

  // Update tracking URL in both metabox & modal
  function updateTrackingUrl(isModal = false) {
    let selectedProvider = isModal
      ? $("#zerov99_modal_carrier_name").val()
      : carrierSelect.val();
    let trackingCode = isModal
      ? $("#zerov99_modal_tracking_code").val().trim()
      : trackingInput.val().trim();
    let websiteInputField = isModal
      ? $("#zerov99_modal_carrier_website")
      : websiteInput;

    let providerData = providers.find(
      (provider) => provider.name === selectedProvider
    );

    if (providerData && trackingCode) {
      websiteInputField.val(
        providerData.url.replace(
          "{tracking_code}",
          encodeURIComponent(trackingCode)
        )
      );
    } else {
      websiteInputField.val("");
    }
  }

  // Update visibility of tracking metabox
  function updateMetaboxVisibility() {
    let selectedStatus = orderStatusField.val();
    let isShipped = selectedStatus === "wc-shipped";

    trackingContainer.toggle(isShipped || selectedStatus === "wc-processing");
    trackingFields.toggle(isShipped);
    orderPickedUpToggle.prop("checked", isShipped);
  }

  // ==========================
  // MODAL FUNCTIONALITIES
  // ==========================

  $(".zerov99-shipped").on("click", function (e) {
    e.preventDefault();
    let orderId = $(this).closest("tr").attr("id").replace("order-", "");
    $("#zerov99_order_tracking_modal").fadeIn();
    $("#zerov99_save_tracking_modal").data("order-id", orderId);
  });

  $("#zerov99_close_tracking_modal, #zerov99_order_tracking_modal").on(
    "click",
    function (e) {
      if (e.target === this) {
        $("#zerov99_order_tracking_modal").fadeOut();
        resetModalFields();
      }
      
    }
  );

  $("#zerov99_tracking_form").on("submit", function (e) {
    e.preventDefault();
    
    let orderId = $("#zerov99_save_tracking_modal").data("order-id");
    let trackingCode = modalTrackingInput.val();
    let carrierName = $("#zerov99_modal_carrier_name").val();
    let pickUpDate = $("#zerov99_modal_pickup_date").val();
    let carrierWebsite = $("#zerov99_modal_carrier_website").val();
    let sendEmail = $("#zerov99_modal_send_email").prop("checked")
      ? "yes"
      : "no";

    $("#zerov99_save_tracking_modal").html(
      '<span class="zerov99-spinner"></span> Saving...'
    );

    $.ajax({
      type: "POST",
      url: ajaxurl,
      data: {
        action: "save_tracking_modal_data",
        order_id: orderId,
        tracking_code: trackingCode,
        carrier_name: carrierName,
        pickup_date: pickUpDate,
        carrier_website: carrierWebsite,
        send_email: sendEmail,
        _ajax_nonce: nonce,
      },
      success: function (response) {
        if (response.success) {
          location.reload();
        } else {
          const errorMessage = response.data?.message || "Error";
          $("#zerov99_tracking_form").prepend(
            '<div class="zerov99-error-notice">' + errorMessage + "</div>"
          );
        }
      },
      error: function () {
        alert(zerov99Data.serverAlert);
      },
      complete: function () {
        $("#zerov99_save_tracking_modal").html("Save & Ship");
      },
    });
  });

  // Reset field (used when closing modal without submit)
  const resetModalFields = () => {
    $("#zerov99_tracking_form .zerov99-error-notice").remove();
    $("#zerov99_modal_tracking_code").val("").css("border", "");
    $("#zerov99_modal_carrier_name").val("");
    $("#zerov99_modal_pickup_date").val(new Date().toISOString().split("T")[0]);
    $("#zerov99_modal_carrier_website").val("");
    $("#zerov99_modal_send_email").prop("checked", initialSendEmailState);
  };

  // ==========================
  // EVENT LISTENERS
  // ==========================

  toggle.on("change", toggleTrackingFields);
  carrierSelect.on("change", function () {
    updateTrackingUrl(false);
  });
  trackingInput.on("input", function () {
    validateTrackingInput(this);
    updateTrackingUrl(false);
  });

  orderStatusField.on("change", function () {
    updateMetaboxState();
    updateMetaboxVisibility();
  });

  orderPickedUpToggle.on("change", function () {
    orderStatusField
      .val(orderPickedUpToggle.is(":checked") ? "wc-shipped" : "wc-processing")
      .trigger("change");
  });

  modalTrackingInput.on("input", function () {
    validateTrackingInput(this);
    updateTrackingUrl(true);
  });

  $("#zerov99_modal_carrier_name").on("change", function () {
    updateTrackingUrl(true);
  });

  // ==========================
  // INITIALIZATION
  // ==========================
  toggleTrackingFields();
  updateMetaboxState();
  updateMetaboxVisibility();
});
