<?php
/**
 * Core plugin functionality for ZeroState.
 *
 * @package ZeroState
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

/**
 * Class ZeroState_Core
 *
 * Handles the core reset functionality for the ZeroState plugin.
 */
class ZeroState_Core {
    /**
     * Core WordPress tables (without prefix).
     *
     * @var array
     */
    private static $core_tables = [
        'commentmeta', 'comments', 'links', 'options', 'postmeta', 'posts',
        'term_relationships', 'term_taxonomy', 'termmeta', 'terms', 'usermeta', 'users'
    ];

    /**
     * Initialize the plugin.
     */
    public static function init() {
        add_action('admin_action_zerostate_full_reset_action', [__CLASS__, 'handle_full_reset_action']);
        add_action('admin_notices', [__CLASS__, 'zerostate_admin_notice']);
    }

    /**
     * Get core tables with database prefix.
     *
     * @return array List of prefixed table names.
     */
    private static function get_prefixed_core_tables() {
        global $wpdb;
        return array_map(function ($tbl) use ($wpdb) {
            return sanitize_text_field($wpdb->prefix . $tbl);
        }, self::$core_tables);
    }

    /**
     * Initialize the WordPress filesystem.
     *
     * @return true|WP_Error True on success, WP_Error on failure.
     */
    private static function init_filesystem() {
        if (!function_exists('request_filesystem_credentials')) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        if (!WP_Filesystem()) {
            ob_start();
            $credentials = request_filesystem_credentials(site_url() . '/wp-admin/', '', true, false, null);
            $form_output = ob_get_clean();

            if (false === $credentials) {
                return new WP_Error('filesystem_error', esc_html__('Could not initialize filesystem. Filesystem credentials might be required.', 'zerostate'));
            }
            if (!WP_Filesystem($credentials)) {
                return new WP_Error('filesystem_error', esc_html__('Could not initialize filesystem with provided credentials.', 'zerostate'));
            }
        }
        return true;
    }

    /**
     * Handle the full reset action.
     */
    public static function handle_full_reset_action() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission to perform this action.', 'zerostate'));
        }

        if (!isset($_POST['zerostate_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['zerostate_nonce'])), 'zerostate_full_reset_action')) {
            wp_die(esc_html__('Security check failed.', 'zerostate'));
        }

        $confirm_value = isset($_POST['zerostate_confirm']) ? absint($_POST['zerostate_confirm']) : 0;
        if ($confirm_value !== 100) {
            wp_safe_redirect(add_query_arg('reset_error', 'slider', wp_get_referer()));
            exit;
        }

        $result = self::do_site_reset();

        if (is_wp_error($result)) {
            wp_safe_redirect(add_query_arg('reset_error', 'fail_' . $result->get_error_code(), wp_get_referer()));
            exit;
        }
    }

    /**
     * Perform a full site reset.
     *
     * @return true|WP_Error True on success, WP_Error on failure.
     */
    public static function do_site_reset() {
        global $current_user, $wpdb, $wp_filesystem;

        if (!current_user_can('manage_options')) {
            return new WP_Error('permission_denied', esc_html__('Insufficient permissions.', 'zerostate'));
        }

        $fs_init = self::init_filesystem();
        if (is_wp_error($fs_init)) {
            return $fs_init;
        }

        if (!function_exists('wp_install')) {
            if (!file_exists(ABSPATH . 'wp-admin/includes/upgrade.php')) {
                return new WP_Error('dependency_missing', esc_html__('Reset failed: Required WordPress files (upgrade.php) are missing.', 'zerostate'));
            }
            require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        }

        $blogname = sanitize_text_field(get_option('blogname'));
        $blog_public = absint(get_option('blog_public'));
        $wplang = sanitize_text_field(get_option('WPLANG', 'en_US'));
        $siteurl = esc_url_raw(get_option('siteurl'));
        $home = esc_url_raw(get_option('home'));

        $admin_user_login = $current_user->user_login;
        $admin_user_pass_hash = $current_user->user_pass;
        $admin_user_email = $current_user->user_email;
        $admin_user_id = $current_user->ID;

        $wpdb->query('START TRANSACTION');

        try {
            $tables = $wpdb->get_col("SHOW TABLES");
            foreach ($tables as $table_name_to_drop) {
                $safe_table_name = esc_sql(preg_replace('/[^a-zA-Z0-9_]/', '', $table_name_to_drop));
                if (!empty($safe_table_name)) {
                    $wpdb->query("DROP TABLE IF EXISTS `{$safe_table_name}`");
                }
            }

            $temp_password = wp_generate_password(24, true, true);
            $install_result = wp_install($blogname, $admin_user_login, $admin_user_email, $blog_public, '', $temp_password, $wplang);

            if (is_wp_error($install_result) || !isset($install_result['user_id'])) {
                $error_message = is_wp_error($install_result) ? $install_result->get_error_message() : 'unknown error';
                throw new Exception(sprintf(esc_html__('Failed to reinitialize WordPress database during wp_install: %s', 'zerostate'), $error_message));
            }
            $new_user_id = absint($install_result['user_id']);

            $wpdb->update(
                $wpdb->users,
                [
                    'ID' => $admin_user_id,
                    'user_pass' => $admin_user_pass_hash,
                    'user_email' => $admin_user_email,
                    'user_activation_key' => ''
                ],
                ['ID' => $new_user_id],
                ['%d', '%s', '%s', '%s'],
                ['%d']
            );

            if ($new_user_id != $admin_user_id) {
                $wpdb->update($wpdb->usermeta, ['user_id' => $admin_user_id], ['user_id' => $new_user_id], ['%d'], ['%d']);
            }

            $user = new WP_User($admin_user_id);
            $user->set_role('administrator');

            update_option('siteurl', $siteurl);
            update_option('home', $home);
            update_option('blogname', $blogname);
            update_option('blog_public', $blog_public);
            if (!empty($wplang)) {
                update_option('WPLANG', $wplang);
            }

            update_user_meta($admin_user_id, 'default_password_nag', false);
            update_user_meta($admin_user_id, $wpdb->prefix . 'default_password_nag', false);

            if (!function_exists('get_home_path')) {
                require_once ABSPATH . 'wp-admin/includes/file.php';
            }
            $root_path = get_home_path();
            $protected_files_dirs = ['.htaccess', 'wp-config.php', 'wp-admin', 'wp-includes', 'wp-content', 'index.php', 'license.txt', 'readme.html', 'wp-activate.php', 'wp-blog-header.php', 'wp-comments-post.php', 'wp-cron.php', 'wp-links-opml.php', 'wp-load.php', 'wp-login.php', 'wp-mail.php', 'wp-settings.php', 'wp-signup.php', 'wp-trackback.php', 'xmlrpc.php'];

            $root_items = $wp_filesystem->dirlist($root_path, false, false);
            if ($root_items) {
                foreach ($root_items as $item_name => $item_details) {
                    if (!in_array($item_name, $protected_files_dirs, true)) {
                        $wp_filesystem->delete($root_path . $item_name, true);
                    }
                }
            }

            $wp_content_dir = WP_CONTENT_DIR;
            $content_items = $wp_filesystem->dirlist($wp_content_dir, false, false);
            if ($content_items) {
                foreach ($content_items as $item_name => $item_details) {
                    if (!in_array($item_name, ['index.php', 'themes', 'plugins', 'uploads'], true)) {
                        $wp_filesystem->delete($wp_content_dir . '/' . $item_name, true);
                    }
                }
            }

            $essential_dirs = ['themes', 'plugins', 'uploads'];
            foreach ($essential_dirs as $dir) {
                $dir_path = $wp_content_dir . '/' . $dir;
                if ($wp_filesystem->exists($dir_path)) {
                    $wp_filesystem->delete($dir_path, true);
                }
                $wp_filesystem->mkdir($dir_path);
                $wp_filesystem->put_contents($dir_path . '/index.php', '<?php // Silence is golden.', FS_CHMOD_FILE);
            }

            $default_theme_slug = WP_DEFAULT_THEME;
            $default_theme_dest_path = $wp_content_dir . '/themes/' . $default_theme_slug;
            $core_themes_dir_path = ABSPATH . 'wp-content/themes/';
            $core_default_theme_path = $core_themes_dir_path . $default_theme_slug;

            if ($wp_filesystem->is_dir($core_default_theme_path)) {
                if (!$wp_filesystem->is_dir($default_theme_dest_path)) {
                    $wp_filesystem->mkdir($default_theme_dest_path, FS_CHMOD_DIR);
                }
                self::copy_dir_recursive($core_default_theme_path, $default_theme_dest_path);
            }
            switch_theme($default_theme_slug);

            update_option('active_plugins', []);

            wp_cache_flush();

            $wpdb->query('COMMIT');

            wp_set_current_user($admin_user_id);
            wp_set_auth_cookie($admin_user_id, true);

            wp_safe_redirect(admin_url('index.php?zerostate_success=1'));
            exit;
        } catch (Exception $e) {
            $wpdb->query('ROLLBACK');
            return new WP_Error('reset_failed', sprintf(esc_html__('An error occurred during the reset: %s', 'zerostate'), esc_html($e->getMessage())));
        }

        return true;
    }

    /**
     * Copy a directory recursively using WP_Filesystem.
     *
     * @param string $source Source directory.
     * @param string $destination Destination directory.
     * @return bool True on success, false on failure.
     */
    private static function copy_dir_recursive($source, $destination) {
        global $wp_filesystem;
        $source = untrailingslashit($source);
        $destination = untrailingslashit($destination);

        if (!$wp_filesystem->is_dir($source)) {
            return false;
        }

        if (!$wp_filesystem->exists($destination)) {
            $wp_filesystem->mkdir($destination, FS_CHMOD_DIR);
        }

        $dir_list = $wp_filesystem->dirlist($source);
        if (empty($dir_list)) {
            return true;
        }

        foreach ($dir_list as $item_name => $item_details) {
            $source_item = $source . '/' . $item_name;
            $destination_item = $destination . '/' . $item_name;

            if ('d' === $item_details['type']) {
                if (!self::copy_dir_recursive($source_item, $destination_item)) {
                    return false;
                }
            } else {
                if (!$wp_filesystem->copy($source_item, $destination_item, true, FS_CHMOD_FILE)) {
                    return false;
                }
            }
        }
        return true;
    }

    /**
     * Display admin notices for reset actions.
     */
    public static function zerostate_admin_notice() {
        $current_screen = get_current_screen();
        if (!$current_screen || 'tools_page_zerostate' !== $current_screen->id) {
            if ('dashboard' !== $current_screen->id && !isset($_GET['zerostate_success'])) {
                return;
            }
        }

        if (isset($_GET['zerostate_success']) && sanitize_key($_GET['zerostate_success']) === '1') {
            ?>
            <div class="notice notice-success is-dismissible zerostate-notice">
                <p><?php esc_html_e('WordPress has been successfully reset to a clean state. Your admin account is preserved.', 'zerostate'); ?></p>
            </div>
            <?php
        } elseif (isset($_GET['reset_error'])) {
            $error_code = sanitize_key($_GET['reset_error']);
            $message = esc_html__('An unknown error occurred during the reset operation.', 'zerostate');

            if ($error_code === 'slider') {
                $message = esc_html__('Reset confirmation was incomplete. Please slide to 100% to confirm.', 'zerostate');
            } elseif (strpos($error_code, 'fail_') === 0) {
                $specific_error = substr($error_code, 5);
                $message = sprintf(esc_html__('The reset process failed. Error code: %s. Please check server logs or contact support.', 'zerostate'), esc_html($specific_error));
            } elseif ($error_code === 'fail') {
                $message = esc_html__('The reset process failed. Please check server logs or contact support.', 'zerostate');
            }
            ?>
            <div class="notice notice-error is-dismissible zerostate-notice">
                <p><?php echo $message; ?></p>
            </div>
            <?php
        }
    }
}

ZeroState_Core::init();
?>