<?php
/**
 * Admin settings page for ZeroState.
 *
 * @package ZeroState
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

/**
 * Class ZeroState_Settings_Page
 *
 * Handles the admin settings page and UI for the ZeroState plugin.
 */
class ZeroState_Settings_Page {
    /**
     * Initialize the settings page.
     */
    public static function init() {
        add_action('admin_menu', [__CLASS__, 'register_menu']);
        add_action('admin_enqueue_scripts', [__CLASS__, 'enqueue_assets']);
    }

    /**
     * Register the admin menu.
     */
    public static function register_menu() {
        add_submenu_page(
            'tools.php',
            esc_html__('ZeroState Reset', 'zerostate'), // Page Title
            esc_html__('ZeroState Reset', 'zerostate'), // Menu Title
            'manage_options',
            'zerostate', // Menu Slug
            [__CLASS__, 'render_dashboard']
        );
    }

    /**
     * Enqueue admin assets.
     *
     * @param string $hook The current admin page hook.
     */
    public static function enqueue_assets($hook) {
        if ('tools_page_zerostate' !== $hook) {
            return;
        }

        // Enqueue Tailwind CSS (local copy).
        wp_enqueue_style(
            'zerostate-tailwind',
            ZEROSTATE_PLUGIN_URL . 'admin/css/tailwind.min.css',
            [],
            '2.2.19' // Version of Tailwind CSS file
        );

        // Enqueue custom styles for the settings page.
        wp_register_style('zerostate-settings-styles', false, [], ZEROSTATE_VERSION); // Use false for inline styles
        wp_enqueue_style('zerostate-settings-styles');

        $custom_css = "
            .zerostate-range::-webkit-slider-thumb {
                -webkit-appearance: none; appearance: none;
                width: 1.5rem; height: 1.5rem;
                background: #ef4444; border-radius: 50%; cursor: pointer;
                transition: transform 0.2s ease;
            }
            .zerostate-range::-webkit-slider-thumb:hover { transform: scale(1.1); }
            .zerostate-range::-moz-range-thumb {
                width: 1.5rem; height: 1.5rem;
                background: #ef4444; border-radius: 50%; cursor: pointer;
                transition: transform 0.2s ease; border: none;
            }
            .zerostate-range::-moz-range-thumb:hover { transform: scale(1.1); }
            .zerostate-notice { margin-left: 0 !important; margin-right:0 !important; }
            .glassmorphic-card {
                background: rgba(255, 255, 255, 0.1);
                backdrop-filter: blur(10px);
                -webkit-backdrop-filter: blur(10px);
                border: 1px solid rgba(255, 255, 255, 0.2);
                border-radius: 12px;
                transition: transform 0.3s ease, box-shadow 0.3s ease;
            }
            .glassmorphic-card:hover {
                transform: translateY(-5px);
                box-shadow: 0 10px 20px rgba(0, 0, 0, 0.2);
            }
            .upgrade-button {
                background-color: #22c55e;
                transition: background-color 0.3s ease;
            }
            .upgrade-button:hover {
                background-color: #16a34a;
            }
        ";
        wp_add_inline_style('zerostate-settings-styles', $custom_css);

        // Enqueue custom scripts.
        wp_register_script('zerostate-settings-js', false, ['jquery'], ZEROSTATE_VERSION, true); // Use false for inline script
        wp_enqueue_script('zerostate-settings-js');

        $inline_js = '
            jQuery(document).ready(function($) {
                // Slider handling for full reset confirmation
                var $slider = $("#zerostate_confirm_slider");
                var $submitButton = $("#zerostate-reset-form input[type=\"submit\"]");
                var $sliderValue = $("#zerostate-slider-value");

                if ($slider.length) {
                    $slider.on("input change", function() {
                        var value = parseInt($(this).val(), 10);
                        $sliderValue.text(value + "%");
                        $submitButton.prop("disabled", value !== 100);
                        if (value === 100) {
                            $(this).removeClass("bg-gray-300").addClass("bg-red-500");
                        } else {
                            $(this).removeClass("bg-red-500").addClass("bg-gray-300");
                        }
                    }).trigger("change");
                }

                // Full Reset Form submission
                $("#zerostate-reset-form").on("submit", function(e) {
                    if ($slider.length && parseInt($slider.val(), 10) !== 100) {
                        alert(' . wp_json_encode(esc_js(__('Please slide to 100% to confirm the reset.', 'zerostate'))) . ');
                        e.preventDefault();
                        return false;
                    }
                    if (!confirm(' . wp_json_encode(esc_js(__('FINAL WARNING: This will delete ALL site data and files (posts, pages, media, plugins, themes, etc.) except your current admin account. This cannot be undone. BACK UP YOUR SITE BEFORE PROCEEDING! You will need to manually install or reinstall a theme after the reset. Click OK to proceed.', 'zerostate'))) . ')) {
                        e.preventDefault();
                        return false;
                    }
                    $submitButton.prop("disabled", true).val(' . wp_json_encode(esc_js(__('Resetting...', 'zerostate'))) . ');
                    $submitButton.after("<span class=\"spinner is-active\" style=\"vertical-align: middle; margin-left: 5px;\"></span>");
                });
            });
        ';
        wp_add_inline_script('zerostate-settings-js', $inline_js);
    }

    /**
     * Render the admin dashboard.
     */
    public static function render_dashboard() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'zerostate'));
        }
        ?>
        <div class="wrap zerostate-dashboard bg-gray-100 min-h-screen transition-colors duration-300 py-8">
            <div class="container mx-auto px-4">
                <div class="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-6">
                    <h1 class="text-2xl font-semibold text-gray-900 mb-4"><?php esc_html_e('ZeroState WordPress Reset', 'zerostate'); ?></h1>
                </div>
                <p class="text-lg text-gray-700 mb-8"><?php esc_html_e('Reset your WordPress site to a clean, default state.', 'zerostate'); ?></p>

                <?php do_action('admin_notices'); ?>

                <div class="bg-white p-6 rounded-lg shadow">
                    <h2 class="text-2xl font-semibold text-gray-900 mb-4"><?php esc_html_e('Full Site Reset', 'zerostate'); ?></h2>
                    <div class="p-4 mb-4 bg-red-100 border border-red-400 rounded">
                        <p class="text-red-700 font-medium"><?php esc_html_e('WARNING: This action is irreversible and will delete ALL site data and files, including:', 'zerostate'); ?></p>
                        <ul class="list-disc pl-5 mt-2 text-red-700">
                            <li><?php esc_html_e('All posts, pages, comments, and media (database and files).', 'zerostate'); ?></li>
                            <li><?php esc_html_e('All plugins and themes.', 'zerostate'); ?></li>
                            <li><?php esc_html_e('All custom files in the WordPress root and wp-content directories (standard WP core files & dirs will be preserved or regenerated).', 'zerostate'); ?></li>
                            <li><?php esc_html_e('All custom database tables.', 'zerostate'); ?></li>
                        </ul>
                        <p class="mt-2 text-red-700 font-bold"><?php esc_html_e('Your current admin account will be preserved.', 'zerostate'); ?></p>
                        <p class="mt-2 text-red-700 font-bold"><?php esc_html_e('BACK UP YOUR SITE BEFORE PROCEEDING!', 'zerostate'); ?></p>
                        <p class="mt-2 text-red-700 font-bold"><?php esc_html_e('Note: You will need to manually install or reinstall a theme after the reset.', 'zerostate'); ?></p>
                    </div>
                    <form id="zerostate-reset-form" method="post" action="<?php echo esc_url(admin_url('admin.php')); ?>">
                        <input type="hidden" name="action" value="zerostate_full_reset_action">
                        <?php wp_nonce_field('zerostate_full_reset_action', 'zerostate_nonce'); ?>

                        <div class="mb-4">
                            <label for="zerostate_confirm_slider" class="block text-sm font-medium text-gray-700 mb-1"><?php esc_html_e('Slide to 100% to confirm reset:', 'zerostate'); ?></label>
                            <div class="flex items-center mt-2">
                                <input type="range" id="zerostate_confirm_slider" name="zerostate_confirm" min="0" max="100" value="0" class="zerostate-range w-full sm:w-48 h-2 bg-gray-300 rounded-full outline-none appearance-none cursor-pointer" />
                                <span id="zerostate-slider-value" class="ml-4 text-lg font-semibold text-red-500 w-12 text-center">0%</span>
                            </div>
                        </div>
                        <div>
                            <input type="submit" class="inline-flex items-center px-4 py-2 bg-red-600 hover:bg-red-700 text-white font-semibold rounded shadow disabled:opacity-50 disabled:cursor-not-allowed" value="<?php esc_attr_e('Reset Site Now', 'zerostate'); ?>" disabled>
                        </div>
                    </form>
                </div>

                <div class="mt-8">
                    <h2 class="text-2xl font-semibold text-gray-900 mb-6 text-center"><?php esc_html_e('Upgrade to Partial Reset Tools', 'zerostate'); ?></h2>
                    <p class="text-lg text-gray-700 mb-6 text-center"><?php esc_html_e('Unlock advanced partial reset tools to manage specific parts of your WordPress site with precision.', 'zerostate'); ?></p>
                    <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-6">
                        <div class="glassmorphic-card p-6 text-gray-700">
                            <h3 class="text-lg font-semibold mb-2"><?php esc_html_e('Delete Transients 🕒', 'zerostate'); ?></h3>
                            <p class="text-sm"><?php esc_html_e('Clear all temporary cache entries to boost performance and free up database space.', 'zerostate'); ?></p>
                        </div>
                        <div class="glassmorphic-card p-6 text-gray-700">
                            <h3 class="text-lg font-semibold mb-2"><?php esc_html_e('Reset Theme Options 🎨', 'zerostate'); ?></h3>
                            <p class="text-sm"><?php esc_html_e('Revert theme customizations made via the Customizer to their default settings.', 'zerostate'); ?></p>
                        </div>
                        <div class="glassmorphic-card p-6 text-gray-700">
                            <h3 class="text-lg font-semibold mb-2"><?php esc_html_e('Delete Inactive Themes 🖼️', 'zerostate'); ?></h3>
                            <p class="text-sm"><?php esc_html_e('Remove all unused themes, keeping only the active theme and its parent (if applicable).', 'zerostate'); ?></p>
                        </div>
                        <div class="glassmorphic-card p-6 text-gray-700">
                            <h3 class="text-lg font-semibold mb-2"><?php esc_html_e('Delete Inactive Plugins 🔌', 'zerostate'); ?></h3>
                            <p class="text-sm"><?php esc_html_e('Deactivate and delete all plugins except ZeroState and other active plugins.', 'zerostate'); ?></p>
                        </div>
                        <div class="glassmorphic-card p-6 text-gray-700">
                            <h3 class="text-lg font-semibold mb-2"><?php esc_html_e('Clean Uploads Folder 📁', 'zerostate'); ?></h3>
                            <p class="text-sm"><?php esc_html_e('Delete all files and folders in the /wp-content/uploads/ directory for a clean slate.', 'zerostate'); ?></p>
                        </div>
                        <div class="glassmorphic-card p-6 text-gray-700">
                            <h3 class="text-lg font-semibold mb-2"><?php esc_html_e('Empty or Delete Custom Tables 🗄️', 'zerostate'); ?></h3>
                            <p class="text-sm"><?php esc_html_e('Truncate or drop custom database tables not part of WordPress core.', 'zerostate'); ?></p>
                        </div>
                        <div class="glassmorphic-card p-6 text-gray-700">
                            <h3 class="text-lg font-semibold mb-2"><?php esc_html_e('Delete .htaccess File 📜', 'zerostate'); ?></h3>
                            <p class="text-sm"><?php esc_html_e('Remove the .htaccess file, which WordPress can regenerate as needed.', 'zerostate'); ?></p>
                        </div>
                    </div>
                    <div class="text-center mt-6">
                        <a href="https://securelywp.com/zerostate/?skip" class="inline-flex items-center px-6 py-3 upgrade-button text-white font-semibold rounded-lg shadow hover:shadow-lg"><?php esc_html_e('Upgrade Now', 'zerostate'); ?></a>
                    </div>
                </div>

                <div class="mt-8 text-center text-sm text-gray-500">
                    <p><?php printf(esc_html__('ZeroState Plugin Version %s', 'zerostate'), esc_html(ZEROSTATE_VERSION)); ?></p>
                    <p><?php esc_html_e('Please use with extreme caution. Always back up your site before performing any reset operations.', 'zerostate'); ?></p>
                </div>
            </div>
        </div>
        <?php
    }
}

ZeroState_Settings_Page::init();
?>