<?php
/**
 * ZenVPN Settings
 *
 * @package zenVPN
 * @subpackage settings
 * @license https://www.gnu.org/licenses/gpl-3.0.txt GNU/GPLv3
 */

namespace zenVPN\Settings;

use zenVPN\Blocker\ZV_IP_Blocker;

/**
 * Class for register and render plugin's settings and settings' page.
 */
class ZV_Settings implements ZV_Settings_Interface {

	/**
	 * Settings list
	 *
	 * @var array<SettingMeta>
	 */
	private array $settings;

	/**
	 * Sigleton instance
	 *
	 * @var ZV_Settings
	 */
	private static ZV_Settings $instance;

	/**
	 * Class constructor.
	 *
	 * @param array<SettingMeta> $settings settings array.
	 */
	private function __construct( array $settings ) {
		$this->settings = $settings;
	}

	/**
	 * Get instance of current class.
	 */
	public static function get_instance(): ZV_Settings {
		if ( ! isset( self::$instance ) ) {
			self::init( array() );
		}
		return self::$instance;
	}

	/**
	 * Init function.
	 *
	 * @param array<SettingMeta> $settings settings array.
	 */
	public static function init( array $settings ): ZV_Settings {
		if ( ! isset( self::$instance ) ) {
			self::$instance = new self( $settings );
		}
		return self::$instance;
	}

	/**
	 * Register plugin setting's option.
	 *
	 * @return void
	 */
	private function register_settings_option(): void {
		register_setting(
			ZV_PREFIX . 'settings',
			ZV_PREFIX . 'settings',
			array( $this, 'validate_settings' )
		);

		register_setting(
			ZV_PREFIX . 'ip_settings',
			ZV_PREFIX . 'ip_settings'
		);
	}

	/**
	 * Register plugin settings' sections.
	 *
	 * @return void
	 */
	private function register_settings_section(): void {
		add_settings_section(
			ZV_PREFIX . 'main_section',
			esc_html__( 'General', 'zenvpn' ),
			array( $this, 'main_section_callback' ),
			ZV_PREFIX . 'settings'
		);
	}

	/**
	 * Register settings fields.
	 *
	 * @return void
	 */
	private function register_settings_fields(): void {
		foreach ( $this->settings as $settings_field ) {
			add_settings_field(
				$settings_field['name'],
				'',
				array( $this, $settings_field['callback'] ),
				ZV_PREFIX . 'settings',
				$settings_field['section'],
				$settings_field['args']
			);
		}
	}

	/**
	 * Register settings.
	 *
	 * @return void
	 */
	public function register_settings(): void {
		$this->register_settings_option();
		$this->register_settings_section();
		$this->register_settings_fields();
	}

	/**
	 * Add plugin's option page.
	 *
	 * @return void
	 */
	public function add_options_page(): void {
		add_options_page(
			esc_html__( 'zenVPN', 'zenvpn' ),
			esc_html__( 'zenVPN', 'zenvpn' ),
			'manage_options',
			ZV_PREFIX . 'settings',
			array( $this, 'render_options_page' )
		);
	}

	/**
	 * Render plugin's option page.
	 *
	 * @return void
	 */
	public function render_options_page(): void {
		?>
		<p id="statusMessage" class="notice hidden"></p>
		<h1>
			<?php esc_html_e( 'Configure zenVPN protection', 'zenvpn' ); ?>
		</h1>
		<p>
			<?php echo esc_html__( 'This plugin protects your WordPress Admin from intrusion attempts by allowing access only through zenVPN host you configure. It requires an account with zenVPN and you can use this plugin for free as long as you do not need other zenVPN services.', 'zenvpn' ); ?>
		</p>
		<div class="wrap">
			<div class="postbox-container-outer">
				<div class="postbox">
					<div class="inside">
						<form id="settingsForm" method="post" action="options.php">
							<?php
							settings_fields( ZV_PREFIX . 'settings' );
							$this->main_section_callback();
							do_settings_fields( ZV_PREFIX . 'settings', ZV_PREFIX . 'main_section' );
							wp_nonce_field( 'zenvpn_save_nonce' );
							?>
					</div><!-- end inside -->
					</form>
				</div><!-- end postbox -->
				<div class="postbox-container">
					<div class="postbox">
						<div class="inside">
							<h2>
								<?php esc_html_e( 'New to zenVPN?', 'zenvpn' ); ?>
							</h2>
							<p class="txt-big">
								<?php echo esc_html__( 'If you\'re new to zenVPN —', 'zenvpn' ); ?>
								<a href="<?php echo esc_url( ZEN_VPN_APP . '/signup?next=/api-access%3Fwp&domain_name=' . str_replace( array( 'https://', 'http://' ), '', get_site_url( null, '', 'https' ) ) ); ?>" target="_blank">
									<?php echo esc_html__( 'create a free account', 'zenvpn' ); ?>
								</a>
							</p>
						</div><!-- end inside -->
					</div><!-- end postbox -->
				</div><!-- end postbox-container -->
			</div><!-- end postbox-container-outer -->
			<div class="postbox nobg">
				<div class="inside">
					<h2>
						<?php esc_html_e( 'Explore zenVPN or get support', 'zenvpn' ); ?>
					</h2>
					<p>
						<?php echo esc_html__( 'zenVPN can tunnel your entire team to this admin panel and other resources. It is inexpensive way to protect your admin access from intrusion attempts. With split tunneling you may keep zenVPN running on your computer all the time, and the rest of the traffic will use your default Internet connection. Basic VPN provides you and your team mates with classic VPN features.', 'zenvpn' ); ?>
					</p>
					<div class="icon-list-outer">
						<ul class="icon-list">
							<li class="icon-list__item"><span class="dashicons dashicons-shield"></span>
								<?php echo esc_html__( 'An extra layer of protection against intrusion attempts', 'zenvpn' ); ?>
							</li>
							<li class="icon-list__item"><span class="dashicons dashicons-groups"></span>
								<?php echo esc_html__( 'Hassle-free, zero-config for your entire team', 'zenvpn' ); ?>
							</li>
						</ul>
						<ul class="icon-list">
							<li class="icon-list__item"><span class="dashicons dashicons-randomize"></span>
								<?php echo esc_html__( 'Split-tunneling, so your main Internet stays the same', 'zenvpn' ); ?>
							</li>
							<li class="icon-list__item"><span class="dashicons dashicons-lock"></span>
								<?php echo esc_html__( 'Easy access to classic VPN', 'zenvpn' ); ?>
							</li>
						</ul>
					</div>
					<a href="<?php echo esc_url( ZEN_VPN_WEBSITE ); ?>" class="button button-primary" target="_blank">
						<?php echo esc_html__( 'Explore all features', 'zenvpn' ); ?>
					</a>
					<p>
						<?php echo esc_html__( 'If you need any support, please email us at support@zenvpn.net', 'zenvpn' ); ?>
					</p>					
				</div><!-- end inside -->
			</div><!-- end postbox -->
		</div><!-- end wrap -->
		<?php
	}

	/**
	 * Unregister plugin's settings. Called on hook deactivation.
	 *
	 * @return void
	 */
	public function unregister_settings(): void {
		unregister_setting(
			ZV_PREFIX . 'settings',
			ZV_PREFIX . 'settings'
		);

		unregister_setting(
			ZV_PREFIX . 'ip_settings',
			ZV_PREFIX . 'ip_settings'
		);

		delete_option( ZV_PREFIX . 'settings' );
		delete_option( ZV_PREFIX . 'ip_settings' );
	}

	/**
	 * Validate settings.
	 *
	 * @param array{token?: string, zenvpn_protect_wp_admin?: bool} $input array of settings.
	 * @return array{token?: string, zenvpn_protect_wp_admin?: bool}
	 */
	public function validate_settings( array $input ): array {
		$output = array();

		if ( isset( $input['token'] ) ) {
			$output['token'] = sanitize_text_field( wp_unslash( $input['token'] ) );
		}

		return $output;
	}

	/**
	 * Callback function for main section.
	 *
	 * @return void
	 */
	public function main_section_callback(): void {
		?>
		<h2>
			<?php echo esc_html__( 'Link this to your zenVPN', 'zenvpn' ); ?>
		</h2>
		<p class="txt-big">
			<?php echo wp_kses_post( __( 'To protect the access to your WP admin, this plugin needs to fetch your tunneling configuration from zenVPN cloud and compare incoming traffic against authorized sources.', 'zenvpn' ) ); ?>
		</p>
		<p class="txt-big">
			<?php echo wp_kses_post( __( 'Enter your zenVPN key, as you can see in <a href="https://app.zenvpn.net/api-access" target="_blank">your account.</a>', 'zenvpn' ) ); ?>
		</p>
		<?php
	}

	/**
	 * Callback function for token field.
	 *
	 * @param array{label_for: string, description?: string} $args args for token field callback.
	 * @return void
	 */
	public function token_field_callback( array $args ): void {
		/**
		 * Get the settings option from the database.
		 *
		 * @var array{token?: string, zenvpn_protect_wp_admin?: bool} settings
		 */
		$settings    = get_option( ZV_PREFIX . 'settings' );
		$token       = ! empty( $settings['token'] ) ? $settings['token'] : '';
		$field_id    = $args['label_for'];
		$description = $args['description'];
		?>
		<div class="button-container">
			<input type="text" name="<?php echo esc_attr( ZV_PREFIX . 'settings[token]' ); ?>"
				id="<?php echo esc_attr( $field_id ); ?>" value="<?php echo esc_attr( $token ); ?>"
				placeholder="<?php echo esc_attr( $description ); ?>" disabled />

			<button id=editButton type=button class="button button-primary">
				<?php echo esc_html__( 'Edit', 'zenvpn' ); ?>
			</button>

			<button id=testButton type=button class="button button-primary hidden">
				<?php echo esc_html__( 'Test API key', 'zenvpn' ); ?>
			</button>
		</div>
		<?php
	}

	/**
	 * Function to render a checkbox field for wp-admin protection settings.
	 *
	 * @param array{label_for: string, description?: string} $args array of arguments for the field.
	 * @return void
	 */
	public function protect_admin_field_callback( array $args ): void {
		/**
		 * Get the settings option from the database.
		 *
		 * @var array{token?: string, zenvpn_protect_wp_admin?: bool} settings
		 */
		$settings = get_option( ZV_PREFIX . 'settings' );

		// Get the field id and description from the arguments.
		$field_id = $args['label_for'];

		$description = $args['description'] ?? '';

		// Remove the prefix and the brackets from the field id using regular expressions.
		$field_key = preg_replace( '/^' . ZV_PREFIX . 'settings\[|\]$/', '', $field_id );

		// Get the protection value from the settings option or an empty string.
		$protect = ! empty( $settings[ $field_key ] ) ? $settings[ $field_key ] : '';

		// Render the checkbox input and the description paragraph.
		?>
		<div id="optionsBlock" class="hidden">
			<?php if ( ! empty( $description ) ) : ?>

				<h2><?php echo esc_html__( 'Switch the protection on', 'zenvpn' ); ?>
				</h2>
				<?php echo esc_html__( 'Before you block access to everyone else, make sure you yourself are connected over that allowed tunnel.', 'zenvpn' ); ?>
				<?php echo esc_html__( 'Please also make sure to invite all other admin users to zenVPN before turning the guard on.', 'zenvpn' ); ?>
				</p>
				<p>
				<label class="switch">
				<input type="checkbox" name="<?php echo esc_attr( ZV_PREFIX . 'settings[' . $field_key . ']' ); ?>"
					id="<?php echo esc_attr( $field_id ); ?>" value="1" <?php checked( $protect, 1 ); ?> />
				<span class="slider round"></span>
				</label>
					<?php echo esc_html( $description ); ?>
				</p>
				<div>
					<div class="zenvpn-alert-icon">!</div>
					<p>
					<?php
					printf(
						/* translators: %s: User's website */
						esc_html__( 'If you ever lock yourself out of your WP Admin, you may take this access control down on zenvpn.net by removing the tunnel to %s. This plugin will allow any IP to access your WP admin again in that case.', 'zenvpn' ),
						esc_html( str_replace( array( 'https://', 'http://' ), '', get_site_url( null, '', 'https' ) ) )
					);
					?>
					</p>
				</div>
			<?php endif; ?>
		</div>
		<div id="buttonsBlock" class="button-container hidden">
			<button id=cancelButton type=button class="button button-primary">
				<?php echo esc_html__( 'Cancel', 'zenvpn' ); ?>
			</button>

			<button id=saveButton type=button class="button button-primary" disabled>
				<?php echo esc_html__( 'Save API key', 'zenvpn' ); ?>
			</button>
		</div>
		<?php
	}

	/**
	 * Function to save position and source of needed ip.
	 *
	 * @param string $ip_string string with ip address.
	 * @return void
	 */
	public function save_ip_settings( string $ip_string ) {
		$result     = array();
		$places     = array(
			'REMOTE_ADDR',
			'X-REAL-IP',
			'True-Client-IP',
			'FORWARDED',
			'HTTP_X_FORWARDED',
			'X-FORWARDED',
			'HTTP_X_FORWARDED_FOR',
			'X-FORWARDED-FOR',
			'HTTP_CLIENT_IP',
			'X-Client-IP',
			'HTTP_CF_CONNECTING_IP',
		);
		$ip_blocker = new ZV_IP_Blocker();
		$is_found   = false;
		foreach ( $places as $place ) {
			if ( ! $is_found && isset( $_SERVER[ $place ] ) ) {
				$result = self::build_result( $place, 'server', $ip_string, $ip_blocker );
				if ( ! empty( $result ) ) {
					$is_found = true;
				}
			}

			if ( ! $is_found && isset( $_SERVER[ 'HTTP_' . $place ] ) ) {
				$result = self::build_result( $place, 'header', $ip_string, $ip_blocker );
				if ( ! empty( $result ) ) {
					$is_found = true;
				}
			}
		}

		update_option( ZV_PREFIX . 'ip_settings', $result );
	}

	/**
	 * Function to build result array.
	 *
	 * @param string        $place name of place, where ip is searched.
	 * @param string        $type name of type.
	 * @param string        $ip_string ip string.
	 * @param ZV_IP_Blocker $ip_blocker IP_Blokcer object.
	 *
	 * @return IPSetting empty if not found. With data if found.
	 */
	private static function build_result( string $place, string $type, string $ip_string, ZV_IP_Blocker $ip_blocker ) {
		$value = '';
		if ( isset( $_SERVER[ $place ] ) ) {
			$value = sanitize_text_field( wp_unslash( $_SERVER[ $place ] ) );
		}
		$value = str_replace( '"', '', $value );
		$ips   = explode( ',', $value );
		foreach ( $ips as $index => $ip ) {
			$ip          = trim( $ip );
			$is_valid_ip = $ip_blocker::compare_ips( $ip, $ip_string );
			if ( $is_valid_ip ) {
				$result = array(
					'type'     => $type,
					'name'     => $place,
					'position' => $index,
				);

				return $result;
			}
		}

		return array();
	}

	/**
	 * Get IP settings.
	 *
	 * @return IPSetting
	 */
	public function get_ip_settings() {
		return get_option( ZV_PREFIX . 'ip_settings' );
	}

	/**
	 * Function to get user's ip address.
	 *
	 * @return string
	 */
	public function get_user_ip() {
		$ip_settings = $this->get_ip_settings();
		if ( ! empty( $ip_settings ) ) {
			$type     = $ip_settings['type'];
			$name     = $ip_settings['name'];
			$position = $ip_settings['position'];
			$key      = 'server' === $type ? $name : 'HTTP_' . $name;
			$value    = sanitize_text_field( wp_unslash( $_SERVER[ $key ] ?? '' ) );
			$ips      = explode( ',', $value );
			$ip       = $ips[ $position ] ?? '';
			$ip       = trim( $ip );
			return $ip;
		}

		return '';
	}
}
