<?php
/**
 * Plugin Name:     ZenCommunity
 * Plugin URI:      http://kodezen.com
 * Description:     All-in-one WordPress community plugin: groups, posts, real-time chat, support tickets & live chat.
 * Version:         1.1.0
 * Author:          kodezen
 * Author URI:      http://zencommunity.pro/
 * License:         GPL-3.0+
 * License URI:     http://www.gnu.org/licenses/gpl-3.0.txt
 * Text Domain:     zencommunity
 * Domain Path:     /i18n/languages/
 *
 * Requires PHP: 7.4
 * Requires at least: 6.8
 * Tested up to: 6.8
 *
 * @package ZenCommunity
 */

use ZenCommunity\ActionQueue;
use ZenCommunity\Exceptions\ZencommunityException;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

final class ZenCommunity {

	private function __construct() {
		$this->define_constants();
		$this->set_global_settings();
		$this->load_dependency();
		register_activation_hook( __FILE__, [ $this, 'activate' ] );
		register_deactivation_hook( __FILE__, [ $this, 'deactivate' ] );
		add_action( 'plugins_loaded', [ $this, 'on_plugins_loaded' ] );
		add_action( 'zencommunity_loaded', [ $this, 'init_plugin' ] );
	}

	public static function init() {
		static $instance = false;

		if ( ! $instance ) {
			$instance = new self();
		}

		return $instance;
	}
	public function define_constants() {
		/**
		 * Defines CONSTANTS for Whole plugins.
		 */
		define( 'ZENCOMMUNITY_VERSION', '1.1.0' );
		define( 'ZENCOMMUNITY_DB_VERSION', '1.0.0' );
		define( 'ZENCOMMUNITY_SETTINGS_NAME', 'zencommunity_settings' );
		define( 'ZENCOMMUNITY_ROLES_OPTION_NAME', 'zencommunity_roles' );
		define( 'ZENCOMMUNITY_PLUGIN_FILE', __FILE__ );
		define( 'ZENCOMMUNITY_PLUGIN_BASENAME', plugin_basename( __FILE__ ) );
		define( 'ZENCOMMUNITY_PLUGIN_SLUG', 'zencommunity' );
		define( 'ZENCOMMUNITY_PLUGIN_ROOT_URI', plugins_url( '/', __FILE__ ) );
		define( 'ZENCOMMUNITY_ROOT_DIR_PATH', plugin_dir_path( __FILE__ ) );
		define( 'ZENCOMMUNITY_APP_ROUTE_QUERY_VAR', 'zencommunity_app_route' );
		define( 'ZENCOMMUNITY_INCLUDES_DIR_PATH', ZENCOMMUNITY_ROOT_DIR_PATH . 'includes/' );
		define( 'ZENCOMMUNITY_TEMPLATES_DIR_PATH', ZENCOMMUNITY_ROOT_DIR_PATH . 'templates/' );
		define( 'ZENCOMMUNITY_ASSETS_DIR_PATH', ZENCOMMUNITY_ROOT_DIR_PATH . 'assets/' );
		define( 'ZENCOMMUNITY_ADDONS_DIR_PATH', ZENCOMMUNITY_ROOT_DIR_PATH . 'addons/' );
		define( 'ZENCOMMUNITY_ASSETS_URI', ZENCOMMUNITY_PLUGIN_ROOT_URI . 'assets/' );
		define( 'ZENCOMMUNITY_TEMPLATE_DEBUG_MODE', false );
	}

	/**
	 * When WP has loaded all plugins, trigger the `zencommunity_loaded` hook.
	 *
	 * This ensures `zencommunity_loaded` is called only after all other plugins
	 * are loaded, to avoid issues caused by plugin directory naming changing
	 *
	 * @since 1.0.0
	 */
	public function on_plugins_loaded() {
		do_action( 'zencommunity_loaded' );
	}

	/**
	 * Initialize the plugin
	 *
	 * @return void
	 */
	public function init_plugin() {
		// Init action.
		do_action( 'zencommunity_before_init' );
		$this->dispatch_hooks();
		$this->load_addons();
		$this->load_commands();
		// Init action.
		do_action( 'zencommunity_init' );
	}

	/**
	 * Initialize the plugin
	 *
	 * @return void
	 */
	public function load_addons() {
		$addon_manager = new ZenCommunity\Classes\Addon\Manager;
		$addon_manager->boot_active_addons();
	}

	public function load_commands() {
		ZenCommunity\Classes\Command\Register::init();
	}

	public function dispatch_hooks() {
		ZenCommunity\Migration::init();
		ZenCommunity\Admin::get_instance();
		ZenCommunity\RewriteRules::init();
		ZenCommunity\Frontend::init();
		ZenCommunity\Assets::init();
		ZenCommunity\RestApiInit::init();
		ZenCommunity\AdminMenu::init();
	}

	public function load_dependency() {
		// Autoload prefixed packages.
		if ( file_exists( ZENCOMMUNITY_ROOT_DIR_PATH . 'vendor/prefixed/autoload.php' ) ) {
			require_once ZENCOMMUNITY_ROOT_DIR_PATH . 'vendor/prefixed/autoload.php';
		}

		// Autoload other packages.
		if ( file_exists( ZENCOMMUNITY_ROOT_DIR_PATH . 'vendor/autoload.php' ) ) {
			require_once ZENCOMMUNITY_ROOT_DIR_PATH . 'vendor/autoload.php';
		}

		require_once ZENCOMMUNITY_INCLUDES_DIR_PATH . 'autoload.php';
		require_once ZENCOMMUNITY_INCLUDES_DIR_PATH . 'functions.php';
		require_once ZENCOMMUNITY_INCLUDES_DIR_PATH . 'hooks.php';
	}

	public function set_global_settings() {
		$GLOBALS['zencommunity_settings'] = json_decode( get_option( ZENCOMMUNITY_SETTINGS_NAME, '{}' ) );
		$GLOBALS['zencommunity_settings']->site_logo = empty( $GLOBALS['zencommunity_settings']->site_logo ?? false ) ? str_replace( rtrim( site_url(), '/' ), '',  plugins_url( '/assets/images/icon_logo.png', ZENCOMMUNITY_PLUGIN_FILE ) ) : $GLOBALS['zencommunity_settings']->site_logo;
	}

	/**
	 * Get queue instance.
	 *
	 * @return ActionQueue
	 * @throws ZencommunityException
	 */
	public function queue(): ActionQueue {
		return ActionQueue::get_instance();
	}

	public function activate() {
		ZenCommunity\Installer::init();
	}

	public function deactivate() {
		$this->queue()->cancel_all( 'zencommunity_cleanup_uploads' );
	}
}

/**
 * Initializes the main plugin
 *
 * @return \ZenCommunity
 */
function zencommunity_start() {
	return ZenCommunity::init();
}

// Plugin Start
zencommunity_start();