<?php
namespace ZenCommunity\Classes;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use ZenCommunity\Exceptions\ZencommunityException;
class MenuManager {
	private string $menu_key;

	const BASE_OPTION_KEY        = 'zencommunity_menu';
	const ACTIVE_STATUS_OPTION   = 'zencommunity_menu_active_status';
	const ORDER_OPTION_SUFFIX    = '_order';
	private array $default_items_map = [];

	public function __construct( string $menu_key = 'primary_menu' ) {
		$this->init_default_menu();
		$this->menu_key = $menu_key;
	}
	
	public function init_default_menu() : void {
		$this->default_items_map = [
			'primary_menu' => [
				[
					'id'    => 'menu_feeds',
					'type'  => 'default',
					'label' => 'Feeds',
					'status' => true,
					'icon'  => [
						'type' => 'emoji',
						'src'  => '<svg xmlns="http://www.w3.org/2000/svg" width="18" height="19" viewBox="0 0 18 19" fill="none">
						<path d="M7.33366 10.8332V16.6665H10.667V10.8332H7.33366ZM17.3337 15.8332C17.3337 16.4962 17.0701 17.1327 16.6012 17.6016C16.1324 18.0703 15.4966 18.3332 14.8337 18.3332H3.16699C2.50405 18.3332 1.86823 18.0703 1.39941 17.6016C0.930573 17.1327 0.666992 16.4962 0.666992 15.8332V8.33317C0.666989 7.96975 0.745982 7.61059 0.898926 7.28092C1.05197 6.95109 1.27553 6.65872 1.55322 6.42399L7.38249 1.42806L7.38656 1.4248L7.56152 1.28971C7.98128 0.994267 8.48345 0.833984 9.00033 0.833984C9.5172 0.833984 10.0194 0.994267 10.4391 1.28971L10.6141 1.4248L10.6182 1.42806L16.4474 6.42399L16.5492 6.51432C16.78 6.73208 16.9678 6.99229 17.1017 7.28092C17.2547 7.61059 17.3337 7.96975 17.3337 8.33317V15.8332ZM12.3337 16.6665H14.8337C15.0546 16.6665 15.2666 16.5793 15.4229 16.4232C15.5791 16.2669 15.667 16.0542 15.667 15.8332V8.33317C15.667 8.21203 15.6407 8.09231 15.5897 7.98242C15.5387 7.87248 15.4641 7.77502 15.3716 7.69678L15.3675 7.69271L15.9095 7.06038L15.9347 7.02946L15.3675 7.69271L9.53825 2.69678V2.69759C9.38784 2.57047 9.19726 2.50065 9.00033 2.50065C8.80339 2.50065 8.61281 2.57047 8.4624 2.69759L8.46159 2.69678L2.63314 7.69271L2.0651 7.02946L2.09115 7.06038L2.63314 7.69271L2.62907 7.69678C2.53651 7.77502 2.46198 7.87248 2.41097 7.98242C2.35999 8.09231 2.33366 8.21203 2.33366 8.33317V15.8332C2.33366 16.0542 2.42152 16.2669 2.5778 16.4232C2.73406 16.5793 2.94608 16.6665 3.16699 16.6665H5.66699V10.8332C5.66709 10.3913 5.8428 9.96726 6.15527 9.65479C6.46782 9.34234 6.89172 9.1665 7.33366 9.1665H10.667C11.1089 9.1665 11.5328 9.34234 11.8454 9.65479C12.1578 9.96726 12.3336 10.3913 12.3337 10.8332V16.6665Z" fill="currentColor"/>
						</svg>',
					],
				],
				[
					'id'    => 'menu_groups',
					'type'  => 'default',
					'label' => 'Groups',
					'status' => true,
					'icon'  => [
						'type' => 'emoji',
						'src'  => '<svg xmlns="http://www.w3.org/2000/svg" width="20" height="18" viewBox="0 0 20 18" fill="none">
						<path d="M12.4997 16.4993V14.8327C12.4997 14.1696 12.2361 13.5339 11.7673 13.0651C11.2984 12.5963 10.6627 12.3327 9.99967 12.3327H4.99967C4.33663 12.3327 3.70094 12.5963 3.2321 13.0651C2.76326 13.5339 2.49967 14.1696 2.49967 14.8327V16.4993C2.49967 16.9596 2.12658 17.3327 1.66634 17.3327C1.2061 17.3327 0.833008 16.9596 0.833008 16.4993V14.8327C0.833008 13.7276 1.27231 12.6681 2.05371 11.8867C2.83511 11.1053 3.89461 10.666 4.99967 10.666H9.99967C11.1047 10.666 12.1642 11.1053 12.9456 11.8867C13.727 12.6681 14.1663 13.7276 14.1663 14.8327V16.4993C14.1663 16.9596 13.7932 17.3327 13.333 17.3327C12.8728 17.3327 12.4997 16.9596 12.4997 16.4993ZM17.4997 16.4993V14.8335L17.4907 14.6268C17.4506 14.1467 17.2726 13.6867 16.9756 13.3035C16.6362 12.8658 16.161 12.5534 15.6247 12.4149C15.179 12.2998 14.9115 11.8449 15.0265 11.3993C15.1417 10.9537 15.5958 10.6861 16.0413 10.8011C16.9352 11.0319 17.7275 11.5526 18.2931 12.2822C18.8587 13.0118 19.1656 13.9088 19.1663 14.8319V16.4993C19.1663 16.9596 18.7932 17.3327 18.333 17.3327C17.8728 17.3327 17.4997 16.9596 17.4997 16.4993ZM9.99967 4.83268C9.99967 3.45197 8.88039 2.33268 7.49967 2.33268C6.11896 2.33268 4.99967 3.45197 4.99967 4.83268C4.99967 6.21339 6.11896 7.33268 7.49967 7.33268C8.88039 7.33268 9.99967 6.21339 9.99967 4.83268ZM15.0062 4.83675C15.0062 4.28168 14.8216 3.74203 14.4813 3.30355C14.1409 2.86511 13.664 2.55255 13.1263 2.41488C12.6805 2.30071 12.4116 1.84672 12.5257 1.40088C12.6399 0.955088 13.0939 0.686156 13.5397 0.800293C14.436 1.02977 15.2307 1.55138 15.798 2.28223C16.3651 3.01298 16.6728 3.91175 16.6728 4.83675C16.6728 5.76193 16.3653 6.66125 15.798 7.39209C15.2307 8.1229 14.4359 8.64373 13.5397 8.87321C13.0939 8.98737 12.6399 8.71848 12.5257 8.27262C12.4116 7.8268 12.6805 7.37278 13.1263 7.25863C13.664 7.12094 14.1409 6.80845 14.4813 6.36995C14.8217 5.93145 15.0062 5.39186 15.0062 4.83675ZM11.6663 4.83268C11.6663 7.13387 9.80086 8.99935 7.49967 8.99935C5.19849 8.99935 3.33301 7.13387 3.33301 4.83268C3.33301 2.5315 5.19849 0.666016 7.49967 0.666016C9.80086 0.666016 11.6663 2.5315 11.6663 4.83268Z" fill="currentColor"/>
						</svg>',
					],
				],
				[
					'id'    => 'menu_members',
					'type'  => 'default',
					'label' => 'Members',
					'status' => true,
					'icon'  => [
						'type' => 'emoji',
						'src'  => '<svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 20 20" fill="none">
						<path d="M14.9997 17.4993V15.8327C14.9997 15.1696 14.7361 14.5339 14.2673 14.0651C13.7984 13.5963 13.1627 13.3327 12.4997 13.3327H7.49967C6.83663 13.3327 6.20094 13.5963 5.7321 14.0651C5.26326 14.5339 4.99967 15.1696 4.99967 15.8327V17.4993C4.99967 17.9596 4.62658 18.3327 4.16634 18.3327C3.7061 18.3327 3.33301 17.9596 3.33301 17.4993V15.8327C3.33301 14.7276 3.77231 13.6681 4.55371 12.8867C5.33511 12.1053 6.39461 11.666 7.49967 11.666H12.4997C13.6047 11.666 14.6642 12.1053 15.4456 12.8867C16.227 13.6681 16.6663 14.7276 16.6663 15.8327V17.4993C16.6663 17.9596 16.2932 18.3327 15.833 18.3327C15.3728 18.3327 14.9997 17.9596 14.9997 17.4993ZM12.4997 5.83268C12.4997 4.45197 11.3804 3.33268 9.99967 3.33268C8.61896 3.33268 7.49967 4.45197 7.49967 5.83268C7.49967 7.21339 8.61896 8.33268 9.99967 8.33268C11.3804 8.33268 12.4997 7.21339 12.4997 5.83268ZM14.1663 5.83268C14.1663 8.13387 12.3009 9.99935 9.99967 9.99935C7.69849 9.99935 5.83301 8.13387 5.83301 5.83268C5.83301 3.5315 7.69849 1.66602 9.99967 1.66602C12.3009 1.66602 14.1663 3.5315 14.1663 5.83268Z" fill="currentColor"/>
						</svg>',
					],
				],
				[
					'id'    => 'menu_settings',
					'type'  => 'default',
					'label' => 'Settings',
					'status' => true,
					'show' => current_user_can( 'manage_options' ),
					'icon'  => [
						'type' => 'emoji',
						'src'  => '<svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 20 20" fill="none">
					<path d="M11.016 3.33398C11.016 3.11297 10.9281 2.90107 10.7718 2.74479C10.6156 2.58858 10.4036 2.50065 10.1826 2.50065H9.81643C9.59547 2.50065 9.38351 2.58858 9.22724 2.74479C9.07096 2.90107 8.9831 3.11297 8.9831 3.33398V3.48454C8.98265 3.92294 8.86639 4.35416 8.647 4.73372C8.42762 5.1131 8.11262 5.42849 7.7331 5.64762L7.73228 5.64681L7.37665 5.85433L6.8371 4.92578L6.95754 5.13411L7.37665 5.85433L7.37421 5.85596C6.99425 6.07524 6.56291 6.19043 6.12421 6.19043C5.68548 6.19037 5.25416 6.07532 4.87421 5.85596L4.77411 5.80225L4.7497 5.78922L5.16636 5.06738L5.16799 5.0625L4.7497 5.78922C4.55849 5.67894 4.33143 5.6492 4.11819 5.70622C3.93147 5.75615 3.76862 5.86885 3.65676 6.02441L3.612 6.09359L3.4289 6.41016C3.31931 6.601 3.28992 6.82739 3.3467 7.04004C3.40373 7.2533 3.54305 7.43558 3.73407 7.54622C3.74917 7.55497 3.76432 7.56422 3.77883 7.57389L3.90334 7.6569C4.26099 7.87355 4.55877 8.17546 4.76923 8.53743C4.98852 8.91461 5.10514 9.34301 5.10777 9.7793V10.2057L5.10289 10.3701C5.07915 10.7535 4.96784 11.1271 4.77574 11.4614C4.5585 11.8394 4.24539 12.1529 3.86916 12.3729L3.86998 12.3737L3.74465 12.4486L3.73407 12.4551L3.32799 11.7536L3.31171 11.7259L3.73407 12.4551C3.54305 12.5657 3.40373 12.748 3.3467 12.9613C3.28992 13.1739 3.31931 13.4003 3.4289 13.5911L3.61119 13.9077C3.72183 14.0987 3.90493 14.2381 4.11819 14.2951C4.33143 14.3521 4.55849 14.3224 4.7497 14.2121L5.16799 14.938L5.16636 14.9339L4.7497 14.2121C4.75767 14.2075 4.76599 14.2034 4.77411 14.1991L4.87421 14.1453L5.01907 14.068C5.36213 13.8989 5.74028 13.8109 6.12421 13.8109C6.56291 13.8109 6.99425 13.9261 7.37421 14.1453L7.37665 14.147L7.73228 14.3537C8.11181 14.5728 8.42762 14.8882 8.647 15.2676C8.86639 15.6471 8.98265 16.0784 8.9831 16.5168V16.6673C8.9831 16.8883 9.07096 17.1002 9.22724 17.2565C9.38351 17.4127 9.59547 17.5006 9.81643 17.5006H10.1826C10.4036 17.5006 10.6156 17.4127 10.7718 17.2565C10.9281 17.1002 11.016 16.8883 11.016 16.6673V16.5168L11.0217 16.3524C11.0471 15.9711 11.1601 15.5997 11.3521 15.2676C11.544 14.9357 11.8093 14.653 12.1268 14.4408L12.266 14.3537L12.6224 14.147L12.6249 14.1453L12.7697 14.068C13.1127 13.899 13.491 13.8109 13.8749 13.8109C14.3136 13.8109 14.7449 13.926 15.1249 14.1453L15.225 14.1991C15.2331 14.2034 15.2414 14.2075 15.2494 14.2121L14.8327 14.9339L14.8303 14.938L15.2494 14.2121C15.4406 14.3224 15.6676 14.3521 15.8809 14.2951C16.0941 14.2381 16.2764 14.0987 16.3871 13.9077L16.5653 13.5911L16.5694 13.5846L16.6068 13.5106C16.6855 13.3361 16.7022 13.1388 16.6524 12.9523C16.5953 12.7392 16.4559 12.5575 16.265 12.4469L16.1657 12.394L16.1413 12.381C15.7598 12.1608 15.4428 11.8434 15.2233 11.4614C15.0038 11.0795 14.8896 10.6462 14.8913 10.2057V9.79232C14.8901 9.35295 15.0044 8.92085 15.2233 8.53988C15.4405 8.16197 15.753 7.84756 16.1291 7.6276L16.2544 7.55273L16.265 7.54622L16.6703 8.24691L16.6825 8.26644L16.265 7.54622C16.456 7.43558 16.5953 7.2533 16.6524 7.04004C16.7092 6.82738 16.6798 6.601 16.5702 6.41016L16.3871 6.09359C16.2764 5.90257 16.0941 5.76325 15.8809 5.70622C15.6676 5.6492 15.4406 5.67894 15.2494 5.78922L14.8303 5.0625L14.8327 5.06738L15.2494 5.78922C15.2414 5.79382 15.2331 5.79791 15.225 5.80225L15.124 5.85514L15.1249 5.85596C14.7449 6.07532 14.3136 6.19037 13.8749 6.19043C13.4362 6.19043 13.0048 6.07524 12.6249 5.85596L12.6224 5.85433L13.0415 5.13411L13.162 4.92578L12.6224 5.85433L12.266 5.64681C11.8865 5.42768 11.5715 5.1131 11.3521 4.73372C11.1327 4.35416 11.0164 3.92294 11.016 3.48454V3.33398ZM11.6662 10.0007C11.6662 9.08018 10.92 8.33398 9.99953 8.33398C9.07906 8.33398 8.33287 9.08018 8.33287 10.0007C8.33287 10.9211 9.07906 11.6673 9.99953 11.6673C10.92 11.6673 11.6662 10.9211 11.6662 10.0007ZM13.3329 10.0007C13.3329 11.8416 11.8405 13.334 9.99953 13.334C8.15859 13.334 6.6662 11.8416 6.6662 10.0007C6.6662 8.1597 8.15859 6.66732 9.99953 6.66732C11.8405 6.66732 13.3329 8.1597 13.3329 10.0007ZM12.6826 3.48291L12.69 3.59196C12.7043 3.69994 12.7401 3.80466 12.7949 3.89958C12.8498 3.99445 12.9225 4.07744 13.009 4.14372L13.1017 4.20557L13.4574 4.41227C13.584 4.48533 13.7287 4.52376 13.8749 4.52376C14.021 4.52371 14.1649 4.48536 14.2915 4.41227L14.3159 4.39844L14.4405 4.33171C15.0091 4.01188 15.6804 3.92701 16.3114 4.0957C16.9512 4.26678 17.4972 4.68476 17.8291 5.25781L18.0122 5.57519L18.013 5.57601L18.1262 5.79574C18.3626 6.3191 18.4124 6.91087 18.2629 7.47054C18.0929 8.10634 17.6781 8.64844 17.1105 8.98096L17.1114 8.98177L16.9868 9.05664C16.9829 9.05903 16.9786 9.06164 16.9746 9.06396C16.8475 9.13739 16.7418 9.24345 16.6686 9.37077C16.5956 9.49802 16.5574 9.64231 16.558 9.78906V10.2122C16.5574 10.359 16.5956 10.5033 16.6686 10.6305C16.7404 10.7553 16.8435 10.8592 16.9673 10.9325L17.0747 10.9902C17.0833 10.9948 17.0923 11 17.1008 11.0049L16.6825 11.7259L16.6809 11.7292L17.1008 11.0049C17.6736 11.3368 18.0918 11.8822 18.2629 12.5218C18.4332 13.1588 18.3437 13.8369 18.0163 14.409L18.0171 14.4098L17.834 14.7354L17.0861 14.313L17.1333 14.3398L17.834 14.7354L17.8291 14.7435L17.6948 14.951C17.3593 15.4172 16.8712 15.7559 16.3114 15.9056C15.6803 16.0743 15.0092 15.988 14.4405 15.668V15.6696L14.3159 15.6029L14.2915 15.589C14.1649 15.5159 14.021 15.4776 13.8749 15.4775C13.7287 15.4775 13.5848 15.516 13.4582 15.589L13.4574 15.5882L13.1017 15.7957L12.611 14.951L12.6826 15.0755L13.1017 15.7957L13.009 15.8576C12.9226 15.9239 12.8498 16.0068 12.7949 16.1017C12.7218 16.2282 12.6828 16.3723 12.6826 16.5184V16.6673C12.6826 17.3302 12.4197 17.9661 11.951 18.4349C11.4822 18.9037 10.8457 19.1673 10.1826 19.1673H9.81643C9.15339 19.1673 8.51688 18.9037 8.04804 18.4349C7.57938 17.9661 7.31643 17.3302 7.31643 16.6673V16.5184L7.3091 16.4093C7.29477 16.3014 7.25899 16.1966 7.20412 16.1017C7.14923 16.0068 7.07652 15.9239 6.99009 15.8576L6.89732 15.7957L7.31643 15.0755L7.38804 14.951L6.89732 15.7957L6.54088 15.5882C6.41425 15.5152 6.2704 15.4775 6.12421 15.4775C5.97804 15.4776 5.83413 15.5159 5.70754 15.589C5.69953 15.5936 5.69129 15.5985 5.68313 15.6029L5.55862 15.6696L5.5578 15.668C4.98921 15.9877 4.31856 16.0743 3.68769 15.9056C3.1279 15.7559 2.6398 15.4172 2.30422 14.951L2.16994 14.7435L1.98684 14.4269L2.76321 13.9769L2.60533 14.068L1.98684 14.4269L1.98603 14.4253L1.87291 14.2056C1.63646 13.6822 1.58663 13.0904 1.73619 12.5308C1.90618 11.8951 2.32027 11.3521 2.88772 11.0195L3.02444 10.9373C3.15161 10.8639 3.25726 10.7578 3.33043 10.6305C3.40351 10.5033 3.44169 10.359 3.4411 10.2122V9.78906L3.43378 9.68083C3.41905 9.57339 3.38284 9.46918 3.32799 9.37484C3.2549 9.2493 3.15025 9.14474 3.02444 9.0721L3.44436 8.34538L3.4411 8.35026L3.02444 9.0721C3.009 9.06319 2.99369 9.05351 2.97886 9.04362L2.85435 8.96061C2.30454 8.62652 1.90273 8.09337 1.73619 7.47054C1.56526 6.83089 1.65521 6.14954 1.98603 5.57601L1.98684 5.57438L2.16994 5.25781L2.30422 5.05029C2.6398 4.58408 3.1279 4.24539 3.68769 4.0957C4.31844 3.92707 4.98928 4.01298 5.5578 4.33252L5.55862 4.33171L5.68313 4.39844L5.70754 4.41227C5.83413 4.48536 5.97804 4.52371 6.12421 4.52376C6.23396 4.52376 6.34247 4.502 6.44322 4.46029L6.54088 4.41227L6.89732 4.20557L6.99009 4.14372C7.07652 4.07744 7.14923 3.99445 7.20412 3.89958C7.27716 3.77323 7.31618 3.62965 7.31643 3.48372V3.33398C7.31643 2.67108 7.57938 2.03522 8.04804 1.56641C8.51688 1.09757 9.15339 0.833984 9.81643 0.833984H10.1826C10.8457 0.833984 11.4822 1.09757 11.951 1.56641C12.4197 2.03522 12.6826 2.67108 12.6826 3.33398V3.48291Z" fill="currentColor"/>
					</svg>',
					],
				],
			],
			'top_menu' => [
				[
					'id'    => 'menu_top_home',
					'type'  => 'default',
					'label' => 'Feeds',
					'status' => true,
					'icon'  => [
						'type' => 'emoji',
						'src'  => '<svg xmlns="http://www.w3.org/2000/svg" width="18" height="19" viewBox="0 0 18 19" fill="none">
						<path d="M7.33366 10.8332V16.6665H10.667V10.8332H7.33366ZM17.3337 15.8332C17.3337 16.4962 17.0701 17.1327 16.6012 17.6016C16.1324 18.0703 15.4966 18.3332 14.8337 18.3332H3.16699C2.50405 18.3332 1.86823 18.0703 1.39941 17.6016C0.930573 17.1327 0.666992 16.4962 0.666992 15.8332V8.33317C0.666989 7.96975 0.745982 7.61059 0.898926 7.28092C1.05197 6.95109 1.27553 6.65872 1.55322 6.42399L7.38249 1.42806L7.38656 1.4248L7.56152 1.28971C7.98128 0.994267 8.48345 0.833984 9.00033 0.833984C9.5172 0.833984 10.0194 0.994267 10.4391 1.28971L10.6141 1.4248L10.6182 1.42806L16.4474 6.42399L16.5492 6.51432C16.78 6.73208 16.9678 6.99229 17.1017 7.28092C17.2547 7.61059 17.3337 7.96975 17.3337 8.33317V15.8332ZM12.3337 16.6665H14.8337C15.0546 16.6665 15.2666 16.5793 15.4229 16.4232C15.5791 16.2669 15.667 16.0542 15.667 15.8332V8.33317C15.667 8.21203 15.6407 8.09231 15.5897 7.98242C15.5387 7.87248 15.4641 7.77502 15.3716 7.69678L15.3675 7.69271L15.9095 7.06038L15.9347 7.02946L15.3675 7.69271L9.53825 2.69678V2.69759C9.38784 2.57047 9.19726 2.50065 9.00033 2.50065C8.80339 2.50065 8.61281 2.57047 8.4624 2.69759L8.46159 2.69678L2.63314 7.69271L2.0651 7.02946L2.09115 7.06038L2.63314 7.69271L2.62907 7.69678C2.53651 7.77502 2.46198 7.87248 2.41097 7.98242C2.35999 8.09231 2.33366 8.21203 2.33366 8.33317V15.8332C2.33366 16.0542 2.42152 16.2669 2.5778 16.4232C2.73406 16.5793 2.94608 16.6665 3.16699 16.6665H5.66699V10.8332C5.66709 10.3913 5.8428 9.96726 6.15527 9.65479C6.46782 9.34234 6.89172 9.1665 7.33366 9.1665H10.667C11.1089 9.1665 11.5328 9.34234 11.8454 9.65479C12.1578 9.96726 12.3336 10.3913 12.3337 10.8332V16.6665Z" fill="currentColor"/>
						</svg>',
					],
				],
				[
					'id'    => 'menu_top_groups',
					'type'  => 'default',
					'label' => 'Groups',
					'status' => true,
					'icon'  => [
						'type' => 'emoji',
						'src'  => '<svg xmlns="http://www.w3.org/2000/svg" width="20" height="18" viewBox="0 0 20 18" fill="none">
						<path d="M12.4997 16.4993V14.8327C12.4997 14.1696 12.2361 13.5339 11.7673 13.0651C11.2984 12.5963 10.6627 12.3327 9.99967 12.3327H4.99967C4.33663 12.3327 3.70094 12.5963 3.2321 13.0651C2.76326 13.5339 2.49967 14.1696 2.49967 14.8327V16.4993C2.49967 16.9596 2.12658 17.3327 1.66634 17.3327C1.2061 17.3327 0.833008 16.9596 0.833008 16.4993V14.8327C0.833008 13.7276 1.27231 12.6681 2.05371 11.8867C2.83511 11.1053 3.89461 10.666 4.99967 10.666H9.99967C11.1047 10.666 12.1642 11.1053 12.9456 11.8867C13.727 12.6681 14.1663 13.7276 14.1663 14.8327V16.4993C14.1663 16.9596 13.7932 17.3327 13.333 17.3327C12.8728 17.3327 12.4997 16.9596 12.4997 16.4993ZM17.4997 16.4993V14.8335L17.4907 14.6268C17.4506 14.1467 17.2726 13.6867 16.9756 13.3035C16.6362 12.8658 16.161 12.5534 15.6247 12.4149C15.179 12.2998 14.9115 11.8449 15.0265 11.3993C15.1417 10.9537 15.5958 10.6861 16.0413 10.8011C16.9352 11.0319 17.7275 11.5526 18.2931 12.2822C18.8587 13.0118 19.1656 13.9088 19.1663 14.8319V16.4993C19.1663 16.9596 18.7932 17.3327 18.333 17.3327C17.8728 17.3327 17.4997 16.9596 17.4997 16.4993ZM9.99967 4.83268C9.99967 3.45197 8.88039 2.33268 7.49967 2.33268C6.11896 2.33268 4.99967 3.45197 4.99967 4.83268C4.99967 6.21339 6.11896 7.33268 7.49967 7.33268C8.88039 7.33268 9.99967 6.21339 9.99967 4.83268ZM15.0062 4.83675C15.0062 4.28168 14.8216 3.74203 14.4813 3.30355C14.1409 2.86511 13.664 2.55255 13.1263 2.41488C12.6805 2.30071 12.4116 1.84672 12.5257 1.40088C12.6399 0.955088 13.0939 0.686156 13.5397 0.800293C14.436 1.02977 15.2307 1.55138 15.798 2.28223C16.3651 3.01298 16.6728 3.91175 16.6728 4.83675C16.6728 5.76193 16.3653 6.66125 15.798 7.39209C15.2307 8.1229 14.4359 8.64373 13.5397 8.87321C13.0939 8.98737 12.6399 8.71848 12.5257 8.27262C12.4116 7.8268 12.6805 7.37278 13.1263 7.25863C13.664 7.12094 14.1409 6.80845 14.4813 6.36995C14.8217 5.93145 15.0062 5.39186 15.0062 4.83675ZM11.6663 4.83268C11.6663 7.13387 9.80086 8.99935 7.49967 8.99935C5.19849 8.99935 3.33301 7.13387 3.33301 4.83268C3.33301 2.5315 5.19849 0.666016 7.49967 0.666016C9.80086 0.666016 11.6663 2.5315 11.6663 4.83268Z" fill="currentColor"/>
						</svg>',
					],
				],
				[
					'id'    => 'menu_top_members',
					'type'  => 'default',
					'label' => 'Members',
					'status' => true,
					'icon'  => [
						'type' => 'emoji',
						'src'  => '<svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 20 20" fill="none">
						<path d="M14.9997 17.4993V15.8327C14.9997 15.1696 14.7361 14.5339 14.2673 14.0651C13.7984 13.5963 13.1627 13.3327 12.4997 13.3327H7.49967C6.83663 13.3327 6.20094 13.5963 5.7321 14.0651C5.26326 14.5339 4.99967 15.1696 4.99967 15.8327V17.4993C4.99967 17.9596 4.62658 18.3327 4.16634 18.3327C3.7061 18.3327 3.33301 17.9596 3.33301 17.4993V15.8327C3.33301 14.7276 3.77231 13.6681 4.55371 12.8867C5.33511 12.1053 6.39461 11.666 7.49967 11.666H12.4997C13.6047 11.666 14.6642 12.1053 15.4456 12.8867C16.227 13.6681 16.6663 14.7276 16.6663 15.8327V17.4993C16.6663 17.9596 16.2932 18.3327 15.833 18.3327C15.3728 18.3327 14.9997 17.9596 14.9997 17.4993ZM12.4997 5.83268C12.4997 4.45197 11.3804 3.33268 9.99967 3.33268C8.61896 3.33268 7.49967 4.45197 7.49967 5.83268C7.49967 7.21339 8.61896 8.33268 9.99967 8.33268C11.3804 8.33268 12.4997 7.21339 12.4997 5.83268ZM14.1663 5.83268C14.1663 8.13387 12.3009 9.99935 9.99967 9.99935C7.69849 9.99935 5.83301 8.13387 5.83301 5.83268C5.83301 3.5315 7.69849 1.66602 9.99967 1.66602C12.3009 1.66602 14.1663 3.5315 14.1663 5.83268Z" fill="currentColor"/>
						</svg>',
					],
				],
			],
			'profile_menu' => [
				[
					'id'    => 'menu_profile_feeds',
					'type'  => 'default',
					'label' => 'Feeds',
					'status' => true,
					'icon'  => [
						'type' => 'emoji',
						'src'  => '<svg xmlns="http://www.w3.org/2000/svg" width="18" height="19" viewBox="0 0 18 19" fill="none">
						<path d="M7.33366 10.8332V16.6665H10.667V10.8332H7.33366ZM17.3337 15.8332C17.3337 16.4962 17.0701 17.1327 16.6012 17.6016C16.1324 18.0703 15.4966 18.3332 14.8337 18.3332H3.16699C2.50405 18.3332 1.86823 18.0703 1.39941 17.6016C0.930573 17.1327 0.666992 16.4962 0.666992 15.8332V8.33317C0.666989 7.96975 0.745982 7.61059 0.898926 7.28092C1.05197 6.95109 1.27553 6.65872 1.55322 6.42399L7.38249 1.42806L7.38656 1.4248L7.56152 1.28971C7.98128 0.994267 8.48345 0.833984 9.00033 0.833984C9.5172 0.833984 10.0194 0.994267 10.4391 1.28971L10.6141 1.4248L10.6182 1.42806L16.4474 6.42399L16.5492 6.51432C16.78 6.73208 16.9678 6.99229 17.1017 7.28092C17.2547 7.61059 17.3337 7.96975 17.3337 8.33317V15.8332ZM12.3337 16.6665H14.8337C15.0546 16.6665 15.2666 16.5793 15.4229 16.4232C15.5791 16.2669 15.667 16.0542 15.667 15.8332V8.33317C15.667 8.21203 15.6407 8.09231 15.5897 7.98242C15.5387 7.87248 15.4641 7.77502 15.3716 7.69678L15.3675 7.69271L15.9095 7.06038L15.9347 7.02946L15.3675 7.69271L9.53825 2.69678V2.69759C9.38784 2.57047 9.19726 2.50065 9.00033 2.50065C8.80339 2.50065 8.61281 2.57047 8.4624 2.69759L8.46159 2.69678L2.63314 7.69271L2.0651 7.02946L2.09115 7.06038L2.63314 7.69271L2.62907 7.69678C2.53651 7.77502 2.46198 7.87248 2.41097 7.98242C2.35999 8.09231 2.33366 8.21203 2.33366 8.33317V15.8332C2.33366 16.0542 2.42152 16.2669 2.5778 16.4232C2.73406 16.5793 2.94608 16.6665 3.16699 16.6665H5.66699V10.8332C5.66709 10.3913 5.8428 9.96726 6.15527 9.65479C6.46782 9.34234 6.89172 9.1665 7.33366 9.1665H10.667C11.1089 9.1665 11.5328 9.34234 11.8454 9.65479C12.1578 9.96726 12.3336 10.3913 12.3337 10.8332V16.6665Z" fill="#2A3BEE"/>
						</svg>',
					],
				],
				[
					'id'    => 'menu_profile_groups',
					'type'  => 'default',
					'label' => 'Groups',
					'status' => true,
					'icon'  => [
						'type' => 'emoji',
						'src'  => '<svg xmlns="http://www.w3.org/2000/svg" width="20" height="18" viewBox="0 0 20 18" fill="none">
						<path d="M12.4997 16.4993V14.8327C12.4997 14.1696 12.2361 13.5339 11.7673 13.0651C11.2984 12.5963 10.6627 12.3327 9.99967 12.3327H4.99967C4.33663 12.3327 3.70094 12.5963 3.2321 13.0651C2.76326 13.5339 2.49967 14.1696 2.49967 14.8327V16.4993C2.49967 16.9596 2.12658 17.3327 1.66634 17.3327C1.2061 17.3327 0.833008 16.9596 0.833008 16.4993V14.8327C0.833008 13.7276 1.27231 12.6681 2.05371 11.8867C2.83511 11.1053 3.89461 10.666 4.99967 10.666H9.99967C11.1047 10.666 12.1642 11.1053 12.9456 11.8867C13.727 12.6681 14.1663 13.7276 14.1663 14.8327V16.4993C14.1663 16.9596 13.7932 17.3327 13.333 17.3327C12.8728 17.3327 12.4997 16.9596 12.4997 16.4993ZM17.4997 16.4993V14.8335L17.4907 14.6268C17.4506 14.1467 17.2726 13.6867 16.9756 13.3035C16.6362 12.8658 16.161 12.5534 15.6247 12.4149C15.179 12.2998 14.9115 11.8449 15.0265 11.3993C15.1417 10.9537 15.5958 10.6861 16.0413 10.8011C16.9352 11.0319 17.7275 11.5526 18.2931 12.2822C18.8587 13.0118 19.1656 13.9088 19.1663 14.8319V16.4993C19.1663 16.9596 18.7932 17.3327 18.333 17.3327C17.8728 17.3327 17.4997 16.9596 17.4997 16.4993ZM9.99967 4.83268C9.99967 3.45197 8.88039 2.33268 7.49967 2.33268C6.11896 2.33268 4.99967 3.45197 4.99967 4.83268C4.99967 6.21339 6.11896 7.33268 7.49967 7.33268C8.88039 7.33268 9.99967 6.21339 9.99967 4.83268ZM15.0062 4.83675C15.0062 4.28168 14.8216 3.74203 14.4813 3.30355C14.1409 2.86511 13.664 2.55255 13.1263 2.41488C12.6805 2.30071 12.4116 1.84672 12.5257 1.40088C12.6399 0.955088 13.0939 0.686156 13.5397 0.800293C14.436 1.02977 15.2307 1.55138 15.798 2.28223C16.3651 3.01298 16.6728 3.91175 16.6728 4.83675C16.6728 5.76193 16.3653 6.66125 15.798 7.39209C15.2307 8.1229 14.4359 8.64373 13.5397 8.87321C13.0939 8.98737 12.6399 8.71848 12.5257 8.27262C12.4116 7.8268 12.6805 7.37278 13.1263 7.25863C13.664 7.12094 14.1409 6.80845 14.4813 6.36995C14.8217 5.93145 15.0062 5.39186 15.0062 4.83675ZM11.6663 4.83268C11.6663 7.13387 9.80086 8.99935 7.49967 8.99935C5.19849 8.99935 3.33301 7.13387 3.33301 4.83268C3.33301 2.5315 5.19849 0.666016 7.49967 0.666016C9.80086 0.666016 11.6663 2.5315 11.6663 4.83268Z" fill="currentColor"/>
						</svg>',
					],
				],
			],
			'sidebar_bottom_links' => []
		];
	}
    public static function registered_menus() : array {
		return apply_filters( 'zencommunity/registered_menus', [
			'primary_menu' => 'Primary menu', 
			'top_menu' => 'Top menu', 
			'profile_menu' => 'Profile menu',
			'sidebar_bottom_links' => 'Sidebar Bottom Links',
		] );
	}
    public static function get_all() : array {
        $menus = [];
        foreach ( static::registered_menus() as $key => $label ) {
            $menus[$key] = [
                'label' => $label,
                'items' => ( new static( $key ) )->get_menu(),
            ];
        }
        return $menus;
    }

	public function get_menu() : array {
        $this->default_items_map = apply_filters( "zencommunity/menu", $this->default_items_map );
		$default_items = $this->default_items_map[ $this->menu_key ] ?? [];
        $default_items = $this->default_items_map[ $this->menu_key ] = apply_filters( "zencommunity/menu/{$this->menu_key}", $default_items );

		$all_menus    = get_option( self::BASE_OPTION_KEY, [] );
		$dynamic_menu = $all_menus[ $this->menu_key ]['items'] ?? [];

		$link_items = array_filter( $dynamic_menu, fn( $item ) => ( $item['type'] ?? '' ) === 'link' );

		// Merge default and link
		$merged = array_merge( $default_items, $link_items );

		// Get activation status from DB
		$active_status = get_option( self::ACTIVE_STATUS_OPTION, [] );

		// Normalize menu structure
		foreach ( $merged as &$item ) {
			$id               = $item['id'] ?? '';
			$item['status']   = $active_status[ $id ] ?? true;
			$item['type']     = $item['type'] ?? 'default';
			$item['label']    = $item['label'] ?? $item['title'] ?? '';
			unset( $item['title'] ); // Cleanup legacy key if exists

			$item['icon'] = $item['icon'] ?? [
				'type' => 'emoji',
				'src'  => '❓',
			];
		}

		// Reorder based on saved order
		$order_option = get_option( 'zenc_' . $this->menu_key . self::ORDER_OPTION_SUFFIX, [] );
		if ( ! empty( $order_option ) && is_array( $order_option ) ) {
			$merged = $this->reorder_menu_by_ids( $merged, $order_option );
		}

		return $merged;
	}

	private function reorder_menu_by_ids( array $menu, array $ordered_ids ): array {
		$menu_by_id = [];
		foreach ( $menu as $item ) {
			if ( ! ( $item['show'] ?? true ) ) {
				continue;
			}

			if ( 
				'link' === ( $item['type'] ?? '' ) && 
				! current_user_can( 'manage_options' ) && 
				! empty( $item['roles'] ?? '' ) && 
				is_array( $item['roles'] ) && 
				! RoleManager::user_has_global_roles(  get_current_user_id(), $item['roles'] )
			) {
				continue;
			}

			if ( 
				! current_user_can( 'manage_options' ) && 
				false === boolval( $item['status'] ?? '' )
			) {
				continue;
			}
			
			$menu_by_id[ $item['id'] ] = $item;
		}

		$new_menu = [];
		foreach ( $ordered_ids as $id ) {
			if ( isset( $menu_by_id[ $id ] ) ) {
				$new_menu[] = $menu_by_id[ $id ];
				unset( $menu_by_id[ $id ] );
			}
		}
		return array_merge( $new_menu, array_values( $menu_by_id ) );
	}

	public function save_order( array $ordered_ids ) : bool {
		return update_option( 'zenc_' . $this->menu_key . self::ORDER_OPTION_SUFFIX, $ordered_ids );
	}

	public function update_is_active( string $item_id, bool $status ) : bool {
		$active_status           = get_option( self::ACTIVE_STATUS_OPTION, [] );
		$active_status[ $item_id ] = $status;
		return update_option( self::ACTIVE_STATUS_OPTION, $active_status );
	}

	public function add_link_item( array $item ) : bool {
		if (
			empty( $item['label'] ) ||
			empty( $item['url'] )
		) {
			throw new ZencommunityException( esc_html( __( "Menu labe, url is required.", 'zencommunity' ) ), 422 );
		}

		if ( empty( $item['id'] ) ) {
			$item['id'] = uniqid( 'menu' );
		}

		if ( empty( $item['roles'] ?? '' ) ) {
			$item['roles'] = null;
		}

		$item['type'] = 'link';

		$all_menus = get_option( self::BASE_OPTION_KEY, [] );
		if ( ! isset( $all_menus[ $this->menu_key ] ) ) {
			$all_menus[ $this->menu_key ] = [ 'items' => [] ];
		}

		$existing_ids = array_column( $this->get_menu(), 'id' );
		if ( in_array( $item['id'], $existing_ids, true ) ) {
			throw new ZencommunityException( esc_html( __( "Menu ID is already exists.", 'zencommunity' ) ), 422 );
		}

		// Default icon fallback
		$item['icon'] = $item['icon'] ?? [
			'type' => 'emoji',
			'src'  => '🔗',
		];

		$all_menus[ $this->menu_key ]['items'][] = $item;
		return update_option( self::BASE_OPTION_KEY, $all_menus );
	}

	public function update_link_item( string $item_id, array $new_data ) : bool {
		$all_menus = get_option( self::BASE_OPTION_KEY, [] );
		if ( empty( $all_menus[ $this->menu_key ]['items'] ) ) {
			throw new ZencommunityException( esc_html( __( "Menu is not exists.", 'zencommunity' ) ), 422 );
		}

		foreach ( $all_menus[ $this->menu_key ]['items'] as &$item ) {
			if (
				( $item['id'] ?? '' ) === $item_id &&
				( $item['type'] ?? '' ) === 'link'
			) {
				foreach ( $new_data as $key => $value ) {
					if ( $key !== 'id' && $key !== 'type' ) {
						$item[ $key ] = $value;
					}

					if ( 'roles' === $key && empty( $new_data[ $key ] ) ) {
						$item[ $key ] = null;
					}
				}
				return update_option( self::BASE_OPTION_KEY, $all_menus );
			}
		}
		throw new ZencommunityException( esc_html( __( "Unknown error.", 'zencommunity' ) ), 422 );
	}

	public function delete_link_item( string $item_id ) : bool {
		$all_menus = get_option( self::BASE_OPTION_KEY, [] );
		if ( empty( $all_menus[ $this->menu_key ]['items'] ) ) {
			throw new ZencommunityException( esc_html( __( "Menu is not exists.", 'zencommunity' ) ), 422 );
		}

		$found = false;
		foreach ( $all_menus[ $this->menu_key ]['items'] as $index => $item ) {
			if (
				( $item['id'] ?? '' ) === $item_id &&
				( $item['type'] ?? '' ) === 'link'
			) {
				unset( $all_menus[ $this->menu_key ]['items'][ $index ] );
				$found = true;
				break;
			}
		}

		if ( $found ) {
			$all_menus[ $this->menu_key ]['items'] = array_values( $all_menus[ $this->menu_key ]['items'] );
			return update_option( self::BASE_OPTION_KEY, $all_menus );
		}
		throw new ZencommunityException( esc_html( __( "Menu id not found.", 'zencommunity' ) ), 422 );
	}
}
