<?php
namespace ZenCommunity\Api;
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}
use stdClass;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;
use ZenCommunity\RewriteRules;
use ZenCommunity\Classes\Sanitizer;
use ZenCommunity\Database\Models\Attachment;
use ZenCommunity\Addons\AcademyLms\Api\Common\Permissions;
class Settings extends Common\Base {
    use Schema\SettingSchema;
    public function init() : void {
        register_rest_route( $this->namespace, '/settings', [
            [
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => [ $this, 'save' ],
                'permission_callback' => [ Permissions::class, 'can_update_settings' ],
                'args'                => $this->schema(),
            ],
            [
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => [ $this, 'get' ],
                'permission_callback' => [ Permissions::class, 'can_update_settings' ]
            ]
        ] );
        
        register_rest_route( $this->namespace, '/theme/(?P<mode>\w+)', [
            [
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => [ $this, 'update_theme_mode' ],
                'permission_callback' => [ Permissions::class, 'any_logged_user_can' ],
            ],
        ] );
    }


    public function update_theme_mode( WP_REST_Request $request ) : WP_REST_Response {
        $theme = $request->get_param( 'mode' );
		$uid = get_current_user_id();

        if( ! $uid ) 
            return new WP_REST_Response( [
                'message' => __( 'error', 'zencommunity' )
            ], 401 );

        if( ! in_array( $theme, [ 'light', 'dark' ] ) ) 
            return new WP_REST_Response(  [
                'message' => __( 'Invalid mode', 'zencommunity' )
            ], 400 );

		update_user_meta(
			$uid,
			'zenc_theme_mode',
			$theme
		);

        return new WP_REST_Response( [
                'message' => __( 'Updated', 'zencommunity' )
            ] );
    }

    public function get() : WP_REST_Response {
        return new WP_REST_Response( $this->output( $GLOBALS['zencommunity_settings'] ?? [] ) );
    }

    public function save( WP_REST_Request $request ) : WP_REST_Response {
        // exclude all elements that is not present in schema
        $settings = $this->get_params_by_schema( $request, $this->schema() );
        $rules = apply_filters( 'zencommunity/settings/sanitization_rules', [
            'name'                              => Sanitizer::STRING,
            'path'                              => Sanitizer::STRING,
            'load_on_entire_site'               => Sanitizer::BOOL,
            'media_ids.*'                       => Sanitizer::INT,
            'community_visibility'              => Sanitizer::STRING,
            'guest_redirect_url'                => Sanitizer::STRING,
            'community_visibility_roles.*'      => Sanitizer::STRING,
            'can_see_all_members'               => Sanitizer::STRING,
            'can_see_others_groups_membership'  => Sanitizer::STRING,
            'allow_member_registration'         => Sanitizer::BOOL,
            'registered_member_status'          => Sanitizer::STRING,
			// Notification preferences (booleans)
			'notification.comment_on_post'  => Sanitizer::BOOL,
			'notification.reply_on_comment' => Sanitizer::BOOL,
			'notification.mention'          => Sanitizer::BOOL,
			// Notification preferences (booleans)
			'feed.is_title_required'  => Sanitizer::BOOL,
			'feed.default_sort_by'    => Sanitizer::BOOL,
            'post_char_limit'         => Sanitizer::INT,
            'comment_char_limit'      => Sanitizer::INT,
            'message_char_limit'      => Sanitizer::INT,
            'notification_position'   => Sanitizer::STRING,
            'theme_mode'   => Sanitizer::STRING,

            'theme.light.primary_color'         => Sanitizer::STRING,   
            'theme.light.secondary_color'       => Sanitizer::STRING,
            'theme.light.text_color'            => Sanitizer::STRING,
            'theme.light.content_body_color'    => Sanitizer::STRING,
            'theme.light.content_background_color' => Sanitizer::STRING,
            'theme.light.border_color'          => Sanitizer::STRING,
            
            'theme.dark.primary_color'          => Sanitizer::STRING,
            'theme.dark.secondary_color'        => Sanitizer::STRING,
            'theme.dark.text_color'             => Sanitizer::STRING,
            'theme.dark.content_body_color'     => Sanitizer::STRING,
            'theme.dark.content_background_color' => Sanitizer::STRING,
            'theme.dark.border_color'           => Sanitizer::STRING,
        ] );

        // sanitize data
        $settings = Sanitizer::sanitize( $settings, $rules );
        $settings_as_array = json_decode( json_encode( $GLOBALS['zencommunity_settings'] ), true );
        $settings = wp_parse_args( $settings, $settings_as_array ?? [] );
        $prev_path = $GLOBALS['zencommunity_settings']->path ?? '';
        // apply settings data filter before update
        $settings = apply_filters( 'zencommunity/settings/before_update/data', $settings );
        // apply media handler
        $settings = Attachment::attach_media( 
            0, 'settings', $settings ?? [], [ $this, 'handle_media'] 
        );
        // wp_send_json($settings);

        // fire action just before update
        do_action( 'zencommunity/settings/before_update', $settings, $request );
        // update settings
        update_option( ZENCOMMUNITY_SETTINGS_NAME, json_encode( $settings ) );
        // update global settings with new data
		$GLOBALS['zencommunity_settings'] = json_decode( get_option( ZENCOMMUNITY_SETTINGS_NAME, '{}' ) );
        // fire action  after update
        do_action( 'zencommunity/settings/after_update', $GLOBALS['zencommunity_settings'], $request );

        $path = $GLOBALS['zencommunity_settings']->path ?? '';
        // compare community path before and after update: if it is changed then flush rewrite rules
        if ( $prev_path !== $path ) {
            // update new rewrite rule && flush
            RewriteRules::add_rewrite_rules();
            flush_rewrite_rules();
        }
        
        return new WP_REST_Response( array_merge( [
            'success' => true,
            'message' => __( 'Settings is saved.', 'zencommunity' )
        ], $this->output( $GLOBALS['zencommunity_settings'] ?? [] ) ) );
    }

	public static function handle_media( array $data, array $media, int $id ) : array {

		if ( ! empty( $site_logo = $media['site_logo'][0] ?? [] ) ) {
            $data['attached']['site_logo'] = $site_logo['id'];
			$data['site_logo'] = $site_logo['path'] ?? null;
		}

		// apply filters
		$data = apply_filters( 
			'zencommunity/settings/media_handler', 
			$data, $media, $id
		);
		return $data;
	}
    
    protected function output( $settings ) : array {
        global $wp_roles;
        $roles = $wp_roles->roles;
        $mapped_roles = [];
        foreach ($roles as $role_slug => $role_info) {
            $mapped_roles[$role_slug] = $role_info['name'];
        }

        $settings->load_on_entire_site = (bool) $settings->load_on_entire_site;
        $settings->allow_member_registration = (bool) $settings->allow_member_registration;

        if ( ! isset( $settings->theme ) )
            $settings->theme = new stdClass;

        if ( ! isset( $settings->theme->light ) )
            $settings->theme->light = new stdClass;

        if ( ! isset( $settings->theme->dark ) )
            $settings->theme->dark = new stdClass;

        $settings->theme->light = wp_parse_args(
            $settings->theme->light ?? [],
            zencommunity_default_theme_settings()['theme']['light']
        ); 

        $settings->theme->dark = wp_parse_args(
            $settings->theme->dark ?? [],
            zencommunity_default_theme_settings()['theme']['dark']
        ); 

        return apply_filters( 
			'zencommunity/settings/output', 
            [
                'settings' => $settings,
                'roles' => $mapped_roles,
            ]
        );
    }
}