<?php
namespace ZenCommunity\Api\Schema;
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}
trait SettingSchema  {
    public function schema() : array {
        return apply_filters( 'zencommunity/settings/schema', [
            'name' => [
                'required' => false,
                'type' => 'string',
                'description' => __( 'Community Name', 'zencommunity' ),
                'sanitize_callback' => 'sanitize_text_field',
            ],
            'load_on_entire_site' => [
                'type'        => 'boolean',
                'description' => __( 'Load the Community app across the entire frontend, overriding the active theme.', 'zencommunity' ),
            ],
            'path' => [
                'required' => false,
                'type' => 'string',
                'description' => __( 'Community url path', 'zencommunity' ),
                'sanitize_callback' => [ $this, 'sanitize_path' ],
            ],
            'media_ids' => [
                'required' => false,
                'type'     => 'array',
                'items'    => [
                    'type' => [ 'integer', 'string' ],
                ],
                'description' => __( 'An array of media IDs, media type must be site_logo', 'zencommunity' ),
                'validate_callback' => fn( $ids, $request, $key) : bool => is_array( $ids ) && count( array_filter( $ids, 'is_numeric' ) ) === count( $ids ),
                'sanitize_callback' => fn( $ids, $request, $key ) : array => array_map( 'absint', $ids ),
            ],
            'community_visibility' => [
                'required' => false,
                'type' => 'string',
                'enum'        => [ 'guest', 'logged', 'roles' ],
                'description' => __( 'Who can see & join this community.', 'zencommunity' ),
                'sanitize_callback' => 'sanitize_text_field',
            ],
            'guest_redirect_url' => [
                'required' => false,
                'type' => 'string',
                'description' => __( 'Redirect url for guset user. It will be used when community visibility is not set to logged user.', 'zencommunity' ),
                'sanitize_callback' => 'sanitize_text_field',
            ],
            'community_visibility_roles' => [
                'description' => __( 'List of user roles.', 'zencommunity' ),
                'type'        => 'array',
                'items'       => [
                    'type' => 'string',
                ],
                'required'    => false,
                'sanitize_callback' => [ $this, 'sanitize_roles' ],
            ],
            'can_see_all_members' => [
                'required' => false,
                'type' => 'string',
                'enum'        => [ 'guest', 'logged', 'admin-mod' ],
                'description' => __( 'Who can see all community members page?.', 'zencommunity' ),
                'sanitize_callback' => 'sanitize_text_field',
            ],
            'can_see_others_groups_membership' => [
                'required' => false,
                'type' => 'string',
                'enum'        => [ 'guest', 'logged', 'admin-mod' ],
                'description' => __( 'Who can see membership groups of others profile page.', 'zencommunity' ),
                'sanitize_callback' => 'sanitize_text_field',
            ],
            'allow_member_registration' => [
                'type'        => 'boolean',
                'description' => __( 'Allow member registrationt', 'zencommunity' ),
            ],
            'registered_member_status' => [
                'type'        => 'string',
                'enum'        => [ 'active', 'pending' ],
                'description' => __( 'Indicates the onboarding status of a newly registered member (e.g: active, pending ).', 'zencommunity' ),
                'sanitize_callback' => 'sanitize_text_field',
            ],

            'feed' => [
                'description' => __( 'Feed settings', 'zencommunity' ),
                'type'        => 'object',
                'required'    => false,
                'properties'  => [
                    'is_title_required' => [
                        'type'        => 'boolean',
                        'description' => __( 'Whether the title is required.', 'zencommunity' ),
                    ],
                    'default_sort_by' => [
                        'type'        => 'string',
                        'description' => __( 'Default sorting option.', 'zencommunity' ),
                        'enum'        => [ 'created_at', 'updated_at', 'title', 'like_count', 'comment_count', 'id' ],
                    ],
                ],
            ],
            'title_char_limit' => [
                'type'        => 'integer',
                'description' => __( 'Character limit of a  title.', 'zencommunity' ),
            ],
            'description_char_limit' => [
                'type'        => 'integer',
                'description' => __( 'Character limit of a description.', 'zencommunity' ),
            ],
            'comment_char_limit' => [
                'type'        => 'integer',
                'description' => __( 'Character limit of a comment.', 'zencommunity' ),
            ],
            'message_char_limit' => [
                'type'        => 'integer',
                'description' => __( 'Character limit of a message.', 'zencommunity' ),
            ],
            'notification_position' => [
                'description' => __( 'Position where notifications should appear on screen.', 'zencommunity' ),
                'type'        => 'string',
                'enum'        => [
                    'top-left',
                    'top-center',
                    'top-right',
                    'bottom-left',
                    'bottom-center',
                    'bottom-right',
                ],
            ],
            'notification' => [
                'type'        => 'object',
                'description' => __( 'User notification preferences.', 'zencommunity' ),
                'properties'  => [
                    'comment_on_post' => [
                        'type'              => 'boolean',
                        'description'       => __( 'Notify when someone comments on post.', 'zencommunity' ),
                        'sanitize_callback' => 'rest_sanitize_boolean',
                    ],
                    'reply_on_comment' => [
                        'type'              => 'boolean',
                        'description'       => __( 'Notify when someone replies to comment.', 'zencommunity' ),
                        'sanitize_callback' => 'rest_sanitize_boolean',
                    ],
                    'mention' => [
                        'type'              => 'boolean',
                        'description'       => __( 'Notify when user is mentioned.', 'zencommunity' ),
                        'sanitize_callback' => 'rest_sanitize_boolean',
                    ],
                    'email_on_comment_reply' => [
                        'type'        => 'boolean',
                        'description' => __( 'Send email when comment is replied.', 'zencommunity' ),
                    ],
                ],
                'additionalProperties' => false,
            ],
            'theme' => [
                'description' => __( 'Theme settings', 'zencommunity' ),
                'type'        => 'object',
                'required'    => false,
                'properties'  => [
                    'light' => [
                        'type'       => 'object',
                        'description' => __( 'Light theme settings', 'zencommunity' ),
                        'properties' => [
                            'primary_color' => [
                                'type'        => 'string',
                                'description' => __( 'Primary color of the light theme.', 'zencommunity' ),
                            ],
                            'secondary_color' => [
                                'type'        => 'string',
                                'description' => __( 'Secondary color of the light theme.', 'zencommunity' ),
                            ],
                            'text_color' => [
                                'type'        => 'string',
                                'description' => __( 'Text color of the light theme.', 'zencommunity' ),
                            ],
                            'content_body_color' => [
                                'type'        => 'string',
                                'description' => __( 'Content body color of the light theme.', 'zencommunity' ),
                            ],
                            'content_background_color' => [
                                'type'        => 'string',
                                'description' => __( 'Content background color of the light theme.', 'zencommunity' ),
                            ],
                            'border_color' => [
                                'type'        => 'string',
                                'description' => __( 'Border color of the light theme.', 'zencommunity' ),
                            ],
                        ],
                    ],
                    'dark' => [
                        'type'       => 'object',
                        'description' => __( 'Dark theme settings', 'zencommunity' ),
                        'properties' => [
                            'primary_color' => [
                                'type'        => 'string',
                                'description' => __( 'Primary color of the dark theme.', 'zencommunity' ),
                            ],
                            'secondary_color' => [
                                'type'        => 'string',
                                'description' => __( 'Secondary color of the dark theme.', 'zencommunity' ),
                            ],
                            'text_color' => [
                                'type'        => 'string',
                                'description' => __( 'Text color of the dark theme.', 'zencommunity' ),
                                'example'     => '#f7f7f7',
                            ],
                            'content_body_color' => [
                                'type'        => 'string',
                                'description' => __( 'Content body color of the dark theme.', 'zencommunity' ),
                            ],
                            'content_background_color' => [
                                'type'        => 'string',
                                'description' => __( 'Content background color of the dark theme.', 'zencommunity' ),
                                'example'     => '#222',
                            ],
                            'border_color' => [
                                'type'        => 'string',
                                'description' => __( 'Border color of the dark theme.', 'zencommunity' ),
                            ],
                        ],
                    ],
                ],
            ],

        ] );
    }

    public function sanitize_roles( $value, $request, $param ) {
        if ( ! is_array( $value ) ) {
            return [];
        }
        global $wp_roles;
        $valid_roles = array_keys( $wp_roles->roles ?? [] );
        return array_values( 
            array_filter(
                array_map( 'sanitize_text_field', $value ), 
                fn ( $role ) => in_array( $role, $valid_roles, true ) 
            )
        );
    }

    public function sanitize_path( $value ) {
        $value = wp_parse_url( $value, PHP_URL_PATH );
        $value = sanitize_text_field( $value );
        return '/' . ltrim( $value, '/' );
    }
}