<?php
namespace ZenCommunity\Api;

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;
use ZenCommunity\Exceptions\ZencommunityException;
use ZenCommunity\Classes\MenuManager;
use ZenCommunity\Api\Common\Base;
use ZenCommunity\Addons\AcademyLms\Api\Common\Permissions;
class Menu extends Base {
    use Schema\CommonSchema, 
        Schema\MenuSchema;

    public function init() : void {
        register_rest_route( $this->namespace, '/menus', [
            [
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => [ $this, 'all' ],
                'permission_callback' => [ Permissions::class, 'even_guest_user_can' ],
            ]
        ] );

        register_rest_route( $this->namespace, '/menu/(?P<key>[a-zA-Z0-9_-]+)', [
            [
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => [ $this, 'get_menu' ],
                'permission_callback' => [ Permissions::class, 'admin_can' ],
            ]
        ] );

        register_rest_route( $this->namespace, '/menu/(?P<key>[a-zA-Z0-9_-]+)/items', [
            [
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => [ $this, 'add_item' ],
                'permission_callback' => [ Permissions::class, 'admin_can' ],
                'args'                => $this->item_schema()
            ]
        ] );

        register_rest_route( $this->namespace, '/menu/(?P<key>[a-zA-Z0-9_-]+)/items/(?P<id>[a-zA-Z0-9_\-\.]+)', [
            [
                'methods'             => WP_REST_Server::EDITABLE,
                'callback'            => [ $this, 'update_item' ],
                'permission_callback' => [ Permissions::class, 'admin_can' ],
                'args'                => $this->item_schema()
            ],
            [
                'methods'             => WP_REST_Server::DELETABLE,
                'callback'            => [ $this, 'delete_item' ],
                'permission_callback' => [ Permissions::class, 'admin_can' ]
            ]
        ] );


        register_rest_route( $this->namespace, '/menu/(?P<key>[a-zA-Z0-9_-]+)/items/(?P<id>[a-zA-Z0-9_\-\.]+)/(?P<status>(active|deactive))', [
            [
                'methods'             => WP_REST_Server::EDITABLE,
                'callback'            => [ $this, 'update_item_status' ],
                'permission_callback' => [ Permissions::class, 'admin_can' ],
            ],
        ] );

        register_rest_route( $this->namespace, '/menu/(?P<key>[a-zA-Z0-9_-]+)/reorder', [
            [
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => [ $this, 'reorder' ],
                'permission_callback' => [ Permissions::class, 'admin_can' ],
                'args'                => $this->reorder_schema()
            ]
        ] );
    }

    public function all( WP_REST_Request $request ) : WP_REST_Response {
        try {
            return new WP_REST_Response( MenuManager::get_all(), 200 );
        } catch ( ZencommunityException $e ) {
            return $e->as_rest_response();
        }
    }

    public function get_menu( WP_REST_Request $request ) : WP_REST_Response {
        try {
            $key = $request['key'];
            $manager = new MenuManager( $key );
            return new WP_REST_Response( [
                'label' => MenuManager::registered_menus()[$key] ?? '',
                'items' => $manager->get_menu()
            ], 200 );
        } catch ( ZencommunityException $e ) {
            return $e->as_rest_response();
        }
    }

    public function add_item( WP_REST_Request $request ) : WP_REST_Response {
        try {
            $params = $this->get_params_by_schema( $request, $this->item_schema() );
            $manager = new MenuManager( $request['key'] );
            $manager->add_link_item( $params );
            return new WP_REST_Response( [ 'success' => true ] );
        } catch ( ZencommunityException $e ) {
            return $e->as_rest_response();
        }
    }

    public function update_item_status( WP_REST_Request $request ) : WP_REST_Response {
        try {
            $params = $this->get_params_by_schema( $request, $this->item_schema() );
            $manager = new MenuManager( $request['key'] );
            $manager->update_is_active( $request['id'], 'active' === $request['status'] ? true : false );
            return new WP_REST_Response( [ 'success' => true ] );
        } catch ( ZencommunityException $e ) {
            return $e->as_rest_response();
        }
    }

    public function update_item( WP_REST_Request $request ) : WP_REST_Response {
        try {
            $params = $this->get_params_by_schema( $request, $this->item_schema() );
            $manager = new MenuManager( $request['key'] );
            $manager->update_link_item( $request['id'], $params );
            return new WP_REST_Response( [ 'success' => true ] );
        } catch ( ZencommunityException $e ) {
            return $e->as_rest_response();
        }
    }

    public function delete_item( WP_REST_Request $request ) : WP_REST_Response {
        try {
            $manager = new MenuManager( $request['key'] );
            $manager->delete_link_item( $request['id'] );
            return new WP_REST_Response( [ 'success' => true ] );
        } catch ( ZencommunityException $e ) {
            return $e->as_rest_response();
        }
    }

    public function reorder( WP_REST_Request $request ) : WP_REST_Response {
        try {
            $params = $this->get_params_by_schema( $request, $this->reorder_schema() );
            $manager = new MenuManager( $request['key'] );
            $manager->save_order( $params['ordered_ids'] );
            return new WP_REST_Response( [ 'success' => true ] );
        } catch ( ZencommunityException $e ) {
            return $e->as_rest_response();
        }
    }
}
