<?php
namespace ZenCommunity\Api;
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}
use ZenCommunity\Database\Utils\QueryBuilder;
use ZenCommunity\Exceptions\ZencommunityException;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;
use ZenCommunity\Database\Models\Feed as FeedModel;
use ZenCommunity\Addons\AcademyLms\Api\Common\Permissions;
class Feed extends Common\Base {
    use Schema\FeedSchema;
    public function init() : void {
        register_rest_route( $this->namespace, '/feeds', [
            [
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => [ $this, 'index' ],
                'permission_callback' => [ Permissions::class, 'even_guest_user_can' ],
                'args'                => $this->index_schema(),
            ],
        ] );

        register_rest_route( $this->namespace, '/group/(?P<group_id>\d+)/feeds', [
            [
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => [ $this, 'group_feed_index' ],
                'permission_callback' => [ Permissions::class, 'even_guest_user_can' ],
                'args'                => $this->index_schema(),
            ],
            [
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => [ $this, 'create' ],
                'permission_callback' => [ Permissions::class, 'any_logged_user_can' ],
                'args'                => $this->schema()
            ],
        ] );

        register_rest_route( $this->namespace, '/member/(?P<member_id>\d+)/feeds', [
            [
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => [ $this, 'member_feed_index' ],
                'permission_callback' => [ Permissions::class, 'even_guest_user_can' ],
                'args'                => $this->index_schema(),
            ],
        ] );

        register_rest_route( $this->namespace, '/feed/(?P<feed_id>\d+)', [
            [
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => [ $this, 'get' ],
                'permission_callback' => [ Permissions::class, 'even_guest_user_can' ],
            ],
            [
                'methods'             => WP_REST_Server::EDITABLE,
                'callback'            => [ $this, 'update' ],
                'permission_callback' => [ Permissions::class, 'any_logged_user_can' ],
                'args'                => $this->schema()
            ],
            [
                'methods'             => WP_REST_Server::DELETABLE,
                'callback'            => [ $this, 'delete' ],
                'permission_callback' => [ Permissions::class, 'any_logged_user_can' ],
            ]
        ] );

        register_rest_route( $this->namespace, '/feed/(?P<feed_id>\d+)/comments', [
            [   
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => [ $this, 'comments' ],
                'permission_callback' => [ Permissions::class, 'any_logged_user_can' ],
                'args'                => $this->index_schema()
            ],
            [   
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => [ $this, 'new_comment' ],
                'permission_callback' => [ Permissions::class, 'any_logged_user_can' ],
                'args'                => $this->comment_schema()
            ],
        ] );

        register_rest_route( $this->namespace, '/feed/(?P<feed_id>\d+)/comment/(?P<comment_id>\d+)', [
            [   
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => [ $this, 'comment_replys' ],
                'permission_callback' => [ Permissions::class, 'any_logged_user_can' ],
            ],
            [   
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => [ $this, 'new_reply' ],
                'permission_callback' => [ Permissions::class, 'any_logged_user_can' ],
                'args'                => $this->comment_schema()
            ],
            [
                'methods'             => WP_REST_Server::EDITABLE,
                'callback'            => [ $this, 'update_comment' ],
                'permission_callback' => [ Permissions::class, 'any_logged_user_can' ],
                'args'                => $this->comment_schema()
            ],
            [
                'methods'             => WP_REST_Server::DELETABLE,
                'callback'            => [ $this, 'delete_comment' ],
                'permission_callback' => [ Permissions::class, 'any_logged_user_can' ],
            ]
        ] );
        
        
        register_rest_route( $this->namespace, '/feed/(?P<feed_id>\d+)/react/(?P<type>\w+)', [
            [   
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => [ $this, 'reacted_members' ],
                'permission_callback' => [ Permissions::class, 'even_guest_user_can' ],
            ],
            [   
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => [ $this, 'react_on_feed' ],
                'permission_callback' => [ Permissions::class, 'any_logged_user_can' ],
            ],
        ] );

        register_rest_route( $this->namespace, '/feed/(?P<feed_id>\d+)/comment/(?P<comment_id>\d+)/react/(?P<type>\w+)', [
            [   
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => [ $this, 'reacted_members_on_comment' ],
                'permission_callback' => [ Permissions::class, 'even_guest_user_can' ],
            ],
            [   
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => [ $this, 'react_on_comment' ],
                'permission_callback' => [ Permissions::class, 'any_logged_user_can' ],
            ],
        ] );
    }

    public function index( WP_REST_Request $request ) : WP_REST_Response {
        try {
            $data = $this->get_params_by_schema( $request, $this->index_schema() );
            // show only by_user_memberships feeds if user is not admin
            if( ! current_user_can( 'manage_options' ) ) {
                $data['status'] = 'published';
            }
            return new WP_REST_Response(  FeedModel::index( $data ?? [], get_current_user_id() ) );
        }
        catch( ZencommunityException $e ) {
            return $e->as_rest_response();
        }
    }

    public function member_feed_index( WP_REST_Request $request ) : WP_REST_Response {
        try {
            $data = $this->get_params_by_schema( $request, $this->index_schema(), [ 'member_id' ] );
            $data['member_id'] = absint( $data['member_id'] ?? 0 );
            // show only by_user_memberships feeds if user is not admin
            if( ! current_user_can( 'manage_options' ) ) {
                $data['status'] = 'published';
            }
            return new WP_REST_Response(  FeedModel::index( $data ?? [], get_current_user_id() ) );
        }
        catch( ZencommunityException $e ) {
            return $e->as_rest_response();
        }
    }

    public function group_feed_index( WP_REST_Request $request ) : WP_REST_Response {
        try {
            $data = $this->get_params_by_schema( $request, $this->index_schema(), [ 'group_id' ] );

            $data['group_id'] = absint( $data['group_id'] ?? 0 );

            // show only by_user_memberships feeds if user is not admin
            if( ! current_user_can( 'manage_options' ) ) {
                $data['status'] = 'published';
            }
            return new WP_REST_Response(  FeedModel::index( $data ?? [], get_current_user_id() ) );
        }
        catch( ZencommunityException $e ) {
            return $e->as_rest_response();
        }
    }

    public function create( WP_REST_Request $request ) : WP_REST_Response {
        $data = $this->get_params_by_schema( $request, $this->schema(), [ 'group_id' ] );

        $group_id = absint( $data['group_id'] ?? 0 );
        $user_id = get_current_user_id();

        try {
            if ( $id = FeedModel::create( $group_id, $data, $user_id ) ) {
                //return created record
                return new WP_REST_Response( FeedModel::by_id( $id ) );
            }
            
            return new WP_REST_Response( [
                'success' => $id,
                'message' => __( 'An unknwon error is occured, please try again later.', 'zencommunity' )
            ], 422 );
        }
        catch( ZencommunityException $e ) {
           return $e->as_rest_response();
        }
    }

    public function get( WP_REST_Request $request ) : WP_REST_Response {
        try {
            $params = $request->get_params();
            $user_id = get_current_user_id();
            $cb = null;
            if( ! current_user_can( 'manage_options' ) ) {
                $cb = fn( QueryBuilder $qb ) : QueryBuilder => $qb->join( 'zenc_group_members', 'spm', 'spm.group_id = fd.group_id' )->where( 'spm.user_id', '=', $user_id )->where( 'spm.status', '=', 'active' );
            }

            $data = FeedModel::by_id( absint( $params['feed_id'] ), null, $cb, $user_id );
            if (  empty( $data ) ) {
                return new WP_REST_Response( [
                    'success' => false,
                    'message' => __( 'Unauthorized.', 'zencommunity' )
                ], 401 );
            }
            // if feed status is draft, then check current user is creator or not
            if ( 'draft' === $data['status'] && absint( $data['user_id'] ) !== $user_id ) {
                return new WP_REST_Response( [
                    'success' => false,
                    'message' => __( 'Unauthorized.', 'zencommunity' )
                ], 401 );
            }

            return new WP_REST_Response( $data );
        }
        catch( ZencommunityException $e ) {
            return $e->as_rest_response();
        }
    }

    public function update( WP_REST_Request $request ) : WP_REST_Response {
        try {
            $data = $this->get_params_by_schema( $request, $this->schema(), [ 'feed_id' ]  );

            $feed_id = absint( $data['feed_id'] ?? 0 );
            $user_id = get_current_user_id();

            if ( FeedModel::update( $feed_id, $data, $user_id ) ) {
                return new WP_REST_Response( FeedModel::by_id( $feed_id ) );
            }
            return new WP_REST_Response( [
                'success' => false,
                'message' => __( 'Record not updated.', 'zencommunity' )
            ], 400 );
        }
        catch( ZencommunityException $e ) {
            return $e->as_rest_response();
        }
    }

    public function delete( WP_REST_Request $request ) : WP_REST_Response {
        try { 
            $params = $request->get_params();
            $feed_id = absint( $params['feed_id'] ?? 0 );
            $user_id = get_current_user_id();
            if ( FeedModel::delete( $feed_id, $user_id ) ) {
                return new WP_REST_Response( [
                    'object_id' => $feed_id,
                    'success' => true,
                    'message' =>  __( 'Record is deleted.', 'zencommunity' ) 
                ] );
            }
            return new WP_REST_Response( [
                'object_id' => $feed_id,
                'success' => true,
                'message' => __( 'Record is not deleted.', 'zencommunity' )
            ] );
        }
        catch( ZencommunityException $e ) {
            return $e->as_rest_response();
        }
    }

    
    public function comments( WP_REST_Request $request ) : WP_REST_Response {
        try { 
            $params = $request->get_params();
            $user_id = get_current_user_id();
            $feed_id = absint( $params['feed_id'] ?? 0 );
            $order_by = sanitize_text_field( $params['order_by'] ?? 'created_at' );
            $order = sanitize_text_field( $params['order'] ?? 'ASC' );

            $data = [];
            $data['add_media_ids'] = $params['add_media_ids'] ?? [];
            $data['content'] = wp_kses_post( $params['content'] ?? '' );
            return new WP_REST_Response( FeedModel::get_comments(
                $feed_id,
                $user_id, 
                null, 
                $order_by,
                $order 
            ) );
        }
        catch( ZencommunityException $e ) {
            return $e->as_rest_response();
        }
    }
    
    public function comment_replys( WP_REST_Request $request ) : WP_REST_Response {
        try { 
            $params = $request->get_params();
            $user_id = get_current_user_id();
            $feed_id = absint( $params['feed_id'] ?? 0 );
            $comment_id = absint( $params['comment_id'] ?? 0 );
            $order_by = sanitize_text_field( $params['order_by'] ?? 'created_at' );
            $order = sanitize_text_field( $params['order'] ?? 'ASC' );

            $data = [];
            $data['add_media_ids'] = $params['add_media_ids'] ?? [];
            $data['content'] = wp_kses_post( $params['content'] ?? '' );
            return new WP_REST_Response( FeedModel::get_comments(
                $feed_id,
                $user_id, 
                $comment_id, 
                $order_by,
                $order 
            ) );
        }
        catch( ZencommunityException $e ) {
            return $e->as_rest_response();
        }
    }

    public function new_comment( WP_REST_Request $request ) : WP_REST_Response {
        try { 
            $params = $request->get_params();
            $feed_id = absint( $params['feed_id'] ?? 0 );
            $user_id = get_current_user_id();

            $data = [];
            $data['add_media_ids'] = $params['add_media_ids'] ?? [];
            $data['content'] = wp_kses_post( $params['content'] ?? '' );

            if ( $data = FeedModel::comment( $feed_id, $user_id, $data ) ) {
                return new WP_REST_Response( $data );
            }
        }
        catch( ZencommunityException $e ) {
            return $e->as_rest_response();
        }
    }

    public function new_reply( WP_REST_Request $request ) : WP_REST_Response {
        try { 
            $params = $request->get_params();
            $feed_id = absint( $params['feed_id'] ?? 0 );
            $comment_id = absint( $params['comment_id'] ?? 0 );
            $user_id = get_current_user_id();

            $data = [];
            $data['add_media_ids'] = $params['add_media_ids'] ?? [];
            $data['content'] = wp_kses_post( $params['content'] ?? '' );

            if ( $data = FeedModel::comment( $feed_id, $user_id, $data, $comment_id ) ) {
                return new WP_REST_Response( $data );
            }
        }
        catch( ZencommunityException $e ) {
            return $e->as_rest_response();
        }
    }

    public function update_comment( WP_REST_Request $request ) : WP_REST_Response {
        try { 
            $params = $request->get_params();
            $feed_id = absint( $params['feed_id'] ?? 0 );
            $comment_id = absint( $params['comment_id'] ?? 0 );
            $user_id = get_current_user_id();

            $data = [];
            $data['add_media_ids'] = $params['add_media_ids'] ?? [];
            $data['remove_media_ids'] = $params['remove_media_ids'] ?? [];
            $data['content'] = wp_kses_post( $params['content'] ?? '' );

            if ( $data = FeedModel::edit_comment( $feed_id, $user_id, $comment_id, $data ) ) {
                return new WP_REST_Response( $data );
            }
        }
        catch( ZencommunityException $e ) {
            return $e->as_rest_response();
        }
    }

    public function delete_comment( WP_REST_Request $request ) : WP_REST_Response {
        try { 
            $params = $request->get_params();
            $feed_id = absint( $params['feed_id'] ?? 0 );
            $comment_id = absint( $params['comment_id'] ?? 0 );
            $user_id = get_current_user_id();

            if ( $data = FeedModel::delete_comment( $feed_id, $user_id, $comment_id ) ) {
                return new WP_REST_Response( $data );
            }
        }
        catch( ZencommunityException $e ) {
            return $e->as_rest_response();
        }
    }

    public function react_on_feed( WP_REST_Request $request ) : WP_REST_Response {
        try { 
            $params = $request->get_params();
            $feed_id = absint( $params['feed_id'] ?? 0 );
            $type = sanitize_text_field( $params['type'] ?? '' );
            $user_id = get_current_user_id();

            if ( $data = FeedModel::react( $feed_id, $type, $user_id ) ) {
                return new WP_REST_Response( $data );
            }
        }
        catch( ZencommunityException $e ) {
            return $e->as_rest_response();
        }
    }

    public function react_on_comment( WP_REST_Request $request ) : WP_REST_Response {
        try { 
            $params = $request->get_params();
            $feed_id = absint( $params['feed_id'] ?? 0 );
            $type = sanitize_text_field( $params['type'] ?? '' );
            $comment_id = absint( $params['comment_id'] ?? 0 );
            $user_id = get_current_user_id();

            if ( $data = FeedModel::react( $feed_id, $type, $user_id, $comment_id ) ) {
                return new WP_REST_Response( $data );
            }
        }
        catch( ZencommunityException $e ) {
            return $e->as_rest_response();
        }
    }

    public function reacted_members( WP_REST_Request $request ) : WP_REST_Response {
        try { 
            $params = $request->get_params();
            $feed_id = absint( $params['feed_id'] ?? 0 );
            $type = sanitize_text_field( $params['type'] ?? 'like' );
            $user_id = get_current_user_id();
            $data = FeedModel::get_reacted_members( 
                $feed_id, $user_id, $type 
            );
            return new WP_REST_Response( $data );
        }
        catch( ZencommunityException $e ) {
            return $e->as_rest_response();
        }
    }

    public function reacted_members_on_comment( WP_REST_Request $request ) : WP_REST_Response {
        try { 
            $params = $request->get_params();
            $feed_id = absint( $params['feed_id'] ?? 0 );
            $comment_id = absint( $params['comment_id'] ?? 0 );
            $type = sanitize_text_field( $params['type'] ?? 'like' );
            $user_id = get_current_user_id();
            $data = FeedModel::get_reacted_members( 
                $feed_id, $user_id, $type, $comment_id 
            );
            return new WP_REST_Response( $data );
        }
        catch( ZencommunityException $e ) {
            return $e->as_rest_response();
        }
    }
}