/* global ZDZE, fetch */
(function () {
  'use strict';

  /**
   * Create a <script> tag and resolve when loaded.
   * @param {string} src
   * @returns {Promise<void>}
   */
  function loadScript(src) {
    return new Promise(function (resolve, reject) {
      var existing = document.querySelector('script[data-zdze-src="' + src.replace(/"/g, '') + '"]');
      if (existing) {
        resolve();
        return;
      }
      var s = document.createElement('script');
      s.src = src;
      s.async = true;
      s.defer = true;
      s.setAttribute('data-zdze-src', src);
      s.onload = function () { resolve(); };
      s.onerror = function () { reject(new Error('No se pudo cargar: ' + src)); };
      document.head.appendChild(s);
    });
  }

  /**
   * Add CSS <link> tag once.
   * @param {string} href
   */
  function addCss(href) {
    if (document.querySelector('link[data-zdze-href="' + href.replace(/"/g, '') + '"]')) {
      return;
    }
    var l = document.createElement('link');
    l.rel = 'stylesheet';
    l.type = 'text/css';
    l.href = href;
    l.setAttribute('data-zdze-href', href);
    document.head.appendChild(l);
  }

  /**
   * Load Zoom Meeting SDK (Web) assets from source.zoom.us using configured version.
   * Uses the common CDN pattern shown in Zoom community examples:
   * - bootstrap.css, react-select.css
   * - vendor scripts (react, react-dom, redux, redux-thunk, lodash)
   * - zoom-meeting-embedded-{version}.min.js (and optionally zoom-meeting-{version}.min.js)
   *
   * @param {string} version
   * @returns {Promise<void>}
   */
  function ensureZoomAssets(version) {
    if (window.__zdzeZoomAssetsPromise) {
      return window.__zdzeZoomAssetsPromise;
    }

    window.__zdzeZoomAssetsPromise = (async function () {
      var base = 'https://source.zoom.us/' + encodeURIComponent(version);

      // CSS recommended by several implementations (client/component view)
      addCss(base + '/css/bootstrap.css');
      addCss(base + '/css/react-select.css');

      // Vendor deps (some SDK builds expect these globals)
      await loadScript(base + '/lib/vendor/react.min.js');
      await loadScript(base + '/lib/vendor/react-dom.min.js');
      await loadScript(base + '/lib/vendor/redux.min.js');
      await loadScript(base + '/lib/vendor/redux-thunk.min.js');
      await loadScript(base + '/lib/vendor/lodash.min.js');

      // Meeting SDK scripts
      await loadScript(base + '/zoom-meeting-embedded-' + version + '.min.js');

      // Some setups also load zoom-meeting-{version}.min.js (safe fallback)
      try {
        await loadScript(base + '/zoom-meeting-' + version + '.min.js');
      } catch (e) {
        // Ignore if not present for this version.
      }

      if (typeof window.ZoomMtgEmbedded === 'undefined') {
        throw new Error('ZoomMtgEmbedded no está disponible. Revisa la versión del SDK y el acceso a source.zoom.us.');
      }
    })();

    return window.__zdzeZoomAssetsPromise;
  }

  /**
   * Call WP REST endpoint to get signature.
   * @param {string} token
   * @param {string} meetingNumber
   * @returns {Promise<{signature:string, sdkKey:string}>}
   */
  async function fetchSignature(token, meetingNumber) {
    var res = await fetch(ZDZE.restUrl, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({
        token: token,
        meetingNumber: meetingNumber,
        role: 0
      })
    });

    var data = await res.json().catch(function () { return null; });
    if (!res.ok || !data || !data.ok) {
      var msg = (data && data.error) ? data.error : ('HTTP ' + res.status);
      throw new Error('No se pudo generar signature: ' + msg);
    }
    return { signature: data.signature, sdkKey: data.sdkKey };
  }

  /**
   * Initialize a single embed container.
   * @param {HTMLElement} wrap
   */
  async function initEmbed(wrap) {
    var meetingNumber = (wrap.getAttribute('data-meeting-number') || '').trim();
    var passcode = (wrap.getAttribute('data-passcode') || '').trim();
    var token = (wrap.getAttribute('data-token') || '').trim();
    var autoJoin = (wrap.getAttribute('data-auto-join') || '1') === '1';

    var overlay = wrap.querySelector('.zdze-zoom-overlay');
    var root = wrap.querySelector('.zdze-zoom-root');
    var btn = wrap.querySelector('.zdze-zoom-btn');
    var input = wrap.querySelector('.zdze-zoom-input');
    var msg = wrap.querySelector('.zdze-zoom-msg');
    var subtitle = wrap.querySelector('.zdze-zoom-subtitle');

    var joining = false;

    function setMsg(text) {
      if (msg) msg.textContent = text || '';
    }

    function setSubtitle(text) {
      if (subtitle) subtitle.textContent = text || '';
    }

    function showOverlay(show) {
      if (!overlay) return;
      overlay.style.display = show ? 'flex' : 'none';
      overlay.setAttribute('aria-hidden', show ? 'false' : 'true');
    }

    if (!meetingNumber || !token) {
      setMsg('Configuración incompleta: falta meetingNumber o token.');
      return;
    }

    // Height
    var height = parseInt(wrap.getAttribute('data-height') || '700', 10);
    if (!isNaN(height) && root) {
      root.style.height = height + 'px';
    }

    // user name
    var userName = (wrap.getAttribute('data-user-name') || '').trim();
    if (input && userName) {
      input.value = userName;
    }

    async function joinNow() {
      if (joining) { return; }
      joining = true;
      if (btn) { btn.disabled = true; }

      try {
        setMsg('');
        setSubtitle('Cargando Zoom SDK…');

        await ensureZoomAssets(ZDZE.sdkVersion);

        setSubtitle('Generando acceso seguro…');
        var sig = await fetchSignature(token, meetingNumber);

        setSubtitle('Conectando a la reunión…');

        var client = window.ZoomMtgEmbedded.createClient();

        var leaveUrl = (ZDZE.leaveUrl && ZDZE.leaveUrl.trim()) ? ZDZE.leaveUrl.trim() : window.location.href;

        // init
        await client.init({
          zoomAppRoot: root,
          language: (ZDZE.language || 'es-ES'),
          patchJsMedia: true,
          leaveUrl: leaveUrl
        });

        // join
        await client.join({
          sdkKey: sig.sdkKey,
          signature: sig.signature,
          meetingNumber: meetingNumber,
          password: passcode,
          userName: (input && input.value) ? input.value.trim() : (userName || 'Invitado')
        });

        showOverlay(false);
        // joined successfully
        joining = false;
      } catch (err) {
        setSubtitle('No fue posible iniciar');
        setMsg(err && err.message ? err.message : String(err));
        joining = false;
        if (btn) { btn.disabled = false; }
        showOverlay(true);
      }
    }

    if (btn) {
      btn.addEventListener('click', function () {
        joinNow();
      });
    }

    // Auto join (can still fail due to permissions; user can click again)
    if (autoJoin) {
      joinNow();
    } else {
      showOverlay(true);
      setSubtitle('Listo para entrar');
      setMsg('Haz click en "Entrar a la reunión".');
    }
  }

  function boot() {
    if (!window.ZDZE || !ZDZE.restUrl) {
      return;
    }
    var embeds = document.querySelectorAll('.zdze-zoom-embed');
    if (!embeds || !embeds.length) {
      return;
    }
    embeds.forEach(function (el) { initEmbed(el); });
  }

  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', boot);
  } else {
    boot();
  }
})();
