<?php
if (!defined('ABSPATH')) {
    exit;
}

class ZDZE_Public {

    private $should_enqueue = false;

    public function __construct() {
        add_action('wp', array($this, 'detect_embeds'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_assets'));

        add_shortcode('zd_zoom_embed', array($this, 'shortcode_embed'));
        add_filter('the_content', array($this, 'append_room_embed_to_content'), 20);
    }

    /**
     * Detect if current request needs assets.
     */
    public function detect_embeds() {
        if (is_singular(ZDZE_Room_CPT::POST_TYPE)) {
            $this->should_enqueue = true;
            return;
        }

        if (is_singular()) {
            $post = get_post();
            if ($post && has_shortcode($post->post_content, 'zd_zoom_embed')) {
                $this->should_enqueue = true;
                return;
            }
        }
    }

    public function enqueue_assets() {
        if (!$this->should_enqueue) {
            return;
        }

        wp_enqueue_style(
            'zdze-zoom',
            ZDZE_URL . 'public/css/zdze-zoom.css',
            array(),
            ZDZE_VERSION
        );

        wp_enqueue_script(
            'zdze-zoom',
            ZDZE_URL . 'public/js/zdze-zoom.js',
            array(),
            ZDZE_VERSION,
            true
        );

        $settings = zdze_get_settings();
        $leave_url = isset($settings['leave_url']) ? esc_url_raw($settings['leave_url']) : '';
        $sdk_version = isset($settings['sdk_version']) ? sanitize_text_field($settings['sdk_version']) : '5.1.0';
        $language = isset($settings['language']) ? sanitize_text_field($settings['language']) : 'es-ES';

        wp_localize_script('zdze-zoom', 'ZDZE', array(
            'restUrl'    => esc_url_raw(rest_url(ZDZE_REST::NAMESPACE . '/signature')),
            'sdkVersion' => $sdk_version,
            'language'   => $language,
            'leaveUrl'   => $leave_url,
        ));
    }

    /**
     * Shortcode:
     * [zd_zoom_embed meeting_number="123456789" passcode="ABC123" height="700" auto_join="1" user_name="Invitado"]
     */
    public function shortcode_embed($atts) {
        $atts = shortcode_atts(array(
            'meeting_number' => '',
            'passcode'       => '',
            'height'         => '700',
            'auto_join'      => '1',
            'user_name'      => '',
        ), $atts, 'zd_zoom_embed');

        $meeting_number = preg_replace('/[^0-9]/', '', sanitize_text_field($atts['meeting_number']));
        $passcode       = sanitize_text_field($atts['passcode']);
        $height         = absint($atts['height']);
        $auto_join      = $atts['auto_join'] === '0' ? '0' : '1';
        $user_name      = sanitize_text_field($atts['user_name']);

        if ($meeting_number === '') {
            return '<div class="zdze-zoom-error">' . esc_html__('Falta meeting_number en el shortcode.', 'zd-embed-for-zoom-meeting-sdk') . '</div>';
        }

        // Avoid full-page caching issues (tokens/signatures). Many caching plugins respect DONOTCACHEPAGE.nocache_headers();

        if ($height < 400) {
            $height = 400;
        }

        if ($user_name === '') {
            if (is_user_logged_in()) {
                $u = wp_get_current_user();
                $user_name = $u && $u->display_name ? $u->display_name : __('Invitado', 'zd-embed-for-zoom-meeting-sdk');
            } else {
                $user_name = __('Invitado', 'zd-embed-for-zoom-meeting-sdk');
            }
        }

        // Create one-time token (valid 15 minutes) for signature endpoint.
        $token = wp_generate_password(24, false, false);
        set_transient('zdze_token_' . $token, array(
            'meetingNumber' => $meeting_number,
            'role'          => 0,
            'createdAt'     => time(),
        ), 60 * MINUTE_IN_SECONDS);

        $container_id = 'zdze-zoom-' . wp_generate_password(8, false, false);

        // Ensure assets are enqueued even if shortcode is rendered in unexpected contexts.
        $this->should_enqueue = true;

        ob_start();
        ?>
        <div
            id="<?php echo esc_attr($container_id); ?>"
            class="zdze-zoom-embed"
            data-meeting-number="<?php echo esc_attr($meeting_number); ?>"
            data-passcode="<?php echo esc_attr($passcode); ?>"
            data-height="<?php echo esc_attr((string) $height); ?>"
            data-auto-join="<?php echo esc_attr($auto_join); ?>"
            data-user-name="<?php echo esc_attr($user_name); ?>"
            data-token="<?php echo esc_attr($token); ?>"
        >
            <div class="zdze-zoom-overlay" aria-hidden="false">
                <div class="zdze-zoom-card">
                    <div class="zdze-zoom-title"><?php echo esc_html__('Videoconferencia Zoom', 'zd-embed-for-zoom-meeting-sdk'); ?></div>
                    <div class="zdze-zoom-subtitle"><?php echo esc_html__('Preparando la sala…', 'zd-embed-for-zoom-meeting-sdk'); ?></div>

                    <label class="zdze-zoom-label">
                        <?php echo esc_html__('Tu nombre', 'zd-embed-for-zoom-meeting-sdk'); ?>
                        <input class="zdze-zoom-input" type="text" value="<?php echo esc_attr($user_name); ?>" />
                    </label>

                    <button type="button" class="zdze-zoom-btn">
                        <?php echo esc_html__('Entrar a la reunión', 'zd-embed-for-zoom-meeting-sdk'); ?>
                    </button>

                    <div class="zdze-zoom-msg" role="status" aria-live="polite"></div>
                    <div class="zdze-zoom-small">
                        <?php echo esc_html__('Tip: si el navegador bloquea audio/video, revisa los permisos de cámara y micrófono.', 'zd-embed-for-zoom-meeting-sdk'); ?>
                    </div>
                </div>
            </div>

            <div class="zdze-zoom-root" style="height: <?php echo esc_attr((string) $height); ?>px;"></div>
        </div>
        <?php
        return ob_get_clean();
    }


/**
 * Allowlist for wp_kses() so we can safely append our embed markup via the_content filter.
 *
 * @return array
 */
private function zdze_get_allowed_embed_kses() {
    $allowed = wp_kses_allowed_html('post');

    // Ensure our container and data-* attributes survive KSES.
    $allowed['div'] = isset($allowed['div']) ? $allowed['div'] : array();
    $allowed['div'] = array_merge($allowed['div'], array(
        'id' => true,
        'class' => true,
        'style' => true,
        'aria-hidden' => true,
        'role' => true,
        'aria-live' => true,
        'data-meeting-number' => true,
        'data-passcode' => true,
        'data-height' => true,
        'data-auto-join' => true,
        'data-user-name' => true,
        'data-token' => true,
    ));

    $allowed['label'] = isset($allowed['label']) ? $allowed['label'] : array();
    $allowed['label'] = array_merge($allowed['label'], array(
        'class' => true,
        'for' => true,
    ));

    $allowed['input'] = isset($allowed['input']) ? $allowed['input'] : array();
    $allowed['input'] = array_merge($allowed['input'], array(
        'class' => true,
        'type' => true,
        'value' => true,
    ));

    $allowed['button'] = isset($allowed['button']) ? $allowed['button'] : array();
    $allowed['button'] = array_merge($allowed['button'], array(
        'class' => true,
        'type' => true,
        'disabled' => true,
    ));

    $allowed['a'] = isset($allowed['a']) ? $allowed['a'] : array();
    $allowed['a'] = array_merge($allowed['a'], array(
        'href' => true,
        'class' => true,
        'target' => true,
        'rel' => true,
    ));

    return $allowed;
}

    /**
     * Append embed to "Sala Zoom" single pages automatically.
     */
    public function append_room_embed_to_content($content) {
        if (!is_singular(ZDZE_Room_CPT::POST_TYPE)) {
            return $content;
        }

        if (!in_the_loop() || !is_main_query()) {
            return $content;
        }

        $post_id = get_the_ID();
        $meeting_number = (string) get_post_meta($post_id, '_zdze_meeting_number', true);
        $passcode       = (string) get_post_meta($post_id, '_zdze_passcode', true);
        $auto_join      = (string) get_post_meta($post_id, '_zdze_auto_join', true);
        $height         = (string) get_post_meta($post_id, '_zdze_height', true);

        $meeting_number = preg_replace('/[^0-9]/', '', $meeting_number);
        $passcode = sanitize_text_field($passcode);
        $auto_join = $auto_join === '0' ? '0' : '1';
        $height = $height !== '' ? absint($height) : 700;

        if ($meeting_number === '') {
            $content .= '<div class="zdze-zoom-error">' . esc_html__('Esta Sala Zoom no tiene Meeting Number configurado.', 'zd-embed-for-zoom-meeting-sdk') . '</div>';
            return $content;
        }

        $shortcode = sprintf(
            '[zd_zoom_embed meeting_number="%s" passcode="%s" height="%d" auto_join="%s"]',
            esc_attr($meeting_number),
            esc_attr($passcode),
            (int) $height,
            esc_attr($auto_join)
        );

        $embed_html = do_shortcode($shortcode);

        // KSES allowlist to ensure safe output in the_content filter.
        $content .= wp_kses($embed_html, $this->zdze_get_allowed_embed_kses());

        return $content;
    }
}
