<?php
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Base64 URL-safe encode.
 *
 * @param string $data
 * @return string
 */
function zdze_base64url_encode($data) {
    $b64 = base64_encode($data);
    $b64 = str_replace(array('+', '/', '='), array('-', '_', ''), $b64);
    return $b64;
}

/**
 * Read plugin settings (options) merged with defaults.
 *
 * @return array
 */
function zdze_get_settings() {
    $defaults = array(
        'sdk_key'       => '',
        'sdk_secret'    => '',
        'sdk_version'   => '5.1.0',
        'language'      => 'es-ES',
        'leave_url'     => '',
        'allowed_hosts' => '', // optional comma-separated list (e.g., "example.com,miweb.com")
    );

    $saved = get_option(ZDZE_OPTION_KEY, array());
    if (!is_array($saved)) {
        $saved = array();
    }

    return array_merge($defaults, $saved);
}

/**
 * Get SDK Key (priority: wp-config constants -> options).
 *
 * Define in wp-config.php (recommended):
 *   define('ZDZE_SDK_KEY', '...'); 
 *   define('ZDZE_SDK_SECRET', '...');
 *
 * @return string
 */
function zdze_get_sdk_key() {
    if (defined('ZDZE_SDK_KEY') && is_string(ZDZE_SDK_KEY) && ZDZE_SDK_KEY !== '') {
        return ZDZE_SDK_KEY;
    }
    $s = zdze_get_settings();
    return isset($s['sdk_key']) ? (string) $s['sdk_key'] : '';
}

/**
 * Get SDK Secret (priority: wp-config constants -> options).
 *
 * @return string
 */
function zdze_get_sdk_secret() {
    if (defined('ZDZE_SDK_SECRET') && is_string(ZDZE_SDK_SECRET) && ZDZE_SDK_SECRET !== '') {
        return ZDZE_SDK_SECRET;
    }
    $s = zdze_get_settings();
    return isset($s['sdk_secret']) ? (string) $s['sdk_secret'] : '';
}

/**
 * Generate Zoom Meeting SDK Web signature (JWT HS256).
 *
 * Payload fields are based on Zoom Meeting SDK signature format commonly used in examples:
 * - sdkKey / appKey: SDK Key
 * - mn: meeting number
 * - role: 0 attendee, 1 host
 * - iat: issued at (seconds)
 * - exp: expiration
 * - tokenExp: token expiration
 *
 * @param string $sdk_key
 * @param string $sdk_secret
 * @param string $meeting_number
 * @param int    $role
 * @return string
 */
function zdze_generate_meeting_sdk_signature($sdk_key, $sdk_secret, $meeting_number, $role) {
    $iat = time() - 30; // small clock skew buffer
    $exp = $iat + (2 * 60 * 60); // 2 hours
    $token_exp = $exp;

    $header = array(
        'alg' => 'HS256',
        'typ' => 'JWT',
    );

    $payload = array(
        'sdkKey'   => (string) $sdk_key,
        'appKey'   => (string) $sdk_key,
        'mn'       => (string) $meeting_number,
        'role'     => (int) $role,
        'iat'      => (int) $iat,
        'exp'      => (int) $exp,
        'tokenExp' => (int) $token_exp,
    );

    $header_b64  = zdze_base64url_encode(wp_json_encode($header));
    $payload_b64 = zdze_base64url_encode(wp_json_encode($payload));

    $unsigned = $header_b64 . '.' . $payload_b64;
    $sig_bin  = hash_hmac('sha256', $unsigned, $sdk_secret, true);
    $sig_b64  = zdze_base64url_encode($sig_bin);

    return $unsigned . '.' . $sig_b64;
}

/**
 * Optional: validate allowed hostnames for security (signature endpoint).
 * If settings "allowed_hosts" is empty, allow all.
 *
 * @return bool
 */
function zdze_is_request_host_allowed() {
    $settings = zdze_get_settings();
    $allowed = isset($settings['allowed_hosts']) ? trim((string) $settings['allowed_hosts']) : '';
    if ($allowed === '') {
        return true;
    }

    $host = '';
    if (!empty($_SERVER['HTTP_HOST'])) {
        $host = sanitize_text_field(wp_unslash($_SERVER['HTTP_HOST']));
    }
    if ($host === '') {
        return false;
    }

    $allowed_hosts = array_filter(array_map('trim', explode(',', $allowed)));
    $allowed_hosts = array_map('strtolower', $allowed_hosts);
    $host = strtolower($host);

    return in_array($host, $allowed_hosts, true);
}
