<?php
if (!defined('ABSPATH')) {
    exit;
}

class ZDZE_Settings {

    public function __construct() {
        add_action('admin_menu', array($this, 'add_menu'));
        add_action('admin_init', array($this, 'register_settings'));
    }

    public function add_menu() {
        add_options_page(
            __('ZD Zoom Embed', 'zd-embed-for-zoom-meeting-sdk'),
            __('ZD Zoom Embed', 'zd-embed-for-zoom-meeting-sdk'),
            'manage_options',
            'zdze-settings',
            array($this, 'render_page')
        );
    }

    public function register_settings() {
        register_setting(
            'zdze_settings_group',
            ZDZE_OPTION_KEY,
            array(
                'type'              => 'array',
                'sanitize_callback' => array($this, 'sanitize_settings'),
                'default'           => array(),
            )
        );

        add_settings_section(
            'zdze_main_section',
            __('Zoom Meeting SDK settings', 'zd-embed-for-zoom-meeting-sdk'),
            array($this, 'render_section_help'),
            'zdze-settings'
        );

        add_settings_field(
            'sdk_key',
            __('SDK Key / Client ID', 'zd-embed-for-zoom-meeting-sdk'),
            array($this, 'field_sdk_key'),
            'zdze-settings',
            'zdze_main_section'
        );

        add_settings_field(
            'sdk_secret',
            __('SDK Secret / Client Secret', 'zd-embed-for-zoom-meeting-sdk'),
            array($this, 'field_sdk_secret'),
            'zdze-settings',
            'zdze_main_section'
        );

        add_settings_field(
            'sdk_version',
            __('SDK Version (CDN)', 'zd-embed-for-zoom-meeting-sdk'),
            array($this, 'field_sdk_version'),
            'zdze-settings',
            'zdze_main_section'
        );

        add_settings_field(
            'language',
            __('Language', 'zd-embed-for-zoom-meeting-sdk'),
            array($this, 'field_language'),
            'zdze-settings',
            'zdze_main_section'
        );

        add_settings_field(
            'leave_url',
            __('Leave URL (optional)', 'zd-embed-for-zoom-meeting-sdk'),
            array($this, 'field_leave_url'),
            'zdze-settings',
            'zdze_main_section'
        );

        add_settings_field(
            'allowed_hosts',
            __('Allowed hosts for signature endpoint (optional)', 'zd-embed-for-zoom-meeting-sdk'),
            array($this, 'field_allowed_hosts'),
            'zdze-settings',
            'zdze_main_section'
        );
    }

    public function sanitize_settings($input) {
    $out = array();

    $out['sdk_key']       = isset($input['sdk_key']) ? sanitize_text_field($input['sdk_key']) : '';
    $out['sdk_version']   = isset($input['sdk_version']) ? sanitize_text_field($input['sdk_version']) : '5.1.0';
    $out['language']      = isset($input['language']) ? sanitize_text_field($input['language']) : 'en-US';
    $out['leave_url']     = isset($input['leave_url']) ? esc_url_raw($input['leave_url']) : '';
    $out['allowed_hosts'] = isset($input['allowed_hosts']) ? sanitize_text_field($input['allowed_hosts']) : '';

    /**
     * SDK Secret is a password-like value and should NOT be sanitized with sanitize_text_field()
     * as it can alter/corrupt secrets. We store it as a raw trimmed string, and escape on output.
     * If the field is left empty, keep the previously stored value (to avoid accidental erasure).
     */
    $current = $this->get_settings();
    $raw_secret = '';
    if (isset($input['sdk_secret'])) {
        $raw_secret = trim((string) wp_unslash($input['sdk_secret']));
        $raw_secret = str_replace("\0", '', $raw_secret); // remove null bytes
    }

    if ($raw_secret === '' && isset($current['sdk_secret']) && !defined('ZDZE_SDK_SECRET')) {
        $out['sdk_secret'] = (string) $current['sdk_secret'];
    } else {
        $out['sdk_secret'] = $raw_secret;
    }

    return $out;
}

public function render_section_help() {
        echo '<p style="max-width:900px;">';
        echo esc_html__('This plugin uses the Zoom Meeting SDK (Web) to embed meetings inside WordPress. Recommended: define ZDZE_SDK_KEY and ZDZE_SDK_SECRET in wp-config.php (more secure than saving secrets in the database).', 'zd-embed-for-zoom-meeting-sdk');
        echo '</p>';

        echo '<p style="max-width:900px;">';
        echo esc_html__('If you already defined the constants in wp-config.php, those values take priority and the fields below will be locked.', 'zd-embed-for-zoom-meeting-sdk');
        echo '</p>';

        echo '<div style="margin:12px 0;padding:12px;border:1px solid #dcdcde;background:#fff;border-radius:8px;max-width:900px;">';
        echo '<strong>' . esc_html__('Quick guide: create your Zoom Meeting SDK app', 'zd-embed-for-zoom-meeting-sdk') . '</strong>';
        echo '<ol style="margin:8px 0 0 18px;">';
        echo '<li>' . esc_html__('Zoom App Marketplace → Develop → Build App → General App.', 'zd-embed-for-zoom-meeting-sdk') . '</li>';
        echo '<li>' . esc_html__('Open “Embed” and enable the “Meeting SDK” toggle.', 'zd-embed-for-zoom-meeting-sdk') . '</li>';
        echo '<li>' . esc_html__('Copy Client ID and Client Secret from “App Credentials”.', 'zd-embed-for-zoom-meeting-sdk') . '</li>';
        echo '<li>' . esc_html__('Paste them here or define constants in wp-config.php.', 'zd-embed-for-zoom-meeting-sdk') . '</li>';
        echo '</ol>';

        echo '<p style="margin:10px 0 0;">';
        echo '<a href="' . esc_url('https://developers.zoom.us/docs/meeting-sdk/create/') . '" target="_blank" rel="noopener noreferrer">' . esc_html__('Official Zoom Meeting SDK docs', 'zd-embed-for-zoom-meeting-sdk') . '</a>';
        echo ' | ';
        echo '<a href="' . esc_url('https://marketplace.zoom.us/') . '" target="_blank" rel="noopener noreferrer">' . esc_html__('Open Zoom App Marketplace', 'zd-embed-for-zoom-meeting-sdk') . '</a>';
        echo '</p>';
        echo '</div>';
    }

    private function get_settings() {
        return zdze_get_settings();
    }

    public function field_sdk_key() {
        $s = $this->get_settings();
        $val = isset($s['sdk_key']) ? $s['sdk_key'] : '';
        $is_locked = defined('ZDZE_SDK_KEY');

        echo '<input type="text" class="regular-text" name="' . esc_attr(ZDZE_OPTION_KEY) . '[sdk_key]" value="' . esc_attr($val) . '" ' . disabled($is_locked, true, false) . ' />';

        if ($is_locked) {
            echo '<p class="description">' . esc_html__('Currently using the value from wp-config.php (ZDZE_SDK_KEY).', 'zd-embed-for-zoom-meeting-sdk') . '</p>';
        }
    }

    public function field_sdk_secret() {
        $s = $this->get_settings();
        $val = isset($s['sdk_secret']) ? $s['sdk_secret'] : '';
        $is_locked = defined('ZDZE_SDK_SECRET');

        echo '<input type="password" class="regular-text" autocomplete="new-password" name="' . esc_attr(ZDZE_OPTION_KEY) . '[sdk_secret]" value="' . esc_attr($val) . '" ' . disabled($is_locked, true, false) . ' />';
        echo '<p class="description">' . esc_html__('Avoid sharing the secret. It is recommended to define it in wp-config.php.', 'zd-embed-for-zoom-meeting-sdk') . '</p>';

        if ($is_locked) {
            echo '<p class="description">' . esc_html__('Currently using the value from wp-config.php (ZDZE_SDK_SECRET).', 'zd-embed-for-zoom-meeting-sdk') . '</p>';
        }
    }

    public function field_sdk_version() {
        $s = $this->get_settings();
        $val = isset($s['sdk_version']) ? $s['sdk_version'] : '5.1.0';

        echo '<input type="text" class="regular-text" name="' . esc_attr(ZDZE_OPTION_KEY) . '[sdk_version]" value="' . esc_attr($val) . '" />';
        echo '<p class="description">' . esc_html__('Example: 5.1.0 (used to load assets from https://source.zoom.us/{VERSION}/). If something fails, try 5.0.4.', 'zd-embed-for-zoom-meeting-sdk') . '</p>';
    }

    public function field_language() {
        $s = $this->get_settings();
        $val = isset($s['language']) ? $s['language'] : 'en-US';

        echo '<input type="text" class="regular-text" name="' . esc_attr(ZDZE_OPTION_KEY) . '[language]" value="' . esc_attr($val) . '" />';
        echo '<p class="description">' . esc_html__('Examples: en-US, es-ES, pt-PT.', 'zd-embed-for-zoom-meeting-sdk') . '</p>';
    }

    public function field_leave_url() {
        $s = $this->get_settings();
        $val = isset($s['leave_url']) ? $s['leave_url'] : '';

        echo '<input type="url" class="regular-text" name="' . esc_attr(ZDZE_OPTION_KEY) . '[leave_url]" value="' . esc_attr($val) . '" placeholder="https://yoursite.com/thanks" />';
        echo '<p class="description">' . esc_html__('URL where users will be redirected after leaving. If empty, the current URL is used.', 'zd-embed-for-zoom-meeting-sdk') . '</p>';
    }

    public function field_allowed_hosts() {
        $s = $this->get_settings();
        $val = isset($s['allowed_hosts']) ? $s['allowed_hosts'] : '';

        echo '<input type="text" class="regular-text" name="' . esc_attr(ZDZE_OPTION_KEY) . '[allowed_hosts]" value="' . esc_attr($val) . '" placeholder="example.com, staging.example.com" />';
        echo '<p class="description">' . esc_html__('If set, the signature endpoint will only respond for these domains (HTTP_HOST). Useful for staging/prod.', 'zd-embed-for-zoom-meeting-sdk') . '</p>';
    }

    public function render_page() {
        if (!current_user_can('manage_options')) {
            return;
        }

        echo '<div class="wrap">';
        echo '<h1>' . esc_html__('ZD Zoom Embed', 'zd-embed-for-zoom-meeting-sdk') . '</h1>';
        echo '<p style="max-width:950px;">' . esc_html__('Create “Zoom Rooms” with a public URL, or embed a meeting with the shortcode below.', 'zd-embed-for-zoom-meeting-sdk') . '</p>';

        echo '<hr />';

        echo '<h2>' . esc_html__('Shortcode', 'zd-embed-for-zoom-meeting-sdk') . '</h2>';
        echo '<code>[zd_zoom_embed meeting_number="123456789" passcode="ABC123" height="700" auto_join="1" user_name="Guest"]</code>';
        echo '<p class="description">' . esc_html__('If user_name is not provided, the plugin uses the logged-in user display name or “Guest”.', 'zd-embed-for-zoom-meeting-sdk') . '</p>';

        echo '<hr />';

        echo '<form method="post" action="options.php">';
        settings_fields('zdze_settings_group');
        do_settings_sections('zdze-settings');
        submit_button();
        echo '</form>';

        echo '<hr />';

        echo '<h2>' . esc_html__('Recommended (wp-config.php)', 'zd-embed-for-zoom-meeting-sdk') . '</h2>';
        echo '<pre style="background:#fff;border:1px solid #ccd0d4;padding:12px;max-width:950px;overflow:auto;">';
        echo esc_html("define('ZDZE_SDK_KEY', 'YOUR_SDK_KEY_OR_CLIENT_ID');\n");
        echo esc_html("define('ZDZE_SDK_SECRET', 'YOUR_SDK_SECRET_OR_CLIENT_SECRET');\n");
        echo '</pre>';

        echo '</div>';
    }
}
