<?php
if (!defined('ABSPATH')) {
    exit;
}

class ZDZE_REST {

    const NAMESPACE = 'zd-zoom/v1';

    public function __construct() {
        add_action('rest_api_init', array($this, 'register_routes'));
    }

    public function register_routes() {
        register_rest_route(self::NAMESPACE, '/signature', array(
            'methods'             => 'POST',
            'callback'            => array($this, 'handle_signature'),
            'permission_callback' => '__return_true', // Public endpoint but protected by one-time token + allowed host.
        ));
    }

    /**
     * POST /wp-json/zd-zoom/v1/signature
     * Body: { token: "...", meetingNumber: "...", role: 0 }
     */
    public function handle_signature(WP_REST_Request $request) {

        if (!zdze_is_request_host_allowed()) {
            return new WP_REST_Response(array(
                'ok'    => false,
                'error' => 'Host not allowed.',
            ), 403);
        }

        $token         = (string) $request->get_param('token');
        $meetingNumber = (string) $request->get_param('meetingNumber');
        $role          = (int) $request->get_param('role');

        $token = sanitize_text_field($token);
        $meetingNumber = preg_replace('/[^0-9]/', '', sanitize_text_field($meetingNumber));

        if ($token === '' || $meetingNumber === '') {
            return new WP_REST_Response(array(
                'ok'    => false,
                'error' => 'Missing token or meetingNumber.',
            ), 400);
        }

        // Validate token (one-time transient created when rendering the embed)
        $transient_key = 'zdze_token_' . $token;
        $data = get_transient($transient_key);

        if (!is_array($data) || empty($data['meetingNumber'])) {
            return new WP_REST_Response(array(
                'ok'    => false,
                'error' => 'Invalid or expired token.',
            ), 403);
        }

        // Match meeting number
        if ((string) $data['meetingNumber'] !== (string) $meetingNumber) {
            return new WP_REST_Response(array(
                'ok'    => false,
                'error' => 'Token/meeting mismatch.',
            ), 403);
        }

        // Role check: we only allow attendee signatures by default.
        // You can extend this later for host (role=1) with additional security.
        if ($role !== 0) {
            return new WP_REST_Response(array(
                'ok'    => false,
                'error' => 'Only attendee role (0) is allowed.',
            ), 403);
        }

        $sdk_key    = zdze_get_sdk_key();
        $sdk_secret = zdze_get_sdk_secret();

        if ($sdk_key === '' || $sdk_secret === '') {
            return new WP_REST_Response(array(
                'ok'    => false,
                'error' => 'SDK Key/Secret not configured.',
            ), 500);
        }

        $signature = zdze_generate_meeting_sdk_signature($sdk_key, $sdk_secret, $meetingNumber, $role);

        // Allow retries until token expiry (do not delete transient here).
        // This avoids breaking when auto-join fails and the user clicks "Entrar" again.

        return new WP_REST_Response(array(
            'ok'        => true,
            'signature' => $signature,
            'sdkKey'    => $sdk_key,
        ), 200);
    }
}
