<?php
/**
 * The payment processor class.
 *
 * @package ZCO/Classes
 */

namespace Zaver;

use Exception;
use WC_Order;
use KrokedilZCODeps\Zaver\SDK\Config\PaymentStatus;
use KrokedilZCODeps\Zaver\SDK\Object\PaymentStatusResponse;
use Zaver\Classes\Helpers\Order;
use Zaver\Order_Management as OM;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class Payment_Processor
 *
 * Handles the payment processing.
 */
class Payment_Processor {

	/**
	 * Process a payment for the given order.
	 *
	 * @param WC_Order $order The order to process the payment for.
	 * @return void
	 */
	public static function process( $order ) {
		$payment = Order::create( $order );

		do_action( 'zco_before_process_payment', $payment, $order );
		$response = Plugin::gateway()->api()->createPayment( $payment );
		ZCO()->logger()->info(
			'Created Zaver payment request',
			array(
				'payload'   => $payment,
				'response'  => $response,
				'orderId'   => $order->get_id(),
				'paymentId' => $response->getPaymentId(),
			)
		);

		$token = $response->getToken();

		if ( ZCO()->separate_payment_methods_enabled() ) {
			$selected_payment_method   = str_replace( 'zaver_checkout_', '', $order->get_payment_method() );
			$available_payment_methods = $response->getSpecificPaymentMethodData();
			foreach ( $available_payment_methods as $payment_method ) {
				$title = strtolower( $payment_method['paymentMethod'] );
				if ( $selected_payment_method === $title ) {
					$order->update_meta_data( '_zaver_payment_method', $payment_method['paymentMethod'] );
					$order->update_meta_data( '_zaver_payment_link', $payment_method['paymentLink'] );
					$token = $payment_method['checkoutToken'];
					break;
				}
			}
		} else {
			$order->update_meta_data( '_zaver_payment_link', $response->getPaymentLink() );
		}

		$order->update_meta_data( '_zaver_status_history', array( $response->getPaymentStatus() ) );
		$order->update_meta_data( '_zaver_payment_id', $response->getPaymentId() );
		$order->update_meta_data(
			'_zaver_payment',
			array(
				'id'              => $response->getPaymentId(),
				'token'           => $token,
				'tokenValidUntil' => $response->getValidUntil(),
			)
		);

		$order->save_meta_data();

		// Save all line item IDs generated by Zaver.
		foreach ( $response->getLineItems() as $item ) {
			$meta = $item->getMerchantMetadata();

			if ( isset( $meta['orderItemId'] ) ) {
				wc_add_order_item_meta( $meta['orderItemId'], '_zaver_line_item_id', $item->getId(), true );
			}
		}

		do_action( 'zco_after_process_payment', $payment, $order, $response );
	}

	/**
	 * Handle the response from Zaver.
	 *
	 * @throws Exception If the order key is invalid, the payment ID is missing, or the payment ID does not match.
	 *
	 * @param \WC_Order                  $order The order to handle the response for.
	 * @param PaymentStatusResponse|null $payment_status The payment status response.
	 * @param bool                       $redirect Whether to redirect the user.
	 * @return void
	 */
	public static function handle_response( $order, $payment_status = null, $redirect = true ) {
		// Ensure that the order key is correct.
		$key = filter_input( INPUT_GET, 'key', FILTER_SANITIZE_SPECIAL_CHARS );
		if ( empty( $key ) || ! hash_equals( $order->get_order_key(), $key ) ) {
			throw new Exception( 'Invalid order key' );
		}

		$payment_id = $order->get_meta( '_zaver_payment' )['id'] ?? $order->get_meta( '_zaver_payment_id' );
		if ( empty( $payment_id ) ) {
			throw new Exception( 'Missing payment ID on order' );
		}

		if ( null === $payment_status ) {
			$payment_status = Plugin::gateway()->api()->getPaymentStatus( $payment_id );
			ZCO()->logger()->info(
				'Fetched payment status from Zaver',
				array(
					'payload'  => $payment_id,
					'response' => $payment_status,
					'orderId'  => $order->get_id(),
				)
			);
		} elseif ( $payment_status->getPaymentId() !== $payment_id ) {
			ZCO()->logger()->info(
				"Mismatching payment ID. Expected {$payment_id}, received {$payment_status->getPaymentId()}",
				array(
					'payload'  => $payment_id,
					'response' => $payment_status,
					'orderId'  => $order->get_id(),
				)
			);
			throw new Exception( 'Mismatching payment ID' );
		}

		// To prevent processing the same payment status multiple times, we keep track of the status history.
		$status_history = $order->get_meta( '_zaver_status_history' );
		$new_status     = $payment_status->getPaymentStatus();

		if ( in_array( $new_status, $status_history, true ) ) {
			ZCO()->logger()->debug(
				"Received an already processed payment status from Zaver: $new_status",
				array(
					'orderId'       => $order->get_id(),
					'paymentId'     => $payment_status->getPaymentId(),
					'paymentStatus' => $payment_status->getPaymentStatus(),
					'statusHistory' => $status_history,
				)
			);

			return;
		} else {
			$status_history[] = $new_status;
			$order->update_meta_data( '_zaver_status_history', $status_history );
			$order->save_meta_data();

		}

		do_action( 'zco_process_payment_handle_response', $order, $payment_status, $redirect );

		switch ( $payment_status->getPaymentStatus() ) {
			case PaymentStatus::SETTLED:
				// When the order is initially created, the captured amount is zero. If it is non-zero, it means the payment was settled immediately (e.g., bank transfer).
				if ( 0 >= ( $payment_status->getCapturedAmount() * 100 ) ) {
					ZCO()->logger()->info(
						'Successful payment with Zaver',
						array(
							'orderId'   => $order->get_id(),
							'paymentId' => $payment_status->getPaymentId(),
						)
					);
					// translators: %s is the payment ID.
					$order->add_order_note( sprintf( __( 'Successful payment with Zaver - payment ID: %s.', 'zco' ), $payment_status->getPaymentId() ) );
					$order->payment_complete( $payment_status->getPaymentId() );

					// Adds the metadata to allow the capture to be processed from the admin dashboard.
					OM::set_as_captured( $order );

				} else {
					// Handle capture request from Zaver.
					$currency            = $payment_status->getCurrency();
					$captured            = $payment_status->getCapturedAmount();
					$remaining           = $payment_status->getAmount() - $captured;
					$formatted_remaining = OM::format_price( $remaining, $currency );
					ZCO()->logger()->info(
						'Zaver payment was captured',
						array(
							'orderId'         => $order->get_id(),
							'paymentId'       => $payment_status->getPaymentId(),
							'capturedAmount'  => OM::format_price( $captured ),
							'remainingAmount' => $formatted_remaining,
						)
					);
					// translators: %1$s is the payment ID, %2$s is the captured amount, %3$s is the remaining amount to capture.
					$order->add_order_note( sprintf( __( 'Zaver payment was captured - payment ID: %1$s. Captured amount: %2$s. Remaining amount to capture: %3$s.', 'zco' ), $payment_status->getPaymentId(), OM::format_price( $captured, $currency ), $formatted_remaining ) );

					if ( ( $remaining * 100 ) <= 0 ) {
						// Adds the metadata to allow the capture to be processed from the admin dashboard.
						OM::set_as_captured( $order );
					}

					$order->payment_complete( $payment_status->getPaymentId() );
				}

				break;

			case PaymentStatus::CANCELLED:
				ZCO()->logger()->info(
					'Zaver Payment was cancelled',
					array(
						'orderId'   => $order->get_id(),
						'paymentId' => $payment_status->getPaymentId(),
					)
				);

				if ( $redirect ) {
					wp_safe_redirect( $order->get_cancel_order_url() );
					exit;
				}

				$order->update_status( 'cancelled', __( 'Zaver payment was cancelled - cancelling order.', 'zco' ) );
				break;

			case PaymentStatus::PENDING:
				ZCO()->logger()->info(
					'Zaver payment is pending merchant capture',
					array(
						'orderId'   => $order->get_id(),
						'paymentId' => $payment_status->getPaymentId(),
					)
				);

				$order->payment_complete( $payment_status->getPaymentId() );
				// translators: %s is the payment ID.
				$order->add_order_note( sprintf( __( 'Successful payment with Zaver - payment ID: %s.', 'zco' ), $payment_status->getPaymentId() ) );
				$order->add_order_note( sprintf( __( 'Zaver payment is pending capture.', 'zco' ), $payment_status->getPaymentId() ) );
				$order->save();
				break;

			case PaymentStatus::PENDING_CONFIRMATION:
				ZCO()->logger()->info(
					'Zaver payment is pending confirmation',
					array(
						'orderId'   => $order->get_id(),
						'paymentId' => $payment_status->getPaymentId(),
					)
				);

				$order->set_transaction_id( $payment_status->getPaymentId() );
				$order->update_status( 'on-hold', __( 'Zaver Payment is pending confirmation.', 'zco' ) );
				break;

			default:
				ZCO()->logger()->error(
					'Received unhandled payment status from Zaver',
					array(
						'orderId'       => $order->get_id(),
						'paymentId'     => $payment_status->getPaymentId(),
						'paymentStatus' => $payment_status->getPaymentStatus(),
					)
				);
				break;
		}
	}
}
