<?php
/**
 * Planner functionality for YoApy Social Poster
 *
 * @package YoApySocialPoster
 * @since 1.6.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class YOAPSOPO_Planner
{
    private static $instance = null;
    private $option_key = 'yoapsopo_tasks';

    public static function get_instance()
    {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    public function __construct()
    {
        add_action('admin_menu', array($this, 'admin_menu'));
        add_action('wp_ajax_yoapsopo_task_action', array($this, 'ajax_task_action'));
        add_action('wp_ajax_yoapsopo_save_task_ajax', array($this, 'ajax_save_task'));
        add_action('wp_ajax_yoapsopo_get_tasks_ajax', array($this, 'ajax_get_tasks'));
        add_action('wp_ajax_yoapsopo_get_tasks', array($this, 'ajax_get_tasks')); // For admin.js compatibility
        add_action('admin_post_yoapsopo_save_task', array($this, 'handle_save_task_fallback')); // Apenas como fallback
    }

    /**
     * Get all tasks sorted by ID (descending)
     *
     * @return array
     */
    public function get_tasks()
    {
        $tasks = get_option($this->option_key, array());
        $sorted_tasks = is_array($tasks) ? $tasks : array();
        // Ordena por ID decrescente para mostrar os mais novos primeiro
        usort($sorted_tasks, function ($a, $b) {
            return ($b['id'] ?? 0) <=> ($a['id'] ?? 0);
        });
        return $sorted_tasks;
    }

    /**
     * Public method to get tasks (for external access)
     * @deprecated 1.6.1 Use get_tasks() directly
     */
    public function get_tasks_public()
    {
        return $this->get_tasks();
    }

    private function save_tasks($tasks)
    {
        update_option($this->option_key, array_values($tasks), false);
    }

    private function next_id($tasks)
    {
        if (empty($tasks))
            return 1;
        $ids = wp_list_pluck($tasks, 'id');
        return max($ids) + 1;
    }

    public function admin_menu()
    {
        // Enqueue modular planner scripts
        $planner_modules = array(
            'core' => 'admin/js/planner/planner-core.js',
            'media' => 'admin/js/planner/planner-media.js',
            'preview' => 'admin/js/planner/planner-preview.js',
            'tasks' => 'admin/js/planner/planner-tasks.js',
            'form' => 'admin/js/planner/planner-form.js',
            'ui' => 'admin/js/planner/planner-ui.js',
            'modal' => 'admin/js/planner/planner-modal.js',
        );

        // Enqueue Core first
        if (file_exists(YOAPSOPO_PLUGIN_DIR . $planner_modules['core'])) {
            wp_enqueue_script('yoapsopo-planner-core', YOAPSOPO_PLUGIN_URL . $planner_modules['core'], array('jquery'), YOAPSOPO_VERSION, true);

            wp_localize_script('yoapsopo-planner-core', 'yoapsopo_ajax_object', array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'admin_url' => admin_url(),
                'nonce' => wp_create_nonce('yoapsopo_ajax_nonce'),
                'i18n' => array(
                    'saving' => __('Saving...', 'yoapy-social-poster'),
                    'taskSaved' => __('Task saved successfully!', 'yoapy-social-poster'),
                    'error' => __('An error occurred.', 'yoapy-social-poster'),
                    'deleteConfirm' => __('Are you sure you want to delete this task?', 'yoapy-social-poster'),
                    'confirmDelete' => __('Confirm Delete', 'yoapy-social-poster'),
                    'deleteWarning' => __('Note: This will only delete the task from this list. To delete posts from social networks, you must delete them directly on each platform.', 'yoapy-social-poster'),
                    'actionSuccess' => __('Action successful!', 'yoapy-social-poster'),
                    'actionFailed' => __('Action failed.', 'yoapy-social-poster'),
                    'send' => __('Post', 'yoapy-social-poster'),
                    'delete' => __('Delete', 'yoapy-social-poster'),
                    'cancel' => __('Cancel', 'yoapy-social-poster'),
                    'title' => __('Title', 'yoapy-social-poster'),
                    'networks' => __('Networks', 'yoapy-social-poster'),
                    'status' => __('Status', 'yoapy-social-poster'),
                    'actions' => __('Actions', 'yoapy-social-poster'),
                    'chooseMedia' => __('Choose or Upload Media', 'yoapy-social-poster'),
                ),
            ));
        }

        // Enqueue other modules
        foreach ($planner_modules as $key => $path) {
            if ($key === 'core')
                continue;
            if (file_exists(YOAPSOPO_PLUGIN_DIR . $path)) {
                wp_enqueue_script('yoapsopo-planner-' . $key, YOAPSOPO_PLUGIN_URL . $path, array('jquery', 'yoapsopo-planner-core'), YOAPSOPO_VERSION, true);
            }
        }

        add_menu_page(
            __('YoApy Planner', 'yoapy-social-poster'),
            __('YoApy', 'yoapy-social-poster'),
            'manage_options',
            'yoapsopo_planner',
            array($this, 'render_planner'),
            'dashicons-share',
            26
        );

        add_submenu_page(
            'yoapsopo_planner',
            __('Planner', 'yoapy-social-poster'),
            __('Planner', 'yoapy-social-poster'),
            'manage_options',
            'yoapsopo_planner',
            array($this, 'render_planner')
        );

        add_submenu_page(
            'yoapsopo_planner',
            __('Settings', 'yoapy-social-poster'),
            __('Settings', 'yoapy-social-poster'),
            'manage_options',
            'yoapsopo_settings',
            array($this, 'render_settings')
        );

        add_submenu_page(
            'yoapsopo_planner',
            __('Logs', 'yoapy-social-poster'),
            __('Logs', 'yoapy-social-poster'),
            'manage_options',
            'yoapsopo_logs',
            array($this, 'render_logs')
        );
    }

    public function render_planner()
    {
        wp_enqueue_media();

        // Enqueue planner form JavaScript
        wp_enqueue_script(
            'yoapsopo-planner-form',
            YOAPSOPO_PLUGIN_URL . 'admin/js/planner/planner-form.js',
            array('jquery'),
            YOAPSOPO_VERSION,
            true
        );

        wp_enqueue_script(
            'yoapsopo-planner-inline',
            YOAPSOPO_PLUGIN_URL . 'admin/js/yoapsopo-planner-inline.js',
            array('jquery', 'yoapsopo-admin-js'), // Depends on common admin JS for YOAPSOPO object
            YOAPSOPO_VERSION,
            true
        );

        // Localize script with necessary data
        wp_localize_script(
            'yoapsopo-planner-form',
            'YoApyPlanner',
            array(
                'config' => array(
                    'ajax_url' => admin_url('admin-ajax.php'),
                    'nonce' => wp_create_nonce('yoapsopo_ajax_nonce'),
                    'i18n' => array(
                        'saving' => __('Saving...', 'yoapy-social-poster'),
                        'send' => __('Post', 'yoapy-social-poster'),
                        'error' => __('An error occurred.', 'yoapy-social-poster'),
                        'taskSaved' => __('Task saved successfully!', 'yoapy-social-poster'),
                    )
                )
            )
        );

        $tasks = $this->get_tasks();
        $hasKeys = YOAPSOPO_Client::has_keys();
        include YOAPSOPO_PLUGIN_DIR . 'admin/views/planner-main.php';
    }

    public function render_settings()
    {


        wp_enqueue_script(
            'yoapsopo-settings',
            YOAPSOPO_PLUGIN_URL . 'admin/js/yoapsopo-settings.js',
            array('jquery'),
            YOAPSOPO_VERSION,
            true
        );

        wp_localize_script(
            'yoapsopo-settings',
            'YoApySettings',
            array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('yoapsopo_settings_nonce')
            )
        );

        include YOAPSOPO_PLUGIN_DIR . 'admin/views/settings.php';
    }

    public function render_logs()
    {
        if (isset($_POST['yoapsopo_clear_logs']) && check_admin_referer('yoapsopo_clear_logs')) {
            YOAPSOPO_Logger::clear();
        }
        if (isset($_POST['yoapsopo_delete_log']) && check_admin_referer('yoapsopo_delete_log')) {
            $post = wp_unslash($_POST);
            $line = isset($post['line']) ? absint($post['line']) : 0;
            YOAPSOPO_Logger::delete_line($line);
        }
        include YOAPSOPO_PLUGIN_DIR . 'admin/views/logs.php';
    }

    public function ajax_save_task()
    {
        if (!check_ajax_referer('yoapsopo_ajax_nonce', 'nonce', false) || !current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permission denied.'), 403);
        }

        $tasks = get_option($this->option_key, array()); // Pega na ordem original
        $task = $this->build_task_from_post($tasks, wp_unslash($_POST)); // passa unslashed; sanitiza dentro

        // Set status based on schedule time BEFORE saving
        $task['status'] = !empty($task['when']) && $task['when'] > time() ? 'scheduled' : 'pending';

        // SAVE IMMEDIATELY: Ensure task is persisted before any processing
        $tasks[] = $task;
        update_option($this->option_key, $tasks, false);

        // Log task creation
        YOAPSOPO_Logger::log('task_saved', array('task_id' => $task['id'], 'title' => $task['title'], 'status' => $task['status'], 'scheduled_time' => $task['when']));

        // Process immediate tasks (empty or past when values)
        if (empty($task['when']) || $task['when'] <= time()) {
            // Find the task we just added by ID to make sure we process the right one
            $task_index = null;
            foreach ($tasks as $index => $t) {
                if ($t['id'] == $task['id']) {
                    $task_index = $index;
                    break;
                }
            }

            if ($task_index !== null) {
                // run_task accepts by reference, so it will modify the task in the array
                $this->run_task($tasks[$task_index]); // Pass the reference to the specific task

                // Save again with results from run_task
                update_option($this->option_key, $tasks, false);

                // Log task processing
                YOAPSOPO_Logger::log('task_processed', array('task_id' => $task['id'], 'final_status' => $tasks[$task_index]['status']));
            }
        }

        // Ensure we return the most up-to-date task list after processing
        $updated_tasks = $this->get_tasks();
        wp_send_json_success(array('tasks' => $updated_tasks));
    }

    public function handle_save_task_fallback()
    {
        if (!current_user_can('manage_options') || !check_admin_referer('yoapsopo_save_task')) {
            wp_die('forbidden');
        }

        $tasks = get_option($this->option_key, array());
        $task = $this->build_task_from_post($tasks, wp_unslash($_POST)); // passa unslashed; sanitiza dentro

        // Set status based on schedule time BEFORE saving
        $task['status'] = !empty($task['when']) && $task['when'] > time() ? 'scheduled' : 'pending';

        $tasks[] = $task;

        update_option($this->option_key, $tasks, false);

        // Process immediate tasks after saving
        if (empty($task['when']) || $task['when'] <= time()) {
            // Find the task we just added by ID to make sure we process the right one
            $task_index = null;
            foreach ($tasks as $index => $t) {
                if ($t['id'] == $task['id']) {
                    $task_index = $index;
                    break;
                }
            }

            if ($task_index !== null) {
                $this->run_task($tasks[$task_index]); // Process the task we just added
                update_option($this->option_key, $tasks, false);
            }
        }

        wp_redirect(admin_url('admin.php?page=yoapsopo_planner&saved=1'));
        exit;
    }

    /**
     * Constrói a task a partir de dados de formulário (já passados por quem validou o nonce).
     * NÃO lê $_POST diretamente para evitar avisos de NonceVerification.
     *
     * @param array $tasks      Lista atual de tasks (para calcular o ID).
     * @param array $post_data  Dados do formulário (ex.: $_POST já unslashed).
     * @return array
     */
    private function build_task_from_post($tasks, $post_data)
    {
        $id = $this->next_id($tasks);

        $title = isset($post_data['title']) ? sanitize_text_field($post_data['title']) : '';
        $networks = isset($post_data['networks']) ? array_map('sanitize_text_field', (array) $post_data['networks']) : array();

        // Capture accounts
        $accounts = array();
        if (isset($post_data['accounts']) && is_array($post_data['accounts'])) {
            foreach ($post_data['accounts'] as $net => $accs) {
                if (is_array($accs)) {
                    $accounts[$net] = array_map('sanitize_text_field', $accs);
                }
            }
        }

        $type = isset($post_data['type']) ? sanitize_text_field($post_data['type']) : 'image';
        $text = isset($post_data['text']) ? wp_kses_post($post_data['text']) : '';
        $image_url = isset($post_data['image_url']) ? esc_url_raw($post_data['image_url']) : '';
        $video_url = isset($post_data['video_url']) ? esc_url_raw($post_data['video_url']) : '';
        $article_url = isset($post_data['article_url']) ? esc_url_raw($post_data['article_url']) : '';

        $when_input = isset($post_data['when']) ? sanitize_text_field($post_data['when']) : '';
        $when_input = trim($when_input);
        $when_ts = null;

        if ('' !== $when_input) {
            try {
                $tz = wp_timezone();
                // datetime-local esperado "Y-m-d\TH:i"
                $dt = new DateTimeImmutable($when_input, $tz);
                $when_ts = $dt->getTimestamp();
            } catch (Exception $e) {
                $when_ts = null;
            }
        }

        // TikTok specific fields
        $tiktok_privacy = isset($post_data['tiktok_privacy']) ? sanitize_text_field($post_data['tiktok_privacy']) : '';
        $tiktok_allow_comment = isset($post_data['tiktok_allow_comment']) ? '1' : '0';
        $tiktok_allow_duet = isset($post_data['tiktok_allow_duet']) ? '1' : '0';
        $tiktok_allow_stitch = isset($post_data['tiktok_allow_stitch']) ? '1' : '0';
        $tiktok_commercial_toggle = isset($post_data['tiktok_commercial_toggle']) ? '1' : '0';
        $tiktok_commercial_your_brand = isset($post_data['tiktok_commercial_your_brand']) ? '1' : '0';
        $tiktok_commercial_branded = isset($post_data['tiktok_commercial_branded']) ? '1' : '0';
        $tiktok_consent = isset($post_data['tiktok_consent']) ? '1' : '0';

        return array(
            'id' => $id,
            'title' => $title,
            'networks' => $networks,
            'accounts' => $accounts,
            'type' => $type,
            'text' => $text,
            'image_url' => $image_url,
            'video_url' => $video_url,
            'article_url' => $article_url,
            'when' => $when_ts,
            'status' => 'pending',
            'api_task_ids' => array(),
            'results' => array(),
            // TikTok specific fields
            'tiktok_privacy' => $tiktok_privacy,
            'tiktok_allow_comment' => $tiktok_allow_comment,
            'tiktok_allow_duet' => $tiktok_allow_duet,
            'tiktok_allow_stitch' => $tiktok_allow_stitch,
            'tiktok_commercial_toggle' => $tiktok_commercial_toggle,
            'tiktok_commercial_your_brand' => $tiktok_commercial_your_brand,
            'tiktok_commercial_branded' => $tiktok_commercial_branded,
        );
    }

    public function ajax_task_action()
    {
        if (!check_ajax_referer('yoapsopo_ajax_nonce', 'nonce', false) || !current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permission denied.'), 403);
        }

        $post = wp_unslash($_POST);
        $act = isset($post['act']) ? sanitize_text_field($post['act']) : '';
        $id = isset($post['id']) ? absint($post['id']) : 0;

        $this->do_task_action($act, $id);
        wp_send_json_success(array('tasks' => $this->get_tasks()));
    }

    private function do_task_action($act, $id)
    {
        $tasks = get_option($this->option_key, array());
        $task_key = null;

        foreach ($tasks as $key => $t) {
            if (($t['id'] ?? 0) == $id) {
                $task_key = $key;
                break;
            }
        }

        if ($task_key !== null) {
            if ($act === 'delete') {
                $deleted_task = $tasks[$task_key];
                unset($tasks[$task_key]);
                update_option($this->option_key, array_values($tasks), false);
                YOAPSOPO_Logger::log('task_deleted', array('task_id' => $id, 'title' => $deleted_task['title']));
            } elseif ($act === 'send') {
                $this->run_task($tasks[$task_key]);
                // Save the updated task status back to the database
                update_option($this->option_key, $tasks, false);
            }
        }
    }

    public function run_task(&$task)
    {
        $client = new YOAPSOPO_Client();

        if (!YOAPSOPO_Client::has_keys()) {
            $task['status'] = 'error';
            $task['results']['system'] = array('success' => false, 'message' => 'API keys are not configured.');
            return;
        }

        // Check if this is a scheduled task
        if (!empty($task['when']) && $task['when'] > time()) {
            $task['status'] = 'scheduled';
            return;
        }

        $task['status'] = 'processing';
        $iso = (!empty($task['when']) && $task['when'] > time() + 60)
            ? gmdate('Y-m-d H:i:s', $task['when'])
            : '';

        $all_successful = true;
        $has_async_tasks = false;

        foreach ($task['networks'] as $net) {
            // Prepare TikTok fields if this is a TikTok post
            $tiktok_fields = array();
            if ($net === 'tiktok') {
                $tiktok_fields = array(
                    'tiktok_privacy' => $task['tiktok_privacy'] ?? '',
                    'tiktok_allow_comment' => $task['tiktok_allow_comment'] ?? '0',
                    'tiktok_allow_duet' => $task['tiktok_allow_duet'] ?? '0',
                    'tiktok_allow_stitch' => $task['tiktok_allow_stitch'] ?? '0',
                    'tiktok_commercial_toggle' => $task['tiktok_commercial_toggle'] ?? '0',
                    'tiktok_commercial_your_brand' => $task['tiktok_commercial_your_brand'] ?? '0',
                    'tiktok_commercial_branded' => $task['tiktok_commercial_branded'] ?? '0',
                );
            }

            // Determine accounts to post to for this network
            $target_accounts = array();
            if (!empty($task['accounts'][$net]) && is_array($task['accounts'][$net])) {
                $target_accounts = $task['accounts'][$net];
            } else {
                // Fallback for legacy tasks or if no specific account selected (shouldn't happen with new UI)
                $target_accounts = array(''); // Empty string triggers default account in client
            }

            foreach ($target_accounts as $acc_handle) {
                // Create a unique key for this result: network_handle (or just network if single)
                // For backward compatibility, if it's the only account or empty handle, use network name
                $result_key = $net;
                if ($acc_handle !== '' && count($target_accounts) > 1) {
                    $result_key = $net . ':' . $acc_handle;
                } elseif ($acc_handle !== '') {
                    // Even for single account, if it's explicit, let's try to use a composite key 
                    // BUT for backward compatibility with check_task_results, we might need to be careful.
                    // Let's stick to: if multiple accounts, use composite key. If single, use network key?
                    // Actually, check_task_results iterates over api_task_ids keys. So we can use composite keys freely.
                    $result_key = $net . ':' . $acc_handle;
                }

                $res = $client->create_post_json($net, $task['type'], $task['text'], $task['image_url'], $task['video_url'], $task['article_url'], $iso, $tiktok_fields, $acc_handle);

                if (is_wp_error($res)) {
                    $task['status'] = 'error';
                    $task['results'][$result_key] = array('success' => false, 'message' => $res->get_error_message());
                    $all_successful = false;
                    YOAPSOPO_Logger::log('task_network_error', array(
                        'task_id' => $task['id'],
                        'network' => $net,
                        'account' => $acc_handle,
                        'error' => $res->get_error_message()
                    ));
                } else {
                    $task['results'][$result_key] = $res['body'];
                    if (!empty($res['body']['task_id'])) {
                        $task['api_task_ids'][$result_key] = $res['body']['task_id'];
                        $has_async_tasks = true;
                        YOAPSOPO_Logger::log('task_async_created', array(
                            'task_id' => $task['id'],
                            'network' => $net,
                            'account' => $acc_handle,
                            'api_task_id' => $res['body']['task_id']
                        ));
                    } elseif (isset($res['body']['success']) && true === $res['body']['success']) {
                        // For immediate success, we can mark as complete
                        YOAPSOPO_Logger::log('task_immediate_success', array(
                            'task_id' => $task['id'],
                            'network' => $net,
                            'account' => $acc_handle
                        ));
                    } else {
                        $task['status'] = 'error';
                        $all_successful = false;
                        YOAPSOPO_Logger::log('task_network_failed', array(
                            'task_id' => $task['id'],
                            'network' => $net,
                            'account' => $acc_handle,
                            'response' => $res['body']
                        ));
                    }
                }
            }
        }

        // If all networks were successful and we don't have async tasks, mark as complete
        if ($all_successful && !$has_async_tasks) {
            $task['status'] = 'complete';
            YOAPSOPO_Logger::log('task_completed_immediately', array('task_id' => $task['id']));
        }
    }

    /**
     * Run tasks that are due (scheduled tasks that should be processed now)
     */
    public function run_due_tasks()
    {
        $tasks = get_option($this->option_key, array());
        $updated = false;

        foreach ($tasks as &$task) {
            // Check if task is scheduled and due time has passed
            if (
                isset($task['status']) && $task['status'] === 'scheduled' &&
                isset($task['when']) && $task['when'] <= time()
            ) {
                // Run the task
                YOAPSOPO_Logger::log('do_job_start', array('task_id' => $task['id']));
                $this->run_task($task);
                YOAPSOPO_Logger::log('do_job_end', array('task_id' => $task['id'], 'status' => $task['status']));
                $updated = true;
            }
        }

        // Save updated tasks if any were processed
        if ($updated) {
            update_option($this->option_key, $tasks, false);
        }

        // Return whether any tasks were updated
        return $updated;
    }

    /**
     * Check the status of tasks that are being processed
     */
    public function check_task_results()
    {
        $tasks = get_option($this->option_key, array());
        $updated = false;
        $client = new YOAPSOPO_Client();



        foreach ($tasks as &$task) {
            // Check if task is processing and has API task IDs to check
            // Skip if task is already complete or has an error
            if (
                isset($task['status']) && $task['status'] === 'processing' &&
                !empty($task['api_task_ids']) && is_array($task['api_task_ids'])
            ) {

                $has_errors = false;
                $task_updated = false;

                // Check if we have results for all networks
                $networks_with_results = 0;
                $total_networks = count($task['api_task_ids']);

                foreach ($task['api_task_ids'] as $network => $api_task_id) {
                    // Count networks that have results
                    if (
                        !empty($task['results'][$network]['permalink']) ||
                        (isset($task['results'][$network]['success']) && $task['results'][$network]['success'] === false)
                    ) {
                        $networks_with_results++;
                        // Check if this network has an error
                        if (isset($task['results'][$network]['success']) && $task['results'][$network]['success'] === false) {
                            $has_errors = true;
                        }
                        continue;
                    }

                    // Check the task result from the API
                    $result = $client->get_task_result($api_task_id);

                    if (is_wp_error($result)) {
                        $task['results'][$network] = array(
                            'success' => false,
                            'message' => $result->get_error_message()
                        );
                        $has_errors = true;
                        $task_updated = true;
                        YOAPSOPO_Logger::log('task_result_error', array(
                            'task_id' => $task['id'],
                            'network' => $network,
                            'error' => $result->get_error_message()
                        ));
                    } else {
                        $body = $result['body'];
                        // Handle the correct API response format
                        // API returns: {status: "complete", data: {platform: [{status, permalink, id}]}}
                        if (isset($body['status']) && ($body['status'] === 'complete' || $body['status'] === 'complete_with_error')) {
                            // The $network key might be composite like "facebook:handle" — extract base name
                            $base_network = strpos($network, ':') !== false ? explode(':', $network)[0] : $network;

                            // Look for permalink in the API data
                            $permalink = '';
                            if (isset($body['data']) && is_array($body['data'])) {
                                // Try the exact key first, then base network name
                                $platform_data = $body['data'][$network] ?? $body['data'][$base_network] ?? null;

                                if (is_array($platform_data)) {
                                    // API returns array of results per platform: [{status, permalink, id}]
                                    if (isset($platform_data[0]) && is_array($platform_data[0])) {
                                        $permalink = $platform_data[0]['permalink'] ?? '';
                                    } elseif (isset($platform_data['permalink'])) {
                                        // Direct permalink (in case format changes)
                                        $permalink = $platform_data['permalink'];
                                    }
                                }

                                // If no permalink yet, scan all platform keys for any match
                                if (empty($permalink)) {
                                    foreach ($body['data'] as $p_key => $p_data) {
                                        if (is_array($p_data) && isset($p_data[0]['permalink'])) {
                                            $permalink = $p_data[0]['permalink'];
                                            break;
                                        }
                                    }
                                }
                            }

                            // Normalize relative permalinks (e.g. Facebook returns "/reel/123/" without domain)
                            if (!empty($permalink) && strpos($permalink, '/') === 0 && strpos($permalink, '//') !== 0) {
                                $domain_map = array(
                                    'facebook' => 'https://www.facebook.com',
                                    'instagram' => 'https://www.instagram.com',
                                    'youtube' => 'https://www.youtube.com',
                                    'tiktok' => 'https://www.tiktok.com',
                                );
                                $domain = $domain_map[$base_network] ?? 'https://www.facebook.com';
                                $permalink = $domain . $permalink;
                            }

                            $task['results'][$network] = array(
                                'success' => true,
                                'permalink' => $permalink
                            );
                            $task_updated = true;
                            $networks_with_results++;
                            YOAPSOPO_Logger::log('task_completed', array(
                                'task_id' => $task['id'],
                                'network' => $network,
                                'permalink' => $permalink,
                                'raw_data' => $body['data'] ?? null
                            ));
                        } elseif (isset($body['status']) && $body['status'] === 'failed') {
                            // Extract error message
                            $base_network = strpos($network, ':') !== false ? explode(':', $network)[0] : $network;
                            $network_data = $body['data'][$network] ?? $body['data'][$base_network] ?? array();
                            $error_msg = 'Task failed';
                            if (is_array($network_data) && isset($network_data[0]['error'])) {
                                $error_msg = $network_data[0]['error'];
                            } elseif (isset($network_data['message'])) {
                                $error_msg = $network_data['message'];
                            } elseif (isset($body['message'])) {
                                $error_msg = $body['message'];
                            }
                            $task['results'][$network] = array(
                                'success' => false,
                                'message' => $error_msg
                            );
                            $has_errors = true;
                            $task_updated = true;
                            $networks_with_results++;
                            YOAPSOPO_Logger::log('task_failed', array(
                                'task_id' => $task['id'],
                                'network' => $network,
                                'error' => $error_msg
                            ));
                        } else {
                            // Task is still processing, don't increment networks_with_results
                        }
                    }
                }

                // Update task status based on results
                if ($task_updated) {
                    if (!$has_errors && $networks_with_results === $total_networks) {
                        $task['status'] = 'complete';
                        $updated = true;
                        YOAPSOPO_Logger::log('task_all_complete', array('task_id' => $task['id']));
                    } elseif ($has_errors) {
                        $task['status'] = 'error';
                        $updated = true;
                        YOAPSOPO_Logger::log('task_has_errors', array('task_id' => $task['id']));
                    }
                    // If not all complete and no errors, task remains in processing status
                }
            }
        }

        // Save updated tasks if any were processed
        if ($updated) {
            update_option($this->option_key, $tasks, false);
            YOAPSOPO_Logger::log('check_task_results_updated', array('updated_tasks' => $updated));
        }



        // Return whether any tasks were updated
        return $updated;
    }

    /**
     * Schedule a task for future execution
     *
     * @param int $task_id The task ID (not used in current implementation)
     * @param int $when Timestamp when the task should run
     */
    public function schedule_task($task_id, $when)
    {
        // In the current implementation, tasks are stored with their schedule time
        // and processed by the cron job via run_due_tasks()
        // This method exists for compatibility with the cron class
        return true;
    }

    public function ajax_get_tasks()
    {
        if (!check_ajax_referer('yoapsopo_ajax_nonce', 'nonce', false) || !current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permission denied.'), 403);
        }

        // Make sure we're checking for task results before returning tasks
        $this->check_task_results();

        $tasks = $this->get_tasks();

        // Ensure we're returning a proper array structure
        if (!is_array($tasks)) {
            $tasks = array();
        }

        // Debug: log the tasks to see what's being returned


        wp_send_json_success(array('tasks' => array_values($tasks)));
    }
}
