<?php
/**
 * Cron functionality for YoApy Social Poster
 *
 * @package YoApySocialPoster
 * @since 1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Cron jobs class for handling scheduled tasks
 *
 * @since 1.0.0
 */
class YOAPSOPO_Cron
{
    private static $instance = null;
    /**
     * Get class instance
     *
     * @return YOAPSOPO_Cron
     * @since 1.0.0
     */
    public static function get_instance()
    {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    public function __construct()
    {
        add_action('yoapsopo_tick', array($this, 'tick'));
        add_action('yoapsopo_check_task_results', array($this, 'check_task_results'));
        add_action('yoapsopo_run_task', array($this, 'run_task'));
        add_action('transition_post_status', array($this, 'on_publish'), 10, 3);
        add_action('save_post', array($this, 'maybe_dispatch_on_save'), 20, 3);
    }

    /**
     * Get maximum text length based on selected social networks
     *
     * @param array $networks Selected networks
     * @return int Maximum text length
     */
    private function get_max_text_length($networks)
    {
        // Character limits for various social networks
        $limits = array(
            'twitter' => 280,    // Twitter
            'facebook' => 63206, // Facebook
            'instagram' => 2200, // Instagram
            'linkedin' => 3000,  // LinkedIn
            'youtube' => 5000,   // YouTube
            'tiktok' => 2200     // TikTok
        );

        // Find the smallest limit among selected networks
        $min_limit = PHP_INT_MAX;
        foreach ($networks as $network) {
            $network_key = strtolower($network);
            if (isset($limits[$network_key])) {
                $min_limit = min($min_limit, $limits[$network_key]);
            }
        }

        // If no specific limits found, default to 2200 characters
        return $min_limit === PHP_INT_MAX ? 2200 : $min_limit;
    }

    public function tick()
    {
        YOAPSOPO_Logger::log('tick', array());
        $planner = YOAPSOPO_Planner::get_instance();
        $planner->run_due_tasks();
        $planner->check_task_results();
        YOAPSOPO_Logger::log('tick_end', array());
    }

    public function check_task_results()
    {
        YOAPSOPO_Logger::log('check_task_results_cron_start', array());
        YOAPSOPO_Planner::get_instance()->check_task_results();
        YOAPSOPO_Logger::log('check_task_results_cron_end', array());
    }

    public function run_task($id)
    {
        $planner = YOAPSOPO_Planner::get_instance();
        $key = 'yoapsopo_tasks';
        $tasks = get_option($key, array());
        foreach ($tasks as &$t) {
            if (($t['id'] ?? 0) == $id) {
                YOAPSOPO_Logger::log('do_job_start', array('task_id' => $id));
                $planner->run_task($t);
                YOAPSOPO_Logger::log('do_job_end', array('task_id' => $id, 'status' => $t['status']));
                break;
            }
        }
        update_option($key, $tasks, false);
    }
    public function on_publish($new, $old, $post)
    {
        // Check if this is a post and if it's being published
        if ($new !== 'publish') {
            return;
        }
        
        // Prevent duplicate processing - only process if this is a new publish
        if ($old === 'publish') {
            return;
        }
        
        // Only process public post types that support editor
        if (!is_post_type_viewable($post->post_type) || !post_type_supports($post->post_type, 'editor')) {
            return;
        }
        
        error_log("YoApy Debug: on_publish called for Post ID: {$post->ID}, Old: $old, New: $new");
        YOAPSOPO_Logger::log('on_publish', array('post_id' => $post->ID, 'old' => $old, 'new' => $new));
        $this->dispatch_from_post($post->ID);
    }
    public function maybe_dispatch_on_save($post_id, $post = null, $update = null)
    {
        // Prevent processing of autosaves and revisions
        if (wp_is_post_autosave($post_id) || wp_is_post_revision($post_id)) {
            return;
        }
        
        // Get the post status
        $status = get_post_status($post_id);
        error_log("YoApy Debug: maybe_dispatch_on_save called for Post ID: $post_id, Status: $status");

        // Only process if the post is being published
        if ($status !== 'publish') {
            return;
        }
        
        // Prevent duplicate processing - only fire once
        if (get_post_meta($post_id, '_yoapsopo_sent_once', true)) {
            error_log("YoApy Debug: Post $post_id already sent once. Skipping.");
            return;
        }
        
        YOAPSOPO_Logger::log('on_save_publish', array('post_id' => $post_id));
        $this->dispatch_from_post($post_id);
    }
    private function dispatch_from_post($post_id)
    {
        error_log("YoApy Debug: dispatch_from_post called for Post ID: $post_id");
        
        // Log all post meta for debugging
        $all_meta = get_post_meta($post_id);
        error_log("YoApy Debug: All post meta for Post ID $post_id: " . print_r($all_meta, true));
        
        $enabled = get_post_meta($post_id, '_yoapsopo_enabled', true);
        error_log("YoApy Debug: dispatch_from_post for Post ID: $post_id. Enabled: " . var_export($enabled, true));

        if ($enabled !== '1') {
            error_log("YoApy Debug: Skipping post $post_id - YoApy posting not enabled");
            YOAPSOPO_Logger::log('skip_enabled_off', array('post_id' => $post_id));
            return;
        }
        
        // Check if already sent
        if (get_post_meta($post_id, '_yoapsopo_sent_once', true)) {
            error_log("YoApy Debug: Skipping post $post_id - already sent once");
            YOAPSOPO_Logger::log('skip_already_sent', array('post_id' => $post_id));
            return;
        }

        // Get post meta with better defaults
        $type = get_post_meta($post_id, '_yoapsopo_type', true) ?: 'image';
        $nets = (array) get_post_meta($post_id, '_yoapsopo_networks', true);
        $text = get_post_meta($post_id, '_yoapsopo_text', true);

        // Auto-fill text with post content if empty, fallback to title
        if (!$text) {
            // Get the full post content
            $post_content = get_post_field('post_content', $post_id);
            // Apply the_content filter to process shortcodes, blocks, etc.
            $post_content = apply_filters('the_content', $post_content);
            // Strip HTML tags and trim whitespace
            $post_content = wp_strip_all_tags($post_content);
            // If we have content, use it; otherwise fallback to title
            if (!empty(trim($post_content))) {
                $text = $post_content;
            } else {
                $text = get_the_title($post_id);
            }
        }

        // Limit text length based on social network requirements
        if (!empty($nets) && is_array($nets)) {
            // Get the maximum length for the selected networks
            $max_length = $this->get_max_text_length($nets);
            if (mb_strlen($text) > $max_length) {
                $text = mb_substr($text, 0, $max_length - 3) . '...';
            }
        }

        // Auto-fill image with featured image if empty
        $image = get_post_meta($post_id, '_yoapsopo_image', true);
        if (!$image) {
            $thumb = get_the_post_thumbnail_url($post_id, 'large');
            if ($thumb) {
                $image = $thumb;
            } else {
                // Fallback to first image in post content
                $content = get_post_field('post_content', $post_id);
                $output = preg_match_all('/<img.+src=[\'\"]([^\'\"]+)[\'\"].*>/i', $content, $matches);
                if (!empty($matches[1][0])) {
                    $image = $matches[1][0];
                }
            }
        }

        // Get video URL
        $video = get_post_meta($post_id, '_yoapsopo_video', true);

        // For video posts, require a video URL
        if ($type === 'video' && empty($video)) {
            YOAPSOPO_Logger::log('skip_video_required', array('post_id' => $post_id));
            return;
        }

        // Check if media is required for selected networks
        $requires_media = false;
        if (!empty($nets) && is_array($nets)) {
            // Networks that require media (image or video)
            $media_required_networks = array('instagram', 'tiktok');
            foreach ($nets as $net) {
                if (in_array(strtolower($net), $media_required_networks)) {
                    $requires_media = true;
                    break;
                }
            }
        }

        // If media is required but neither image nor video is available, skip posting
        if ($requires_media && empty($image) && empty($video)) {
            YOAPSOPO_Logger::log('skip_media_required', array('post_id' => $post_id, 'networks' => $nets));
            return;
        }

        // Get TikTok specific metadata
        $tiktok_privacy = get_post_meta($post_id, '_yoapsopo_tiktok_privacy', true);
        $tiktok_allow_comment = get_post_meta($post_id, '_yoapsopo_tiktok_allow_comment', true);
        $tiktok_allow_duet = get_post_meta($post_id, '_yoapsopo_tiktok_allow_duet', true);
        $tiktok_allow_stitch = get_post_meta($post_id, '_yoapsopo_tiktok_allow_stitch', true);
        $tiktok_commercial_toggle = get_post_meta($post_id, '_yoapsopo_tiktok_commercial_toggle', true);
        $tiktok_commercial_your_brand = get_post_meta($post_id, '_yoapsopo_tiktok_commercial_your_brand', true);
        $tiktok_commercial_branded = get_post_meta($post_id, '_yoapsopo_tiktok_commercial_branded', true);
        $tiktok_consent = get_post_meta($post_id, '_yoapsopo_tiktok_consent', true);

        // Auto-fill article URL with post permalink if empty
        $article = get_post_meta($post_id, '_yoapsopo_article', true);
        if (!$article) {
            $article = get_permalink($post_id);
        }

        // Get schedule time (post immediately if not set)
        $when = intval(get_post_meta($post_id, '_yoapsopo_when', true));

        // Get existing tasks to generate next ID
        $tasks = get_option('yoapsopo_tasks', array());
        $next_id = 1;
        if (!empty($tasks)) {
            $ids = wp_list_pluck($tasks, 'id');
            $next_id = max($ids) + 1;
        }

        $task = array(
            'id' => $next_id,
            'title' => get_the_title($post_id),
            'networks' => $nets,
            'type' => $type,
            'text' => $text,
            'image_url' => $image,
            'video_url' => $video,
            'article_url' => $article,
            'when' => $when ?: null,
            'status' => (!empty($when) && $when > time()) ? 'scheduled' : 'pending',
            'api_task_ids' => array(),
            'results' => array()
        );

        // Add TikTok specific metadata to task if TikTok is selected
        if (in_array('tiktok', $nets)) {
            $task['tiktok_privacy'] = $tiktok_privacy;
            $task['tiktok_allow_comment'] = $tiktok_allow_comment;
            $task['tiktok_allow_duet'] = $tiktok_allow_duet;
            $task['tiktok_allow_stitch'] = $tiktok_allow_stitch;
            $task['tiktok_commercial_toggle'] = $tiktok_commercial_toggle;
            $task['tiktok_commercial_your_brand'] = $tiktok_commercial_your_brand;
            $task['tiktok_commercial_branded'] = $tiktok_commercial_branded;
        }

        // Log task creation
        error_log("YoApy Debug: Creating task for post $post_id with title: " . $task['title']);
        error_log("YoApy Debug: Task networks: " . print_r($task['networks'], true));
        error_log("YoApy Debug: Task type: " . $task['type']);
        error_log("YoApy Debug: Task text: " . substr($task['text'], 0, 100) . '...');
        
        // Add task to planner
        $tasks[] = $task;
        $result = update_option('yoapsopo_tasks', $tasks, false);
        
        // Process immediately if no schedule or past schedule
        if (!$when || $when <= time()) {
            YOAPSOPO_Logger::log('do_job_start', array('task_id' => $next_id));
            $planner = YOAPSOPO_Planner::get_instance();
            $planner->run_task($tasks[count($tasks) - 1]); // Pass reference to the task we just added
            YOAPSOPO_Logger::log('do_job_end', array('task_id' => $next_id, 'status' => $tasks[count($tasks) - 1]['status']));
            
            // Save the updated task with its new status
            $result = update_option('yoapsopo_tasks', $tasks, false);
        }
        error_log("YoApy Debug: Task added to planner for post $post_id, update result: " . var_export($result, true));
        error_log("YoApy Debug: Task details: " . print_r($task, true));
        error_log("YoApy Debug: Total tasks in option: " . count($tasks));
        
        // Mark as sent to prevent duplicate processing only after successful task creation
        update_post_meta($post_id, '_yoapsopo_sent_once', '1');
        error_log("YoApy Debug: Marked post $post_id as sent");
    }

    /**
     * Legacy instance method for backward compatibility
     *
     * @deprecated 1.6.0 Use YOAPSOPO_Cron::get_instance() instead.
     * @return YOAPSOPO_Cron
     */
    public static function instance()
    {
        return self::get_instance();
    }
}
