<?php
/**
 * API Client for YoApy Social Poster
 *
 * @package YoApySocialPoster
 * @since 1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * YoApy API Client class
 *
 * @since 1.0.0
 */
class YOAPSOPO_Client
{
    private $base_url;
    private $key_id;
    private $secret_hex;
    private $account_default;
    private $per_network;
    public function __construct()
    {
        $opt = get_option('yoapsopo_settings', array());
        $this->base_url = isset($opt['base_url']) ? rtrim($opt['base_url'], '/') : 'https://api.yoapy.com';
        $this->key_id = $opt['key_id'] ?? '';
        $this->secret_hex = preg_replace('/[^0-9a-f]/i', '', $opt['secret'] ?? '');
        $this->account_default = ltrim($opt['account'] ?? '', '@');
        $this->per_network = array(
            'facebook' => ltrim($opt['account_facebook'] ?? '', '@'),
            'instagram' => ltrim($opt['account_instagram'] ?? '', '@'),
            'youtube' => ltrim($opt['account_youtube'] ?? '', '@'),
            'tiktok' => ltrim($opt['account_tiktok'] ?? '', '@'),
        );
        YOAPSOPO_Logger::log('client_init', array('base_url' => $this->base_url, 'account_selector' => $this->account_default));
    }
    public static function has_keys()
    {
        $opt = get_option('yoapsopo_settings', array());
        return !empty($opt['key_id']) && !empty($opt['secret']);
    }
    private function account_for($network)
    {
        $h = trim($this->per_network[$network] ?? '');
        return $h !== '' ? $h : $this->account_default;
    }
    private function hmac_headers($method, $path, $body_raw)
    {
        $ts = (string) time();
        $nonce = bin2hex(random_bytes(12));
        $hash = hash('sha256', $body_raw ?? '');
        $canonical = implode("\n", array($method, $path, $ts, $nonce, $hash));
        $key = pack("H*", $this->secret_hex);
        $sig = base64_encode(hash_hmac('sha256', $canonical, $key, true));
        YOAPSOPO_Logger::log('hmac_build', array('method' => $method, 'path' => $path, 'ts' => $ts, 'nonce' => $nonce, 'body_sha256' => $hash));
        return array('X-Key-Id' => $this->key_id, 'X-Timestamp' => $ts, 'X-Nonce' => $nonce, 'X-Signature' => $sig);
    }
    public function ping()
    {
        $path = '/v1/auth_ping';
        $url = $this->base_url . $path;
        $res = wp_remote_get($url, array('headers' => $this->hmac_headers('GET', $path, ''), 'timeout' => 20));
        if (is_wp_error($res))
            return $res;
        return array('http_code' => wp_remote_retrieve_response_code($res), 'body' => wp_remote_retrieve_body($res));
    }

    /**
     * Get TikTok creator info
     *
     * @param string $account TikTok account handle
     * @return array|WP_Error API response or WP_Error on failure
     */
    public function get_tiktok_creator_info($account = '')
    {
        // Use provided account or default TikTok account
        if (empty($account)) {
            $account = $this->account_for('tiktok');
        }

        if (empty($account)) {
            return new WP_Error('missing_account', 'TikTok account is required');
        }

        $path = '/v1/tiktok/creator_info?account=' . rawurlencode($account);
        $url = $this->base_url . $path;

        $res = wp_remote_get($url, array(
            'headers' => $this->hmac_headers('GET', $path, ''),
            'timeout' => 20
        ));

        if (is_wp_error($res)) {
            return $res;
        }

        $code = wp_remote_retrieve_response_code($res);
        $body = wp_remote_retrieve_body($res);

        YOAPSOPO_Logger::log('tiktok_creator_info', array(
            'http_code' => $code,
            'response_body' => $body,
            'account' => $account
        ));

        return array(
            'code' => $code,
            'body' => json_decode($body, true),
            'raw' => $body
        );
    }

    /**
     * Get Facebook profile info
     *
     * @param string $account Facebook account handle
     * @return array|WP_Error API response or WP_Error on failure
     */
    public function get_facebook_profile_info($account = '')
    {
        // Use provided account or default Facebook account
        if (empty($account)) {
            $account = $this->account_for('facebook');
        }

        if (empty($account)) {
            return new WP_Error('missing_account', 'Facebook account is required');
        }

        $path = '/v1/facebook/profile_info?account=' . rawurlencode($account);
        $url = $this->base_url . $path;

        $res = wp_remote_get($url, array(
            'headers' => $this->hmac_headers('GET', $path, ''),
            'timeout' => 20
        ));

        if (is_wp_error($res)) {
            return $res;
        }

        $code = wp_remote_retrieve_response_code($res);
        $body = wp_remote_retrieve_body($res);

        YOAPSOPO_Logger::log('facebook_profile_info', array(
            'http_code' => $code,
            'response_body' => $body,
            'account' => $account
        ));

        return array(
            'code' => $code,
            'body' => json_decode($body, true),
            'raw' => $body
        );
    }

    /**
     * Get Instagram profile info
     *
     * @param string $account Instagram account handle
     * @return array|WP_Error API response or WP_Error on failure
     */
    public function get_instagram_profile_info($account = '')
    {
        // Use provided account or default Instagram account
        if (empty($account)) {
            $account = $this->account_for('instagram');
        }

        if (empty($account)) {
            return new WP_Error('missing_account', 'Instagram account is required');
        }

        $path = '/v1/instagram/profile_info?account=' . rawurlencode($account);
        $url = $this->base_url . $path;

        $res = wp_remote_get($url, array(
            'headers' => $this->hmac_headers('GET', $path, ''),
            'timeout' => 20
        ));

        if (is_wp_error($res)) {
            return $res;
        }

        $code = wp_remote_retrieve_response_code($res);
        $body = wp_remote_retrieve_body($res);

        YOAPSOPO_Logger::log('instagram_profile_info', array(
            'http_code' => $code,
            'response_body' => $body,
            'account' => $account
        ));

        return array(
            'code' => $code,
            'body' => json_decode($body, true),
            'raw' => $body
        );
    }

    /**
     * Get YouTube profile info
     *
     * @param string $account YouTube account handle
     * @return array|WP_Error API response or WP_Error on failure
     */
    public function get_youtube_profile_info($account = '')
    {
        // Use provided account or default YouTube account
        if (empty($account)) {
            $account = $this->account_for('youtube');
        }

        if (empty($account)) {
            return new WP_Error('missing_account', 'YouTube account is required');
        }

        $path = '/v1/youtube/profile_info?account=' . rawurlencode($account);
        $url = $this->base_url . $path;

        $res = wp_remote_get($url, array(
            'headers' => $this->hmac_headers('GET', $path, ''),
            'timeout' => 20
        ));

        if (is_wp_error($res)) {
            return $res;
        }

        $code = wp_remote_retrieve_response_code($res);
        $body = wp_remote_retrieve_body($res);

        YOAPSOPO_Logger::log('youtube_profile_info', array(
            'http_code' => $code,
            'response_body' => $body,
            'account' => $account
        ));

        return array(
            'code' => $code,
            'body' => json_decode($body, true),
            'raw' => $body
        );
    }

    /**
     * Get all client accounts with profile pictures
     *
     * @return array|WP_Error API response or WP_Error on failure
     */
    public function get_client_accounts()
    {
        $path = '/v1/get-client-accounts-by-key?key_id=' . rawurlencode($this->key_id) . '&secret_plain=' . rawurlencode($this->secret_hex);
        $url = $this->base_url . $path;

        $res = wp_remote_get($url, array(
            'timeout' => 20
        ));

        if (is_wp_error($res)) {
            return $res;
        }

        $code = wp_remote_retrieve_response_code($res);
        $body = wp_remote_retrieve_body($res);

        YOAPSOPO_Logger::log('get_client_accounts', array(
            'http_code' => $code,
            'response_body' => $body
        ));

        return array(
            'code' => $code,
            'body' => json_decode($body, true),
            'raw' => $body
        );
    }

    public function create_post_json($network, $type, $text, $image_url, $video_url, $article_url = '', $scheduled_iso = '', $tiktok_fields = array(), $account_handle = '', $connection_id = '')
    {
        $path = '/v1/posts';
        $url = $this->base_url . $path;

        // If connection_id is provided (numeric), use it directly — bypasses name resolution
        if ($connection_id !== '' && is_numeric($connection_id)) {
            $payload = array('account' => '*', 'account_ids' => array((string) $connection_id), 'post_type' => $type);
        } else {
            // Fallback: use handle + network name (old behavior)
            $account = $account_handle !== '' ? $account_handle : $this->account_for($network);
            $payload = array('account' => $account, 'account_ids' => array($network), 'post_type' => $type);
        }
        if ($text !== '')
            $payload['text'] = $text;
        if ($image_url !== '' && in_array($type, array('image', 'story')))
            $payload['media_urls'] = array($image_url);
        if ($video_url !== '' && in_array($type, array('video', 'reels', 'live_schedule')))
            $payload['media_urls'] = array($video_url);
        if ($article_url !== '')
            $payload['article_url'] = $article_url;
        if ($scheduled_iso !== '')
            $payload['scheduled_time'] = $scheduled_iso;

        // Add TikTok specific fields if this is a TikTok post
        if ($network === 'tiktok' && !empty($tiktok_fields)) {
            // Add privacy settings
            if (!empty($tiktok_fields['tiktok_privacy'])) {
                $payload['tiktok_privacy'] = $tiktok_fields['tiktok_privacy'];
            }

            // Add interaction settings
            if (isset($tiktok_fields['tiktok_allow_comment'])) {
                $payload['tiktok_allow_comment'] = $tiktok_fields['tiktok_allow_comment'] === '1' || $tiktok_fields['tiktok_allow_comment'] === true;
            }
            if (isset($tiktok_fields['tiktok_allow_duet'])) {
                $payload['tiktok_allow_duet'] = $tiktok_fields['tiktok_allow_duet'] === '1' || $tiktok_fields['tiktok_allow_duet'] === true;
            }
            if (isset($tiktok_fields['tiktok_allow_stitch'])) {
                $payload['tiktok_allow_stitch'] = $tiktok_fields['tiktok_allow_stitch'] === '1' || $tiktok_fields['tiktok_allow_stitch'] === true;
            }

            // Add commercial content settings
            if (isset($tiktok_fields['tiktok_commercial_toggle'])) {
                $payload['tiktok_commercial_toggle'] = $tiktok_fields['tiktok_commercial_toggle'] === '1' || $tiktok_fields['tiktok_commercial_toggle'] === true;

                if ($payload['tiktok_commercial_toggle']) {
                    if (isset($tiktok_fields['tiktok_commercial_your_brand'])) {
                        $payload['tiktok_commercial_your_brand'] = $tiktok_fields['tiktok_commercial_your_brand'] === '1' || $tiktok_fields['tiktok_commercial_your_brand'] === true;
                    }
                    if (isset($tiktok_fields['tiktok_commercial_branded'])) {
                        $payload['tiktok_commercial_branded'] = $tiktok_fields['tiktok_commercial_branded'] === '1' || $tiktok_fields['tiktok_commercial_branded'] === true;
                    }
                }
            }
        }

        $raw = wp_json_encode($payload);
        $headers = array_merge(array('Content-Type' => 'application/json'), $this->hmac_headers('POST', $path, $raw));
        YOAPSOPO_Logger::log('dispatch', array('payload' => $payload));
        YOAPSOPO_Logger::log('req_create_post_json', array('url' => $url, 'endpoint' => $path, 'headers' => $headers, 'body_raw' => $raw, 'json' => $payload));
        $res = wp_remote_post($url, array('headers' => $headers, 'body' => $raw, 'timeout' => 60));
        if (is_wp_error($res))
            return $res;
        $code = wp_remote_retrieve_response_code($res);
        $body = wp_remote_retrieve_body($res);
        YOAPSOPO_Logger::log('res_create_post_json', array('http_code' => $code, 'response_body' => $body));
        return array('code' => $code, 'body' => json_decode($body, true), 'raw' => $body);
    }
    public function get_task_result($task_id)
    {
        $path = '/v1/get_task_result?task_id=' . rawurlencode($task_id);
        $url = $this->base_url . $path;
        $headers = $this->hmac_headers('GET', '/v1/get_task_result', '');
        $res = wp_remote_get($url, array('headers' => $headers, 'timeout' => 20));
        if (is_wp_error($res))
            return $res;
        $code = wp_remote_retrieve_response_code($res);
        $body = wp_remote_retrieve_body($res);
        // YOAPSOPO_Logger::log('res_get_task_result', array('http_code' => $code, 'response_body' => $body));
        return array('code' => $code, 'body' => json_decode($body, true), 'raw' => $body);
    }

    /**
     * Legacy instance method for backward compatibility
     *
     * @deprecated 1.6.0 Use new YOAPSOPO_Client() instead.
     * @return YOAPSOPO_Client
     */
    public static function instance()
    {
        return new self();
    }
}