<?php
/**
 * Admin functionality for YoApy Social Poster
 *
 * @package YoApySocialPoster
 * @since 1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * YoApy Admin class
 *
 * @since 1.0.0
 */
class YOAPSOPO_Admin
{
    /**
     * Plugin instance
     *
     * @var YOAPSOPO_Admin
     * @since 1.0.0
     */
    private static $instance = null;

    /**
     * Get plugin instance
     *
     * @return YOAPSOPO_Admin
     * @since 1.0.0
     */
    public static function get_instance()
    {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     *
     * @since 1.0.0
     */
    private function __construct()
    {
        add_action('add_meta_boxes', array($this, 'add_meta_box'));
        add_action('save_post', array($this, 'save_post_meta'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('wp_ajax_yoapsopo_fetch_tiktok_creator_info', array($this, 'ajax_fetch_tiktok_creator_info'));
        add_action('wp_ajax_yoapsopo_fetch_creator_info', array($this, 'ajax_fetch_creator_info'));
        add_action('wp_ajax_yoapsopo_fetch_importable_accounts', array($this, 'ajax_fetch_importable_accounts'));
        add_action('wp_ajax_yoapsopo_save_imported_accounts', array($this, 'ajax_save_imported_accounts'));
        add_action('wp_ajax_yoapsopo_delete_account', array($this, 'ajax_delete_account'));
        add_action('wp_ajax_yoapsopo_delete_all_accounts', array($this, 'ajax_delete_all_accounts'));
        add_action('wp_ajax_yoapsopo_save_settings', array($this, 'ajax_save_settings'));
        add_action('wp_ajax_yoapsopo_test_connection', array($this, 'ajax_test_connection'));
        add_action('wp_ajax_yoapsopo_parse_post_content', array($this, 'ajax_parse_post_content'));
    }

    /**
     * Add meta box to post editor
     *
     * @param string $post_type Post type.
     * @since 1.0.0
     */
    public function add_meta_box($post_type)
    {
        // Only add meta box to public post types that support editor
        if (!is_post_type_viewable($post_type) || !post_type_supports($post_type, 'editor')) {
            return;
        }

        add_meta_box(
            'yoapsopo_metabox',
            __('YoApy SP', 'yoapy-social-poster'),
            array($this, 'metabox_callback'),
            $post_type,
            'side',
            'default'
        );
    }

    /**
     * Render metabox content
     *
     * @param WP_Post $post Post object.
     * @since 1.0.0
     */
    public function metabox_callback($post)
    {
        // Add nonce field for security
        wp_nonce_field('yoapsopo_metabox', 'yoapsopo_nonce');

        // Get meta values
        $enabled = get_post_meta($post->ID, '_yoapsopo_enabled', true);
        $type = get_post_meta($post->ID, '_yoapsopo_type', true);
        $networks = get_post_meta($post->ID, '_yoapsopo_networks', true);
        $text = get_post_meta($post->ID, '_yoapsopo_text', true);
        $image = get_post_meta($post->ID, '_yoapsopo_image', true);
        $video = get_post_meta($post->ID, '_yoapsopo_video', true);
        $article = get_post_meta($post->ID, '_yoapsopo_article', true);
        $when = get_post_meta($post->ID, '_yoapsopo_when', true);

        // Set defaults
        if (!$type) {
            $type = 'image';
        }
        if (!$networks || !is_array($networks)) {
            $networks = array();
        }

        // Auto-populate text with post content if not set, fallback to title
        if (empty($text)) {
            // Get the full post content
            $post_content = get_post_field('post_content', $post->ID);

            // Process content only if it's not empty and not just "Auto Draft"
            if (!empty($post_content) && trim($post_content) !== '' && trim($post_content) !== 'Auto Draft') {
                // Apply the_content filter to process shortcodes, blocks, etc.
                $processed_content = apply_filters('the_content', $post_content);
                // Strip HTML tags and trim whitespace
                $processed_content = wp_strip_all_tags($processed_content);

                // Use processed content if it's not empty
                if (!empty(trim($processed_content))) {
                    $text = $processed_content;
                } else {
                    // Fallback to raw content without HTML tags
                    $text = wp_strip_all_tags($post_content);
                }
            } else {
                // If no content or just "Auto Draft", use post title
                $post_title = get_the_title($post->ID);

                // If title is also empty or is "Auto Draft", try to get a better title
                if (empty($post_title) || $post_title === 'Auto Draft') {
                    $post_title = get_post_field('post_title', $post->ID);
                }

                // If we still have no title, use a default placeholder
                if (empty($post_title)) {
                    $post_title = __('(No title)', 'yoapy-social-poster');
                }

                $text = $post_title;
            }
        }

        // TikTok specific fields
        $tiktok_privacy = get_post_meta($post->ID, '_yoapsopo_tiktok_privacy', true);
        $tiktok_allow_comment = get_post_meta($post->ID, '_yoapsopo_tiktok_allow_comment', true);
        $tiktok_allow_duet = get_post_meta($post->ID, '_yoapsopo_tiktok_allow_duet', true);
        $tiktok_allow_stitch = get_post_meta($post->ID, '_yoapsopo_tiktok_allow_stitch', true);
        $tiktok_commercial_toggle = get_post_meta($post->ID, '_yoapsopo_tiktok_commercial_toggle', true);
        $tiktok_commercial_your_brand = get_post_meta($post->ID, '_yoapsopo_tiktok_commercial_your_brand', true);
        $tiktok_commercial_branded = get_post_meta($post->ID, '_yoapsopo_tiktok_commercial_branded', true);
        $tiktok_consent = get_post_meta($post->ID, '_yoapsopo_tiktok_consent', true);

        // Get TikTok creator info
        $tiktok_creator_info = get_post_meta($post->ID, '_yoapsopo_tiktok_creator_info', true);
        if (empty($tiktok_creator_info)) {
            $tiktok_creator_info = array();
        }

        // Auto-populate article URL with post permalink if not set
        if (empty($article)) {
            $article = get_permalink($post->ID);
        }

        // Convert timestamp to local datetime for input
        $when_local = '';
        if ($when) {
            $when_local = get_date_from_gmt(gmdate('Y-m-d H:i:s', $when), 'Y-m-d\TH:i');
        }

        // Enqueue required scripts and styles
        wp_enqueue_media();
        wp_enqueue_script(
            'yoapsopo-admin',
            YOAPSOPO_PLUGIN_URL . 'admin/js/yoapsopo-admin.js',
            array('jquery'),
            YOAPSOPO_VERSION,
            true
        );
        wp_enqueue_script(
            'yoapsopo-metabox-script',
            YOAPSOPO_PLUGIN_URL . 'admin/js/yoapsopo-metabox-script.js',
            array('jquery', 'yoapsopo-admin'),
            YOAPSOPO_VERSION,
            true
        );
        wp_localize_script(
            'yoapsopo-admin',
            'YOAPSOPO',
            array(
                'ajax' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('yoapsopo_metabox'),
                'hasKeys' => YOAPSOPO_Client::has_keys(),
                'i18n' => array(
                    'chooseMedia' => __('Choose Media', 'yoapy-social-poster'),
                    'title' => __('Title', 'yoapy-social-poster'),
                    'networks' => __('Networks', 'yoapy-social-poster'),
                    'type' => __('Type', 'yoapy-social-poster'),
                    'when' => __('When', 'yoapy-social-poster'),
                    'status' => __('Status', 'yoapy-social-poster'),
                    'actions' => __('Actions', 'yoapy-social-poster'),
                    'send' => __('Send', 'yoapy-social-poster'),
                    'refreshStatus' => __('Refresh status', 'yoapy-social-poster'),
                    'delete' => __('Delete', 'yoapy-social-poster'),
                    'videoRequired' => __('Video URL is required for video posts.', 'yoapy-social-poster'),
                    'mediaRequired' => __('Media (image or video) is required for Instagram and TikTok posts.', 'yoapy-social-poster'),
                    'tiktokPrivacyStatus' => __('Privacy Status', 'yoapy-social-poster'),
                    'tiktokSelectPrivacy' => __('Select Privacy Status', 'yoapy-social-poster'),
                    'tiktokPrivacyOnlyMe' => __('Only Me', 'yoapy-social-poster'),
                    'tiktokPrivacyFriends' => __('Friends', 'yoapy-social-poster'),
                    'tiktokPrivacyMutualFriends' => __('Mutual Friends', 'yoapy-social-poster'),
                    'tiktokPrivacyEveryone' => __('Everyone', 'yoapy-social-poster'),
                    'tiktokPrivacyNote' => __('Select who can view your TikTok post. For branded content, visibility must be set to Everyone.', 'yoapy-social-poster'),
                    'tiktokInteractionSettings' => __('Interaction Settings', 'yoapy-social-poster'),
                    'tiktokAllowComments' => __('Allow Comments', 'yoapy-social-poster'),
                    'tiktokAllowDuet' => __('Allow Duet', 'yoapy-social-poster'),
                    'tiktokAllowStitch' => __('Allow Stitch', 'yoapy-social-poster'),
                    'tiktokCommercialContent' => __('Commercial Content', 'yoapy-social-poster'),
                    'tiktokPromotesBrand' => __('This content promotes a brand or product', 'yoapy-social-poster'),
                    'tiktokYourBrand' => __('Your Brand (Promoting yourself or your own business)', 'yoapy-social-poster'),
                    'tiktokBrandedContent' => __('Branded Content (Promoting another brand or third party)', 'yoapy-social-poster'),
                    'tiktokPromotionalLabel' => __('Your photo/video will be labeled as "Promotional content"', 'yoapy-social-poster'),
                    'tiktokPaidPartnershipLabel' => __('Your photo/video will be labeled as "Paid partnership"', 'yoapy-social-poster'),
                    'tiktokCommercialNote' => __('You need to indicate if your content promotes yourself, a third party, or both.', 'yoapy-social-poster'),
                    'tiktokBrandedContentWarning' => __('Branded content visibility cannot be set to private. Privacy setting will be automatically changed to "Everyone".', 'yoapy-social-poster'),
                    'tiktokBrandedContentError' => __('Branded content visibility cannot be set to private. Please select "Everyone" for branded content.', 'yoapy-social-poster'),
                    'tiktokConsentMusic' => __('By posting, you agree to TikTok\'s Music Usage Confirmation', 'yoapy-social-poster'),
                    'tiktokConsentBranded' => __('By posting, you agree to TikTok\'s Branded Content Policy and Music Usage Confirmation', 'yoapy-social-poster'),
                    'tiktokSelectPrivacyError' => __('Please select a privacy status for your TikTok post.', 'yoapy-social-poster'),
                    'tiktokCommercialSelectionError' => __('You need to indicate if your content promotes yourself, a third party, or both.', 'yoapy-social-poster'),
                    'tiktokConsentRequired' => __('You must agree to TikTok\'s terms before posting.', 'yoapy-social-poster'),
                    'tiktokPostSettings' => __('TikTok Post Settings', 'yoapy-social-poster'),
                    'tiktokFetchingCreatorInfo' => __('Fetching creator info...', 'yoapy-social-poster'),
                    'tiktokCreatorInfoError' => __('Failed to fetch creator info', 'yoapy-social-poster'),
                ),
            )
        );
        wp_enqueue_style(
            'yoapsopo-admin',
            YOAPSOPO_PLUGIN_URL . 'admin/css/yoapsopo-admin.css',
            array(),
            YOAPSOPO_VERSION
        );

        // Pass variables to metabox template
        $has_keys = YOAPSOPO_Client::has_keys();

        // Include metabox template with TikTok variables
        include YOAPSOPO_PLUGIN_DIR . 'admin/views/metabox.php';
    }

    /**
     * Save post meta data
     *
     * @param int $post_id Post ID.
     * @since 1.0.0
     */
    public function save_post_meta($post_id)
    {
        // Security checks
        if (!isset($_POST['yoapsopo_nonce'])) {
            return;
        }

        // Unslash + sanitize nonce before verifying (fixes MissingUnslash/InputNotSanitized)
        $nonce = sanitize_text_field(wp_unslash($_POST['yoapsopo_nonce']));
        if (!wp_verify_nonce($nonce, 'yoapsopo_metabox')) {
            return;
        }

        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }

        if (wp_is_post_autosave($post_id) || wp_is_post_revision($post_id)) {
            return;
        }

        if (!current_user_can('edit_post', $post_id)) {
            return;
        }

        // Save meta fields (always unslash first, then sanitize)

        // Checkbox
        $enabled = isset($_POST['yoapsopo_enabled']) ? '1' : '';
        update_post_meta($post_id, '_yoapsopo_enabled', $enabled);

        // Type
        $type = isset($_POST['yoapsopo_type']) ? sanitize_text_field(wp_unslash($_POST['yoapsopo_type'])) : 'image';
        update_post_meta($post_id, '_yoapsopo_type', $type);

        // Networks (array) — sanitize on the same line to satisfy the sniff
        $networks_sanitized = isset($_POST['yoapsopo_networks'])
            ? array_map('sanitize_text_field', (array) wp_unslash($_POST['yoapsopo_networks']))
            : array();
        update_post_meta($post_id, '_yoapsopo_networks', $networks_sanitized);

        // Text (allow safe HTML)
        $text = isset($_POST['yoapsopo_text']) ? wp_kses_post(wp_unslash($_POST['yoapsopo_text'])) : '';
        update_post_meta($post_id, '_yoapsopo_text', $text);

        // URLs
        $image = isset($_POST['yoapsopo_image']) ? esc_url_raw(wp_unslash($_POST['yoapsopo_image'])) : '';
        update_post_meta($post_id, '_yoapsopo_image', $image);

        $video = isset($_POST['yoapsopo_video']) ? esc_url_raw(wp_unslash($_POST['yoapsopo_video'])) : '';
        update_post_meta($post_id, '_yoapsopo_video', $video);

        $article = isset($_POST['yoapsopo_article']) ? esc_url_raw(wp_unslash($_POST['yoapsopo_article'])) : '';
        update_post_meta($post_id, '_yoapsopo_article', $article);

        // Validate video field for video posts
        $type = isset($_POST['yoapsopo_type']) ? sanitize_text_field(wp_unslash($_POST['yoapsopo_type'])) : 'image';
        if ($type === 'video' && empty($video)) {
            // Add an error message
            add_action('admin_notices', function () {
                echo '<div class="notice notice-error"><p>' . esc_html__('Video URL is required for video posts.', 'yoapy-social-poster') . '</p></div>';
            });
            // Prevent publishing by changing post status to draft
            remove_action('save_post', array($this, 'save_post_meta'));
            wp_update_post(array('ID' => $post_id, 'post_status' => 'draft'));
            add_action('save_post', array($this, 'save_post_meta'));
            return;
        }

        // Check if YoApy posting is enabled
        $enabled = isset($_POST['yoapsopo_enabled']) ? '1' : '';

        // If YoApy posting is enabled, validate media requirements
        if ($enabled === '1') {
            // Get selected networks
            $networks = isset($_POST['yoapsopo_networks']) ? array_map('sanitize_text_field', (array) wp_unslash($_POST['yoapsopo_networks'])) : array();

            // Check if media is required for selected networks
            $requires_media = false;
            $media_required_networks = array('instagram', 'tiktok');
            foreach ($networks as $network) {
                if (in_array(strtolower($network), $media_required_networks)) {
                    $requires_media = true;
                    break;
                }
            }

            // If media is required but neither image nor video is available, show error
            if ($requires_media && empty($image) && empty($video)) {
                // Add an error message
                add_action('admin_notices', function () {
                    echo '<div class="notice notice-error"><p>' . esc_html__('Media (image or video) is required for Instagram and TikTok posts.', 'yoapy-social-poster') . '</p></div>';
                });
                // Prevent publishing by changing post status to draft
                remove_action('save_post', array($this, 'save_post_meta'));
                wp_update_post(array('ID' => $post_id, 'post_status' => 'draft'));
                add_action('save_post', array($this, 'save_post_meta'));
                return;
            }
        }

        // Schedule time
        $when_input = isset($_POST['yoapsopo_when']) ? sanitize_text_field(wp_unslash($_POST['yoapsopo_when'])) : '';
        $when_input = trim($when_input);
        $when_ts = null;

        if ('' !== $when_input) {
            try {
                $tz = wp_timezone();
                // datetime-local esperado "Y-m-d\TH:i"
                $dt = new DateTimeImmutable($when_input, $tz);
                $when_ts = $dt->getTimestamp();
            } catch (Exception $e) {
                $when_ts = null;
            }
        }

        if (null !== $when_ts) {
            update_post_meta($post_id, '_yoapsopo_when', $when_ts);
        } else {
            delete_post_meta($post_id, '_yoapsopo_when');
        }

        // TikTok specific fields
        $tiktok_privacy = isset($_POST['yoapsopo_tiktok_privacy']) ? sanitize_text_field(wp_unslash($_POST['yoapsopo_tiktok_privacy'])) : '';
        update_post_meta($post_id, '_yoapsopo_tiktok_privacy', $tiktok_privacy);

        $tiktok_allow_comment = isset($_POST['yoapsopo_tiktok_allow_comment']) ? '1' : '0';
        update_post_meta($post_id, '_yoapsopo_tiktok_allow_comment', $tiktok_allow_comment);

        $tiktok_allow_duet = isset($_POST['yoapsopo_tiktok_allow_duet']) ? '1' : '0';
        update_post_meta($post_id, '_yoapsopo_tiktok_allow_duet', $tiktok_allow_duet);

        $tiktok_allow_stitch = isset($_POST['yoapsopo_tiktok_allow_stitch']) ? '1' : '0';
        update_post_meta($post_id, '_yoapsopo_tiktok_allow_stitch', $tiktok_allow_stitch);

        $tiktok_commercial_toggle = isset($_POST['yoapsopo_tiktok_commercial_toggle']) ? '1' : '0';
        update_post_meta($post_id, '_yoapsopo_tiktok_commercial_toggle', $tiktok_commercial_toggle);

        $tiktok_commercial_your_brand = isset($_POST['yoapsopo_tiktok_commercial_your_brand']) ? '1' : '0';
        update_post_meta($post_id, '_yoapsopo_tiktok_commercial_your_brand', $tiktok_commercial_your_brand);

        $tiktok_commercial_branded = isset($_POST['yoapsopo_tiktok_commercial_branded']) ? '1' : '0';
        update_post_meta($post_id, '_yoapsopo_tiktok_commercial_branded', $tiktok_commercial_branded);

        $tiktok_consent = isset($_POST['yoapsopo_tiktok_consent']) ? '1' : '0';
        update_post_meta($post_id, '_yoapsopo_tiktok_consent', $tiktok_consent);
    }

    /**
     * Enqueue admin scripts and styles
     *
     * @param string $hook_suffix The current admin page.
     * @since 1.0.0
     */
    public function enqueue_scripts($hook_suffix)
    {
        // Only enqueue on post edit screens
        if (!in_array($hook_suffix, array('post.php', 'post-new.php'))) {
            return;
        }

        wp_enqueue_style(
            'yoapsopo-admin-global',
            YOAPSOPO_PLUGIN_URL . 'admin/css/yoapsopo-admin.css',
            array(),
            YOAPSOPO_VERSION
        );
    }

    /**
     * AJAX handler to fetch TikTok creator info
     *
     * @since 1.6.19
     */
    public function ajax_fetch_tiktok_creator_info()
    {
        // Check nonce and permissions (support both metabox and planner nonces)
        $nonce_valid = false;
        if (isset($_POST['nonce'])) {
            // Check metabox nonce
            if (check_ajax_referer('yoapsopo_metabox', 'nonce', false)) {
                $nonce_valid = true;
            }
            // Check planner nonce
            else if (check_ajax_referer('yoapsopo_ajax_nonce', 'nonce', false)) {
                $nonce_valid = true;
            }
        }

        if (!$nonce_valid || !current_user_can('edit_posts')) {
            wp_send_json_error(array('message' => 'Permission denied.'));
            return;
        }

        // Get account from request
        $account = isset($_POST['account']) ? sanitize_text_field(wp_unslash($_POST['account'])) : '';

        if (empty($account)) {
            wp_send_json_error(array('message' => 'TikTok account is required.'));
            return;
        }

        // Fetch creator info from API
        $client = new YOAPSOPO_Client();
        $result = $client->get_tiktok_creator_info($account);

        if (is_wp_error($result)) {
            wp_send_json_error(array('message' => $result->get_error_message()));
            return;
        }

        if ($result['code'] !== 200) {
            wp_send_json_error(array('message' => 'Failed to fetch creator info from API.'));
            return;
        }

        // Return creator info
        wp_send_json_success(array(
            'creator_info' => $result['body'],
            'account' => $account
        ));
    }

    /**
     * AJAX handler to fetch creator info for any network
     *
     * @since 1.6.28
     */
    public function ajax_fetch_creator_info()
    {
        // Check nonce and permissions (support both metabox and planner nonces)
        $nonce_valid = false;
        if (isset($_POST['nonce'])) {
            // Check metabox nonce
            if (check_ajax_referer('yoapsopo_metabox', 'nonce', false)) {
                $nonce_valid = true;
            }
            // Check planner nonce
            else if (check_ajax_referer('yoapsopo_ajax_nonce', 'nonce', false)) {
                $nonce_valid = true;
            }
        }

        if (!$nonce_valid || !current_user_can('edit_posts')) {
            wp_send_json_error(array('message' => 'Permission denied.'));
            return;
        }

        // Get network and account from request
        $network = isset($_POST['network']) ? sanitize_text_field(wp_unslash($_POST['network'])) : '';
        $account = isset($_POST['account']) ? sanitize_text_field(wp_unslash($_POST['account'])) : '';

        if (empty($network)) {
            wp_send_json_error(array('message' => 'Network is required.'));
            return;
        }

        if (empty($account)) {
            wp_send_json_error(array('message' => 'Account is required.'));
            return;
        }

        // Fetch creator info from API based on network
        $client = new YOAPSOPO_Client();
        $result = null;

        switch ($network) {
            case 'tiktok':
                // Treat TikTok the same as other networks - fetch all accounts and find the matching one
                $result = $client->get_client_accounts();
                if (is_wp_error($result)) {
                    wp_send_json_error(array('message' => $result->get_error_message()));
                    return;
                }

                if ($result['code'] !== 200 || !isset($result['body']['success']) || !$result['body']['success']) {
                    wp_send_json_error(array('message' => 'Failed to fetch accounts from API.'));
                    return;
                }

                // Find the account for this network
                $avatar_url = '';
                $nickname = $account;

                if (isset($result['body']['accounts']) && is_array($result['body']['accounts'])) {
                    foreach ($result['body']['accounts'] as $acc) {
                        if (
                            isset($acc['platform']) && $acc['platform'] === $network &&
                            isset($acc['user_identifier']) && $acc['user_identifier'] === $account
                        ) {
                            $avatar_url = isset($acc['picture_url']) ? $acc['picture_url'] : '';
                            $nickname = isset($acc['account_name']) ? $acc['account_name'] : $account;
                            break;
                        }
                    }
                }

                // Return profile info with avatar URL
                wp_send_json_success(array(
                    'avatar_url' => $avatar_url,
                    'account' => $account,
                    'network' => $network,
                    'nickname' => $nickname
                ));
                return;

            default:
                // For all networks, fetch all accounts and find the matching one
                $result = $client->get_client_accounts();
                if (is_wp_error($result)) {
                    wp_send_json_error(array('message' => $result->get_error_message()));
                    return;
                }

                if ($result['code'] !== 200 || !isset($result['body']['success']) || !$result['body']['success']) {
                    wp_send_json_error(array('message' => 'Failed to fetch accounts from API.'));
                    return;
                }

                // Find the account for this network
                $avatar_url = '';
                $nickname = $account;

                if (isset($result['body']['accounts']) && is_array($result['body']['accounts'])) {
                    foreach ($result['body']['accounts'] as $acc) {
                        if (
                            isset($acc['platform']) && $acc['platform'] === $network &&
                            isset($acc['user_identifier']) && $acc['user_identifier'] === $account
                        ) {
                            $avatar_url = isset($acc['picture_url']) ? $acc['picture_url'] : '';
                            $nickname = isset($acc['account_name']) ? $acc['account_name'] : $account;
                            break;
                        }
                    }
                }

                // Return profile info with avatar URL
                wp_send_json_success(array(
                    'avatar_url' => $avatar_url,
                    'account' => $account,
                    'network' => $network,
                    'nickname' => $nickname
                ));
                return;
        }

        wp_send_json_error(array('message' => 'Unsupported network.'));
    }
    /**
     * AJAX handler to fetch importable accounts from YoApy API
     *
     * @since 1.6.30
     */
    public function ajax_fetch_importable_accounts()
    {
        check_ajax_referer('yoapsopo_settings_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permission denied.'));
            return;
        }

        $client = new YOAPSOPO_Client();
        if (!$client::has_keys()) {
            wp_send_json_error(array('message' => 'API keys not configured.'));
            return;
        }

        $result = $client->get_client_accounts();

        if (is_wp_error($result)) {
            wp_send_json_error(array('message' => $result->get_error_message()));
            return;
        }

        if ($result['code'] !== 200 || !isset($result['body']['success']) || !$result['body']['success']) {
            wp_send_json_error(array('message' => 'Failed to fetch accounts from API.'));
            return;
        }

        $accounts = isset($result['body']['accounts']) ? $result['body']['accounts'] : array();

        wp_send_json_success($accounts);
    }

    /**
     * AJAX handler to save imported accounts
     *
     * @since 1.6.30
     */
    public function ajax_save_imported_accounts()
    {
        check_ajax_referer('yoapsopo_settings_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permission denied.'));
            return;
        }

        $accounts = isset($_POST['accounts']) ? $_POST['accounts'] : array();

        if (empty($accounts) || !is_array($accounts)) {
            wp_send_json_error(array('message' => 'No accounts selected.'));
            return;
        }

        // Sanitize accounts
        $sanitized_accounts = array();
        foreach ($accounts as $acc) {
            $sanitized_accounts[] = array(
                'account_id' => sanitize_text_field($acc['account_id']),
                'platform' => sanitize_text_field($acc['platform']),
                'user_identifier' => sanitize_text_field($acc['user_identifier']),
                'account_name' => sanitize_text_field($acc['account_name']),
                'picture_url' => esc_url_raw($acc['picture_url']),
            );
        }

        // Save to option
        update_option('yoapsopo_accounts', $sanitized_accounts, false);

        // Also update legacy single-account settings for backward compatibility if they are empty
        $opt = get_option('yoapsopo_settings', array());
        $updated = false;

        foreach ($sanitized_accounts as $acc) {
            $key = 'account_' . $acc['platform'];
            if (empty($opt[$key])) {
                $opt[$key] = $acc['user_identifier'];
                $updated = true;
            }
        }

        if ($updated) {
            update_option('yoapsopo_settings', $opt, false);
        }

        wp_send_json_success(array(
            'message' => 'Accounts imported successfully.',
            'accounts' => $sanitized_accounts
        ));
    }

    /**
     * AJAX handler to delete a single account
     *
     * @since 1.6.31
     */
    public function ajax_delete_account()
    {
        check_ajax_referer('yoapsopo_settings_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permission denied.'));
            return;
        }

        $account_id = isset($_POST['account_id']) ? sanitize_text_field($_POST['account_id']) : '';

        if (empty($account_id)) {
            wp_send_json_error(array('message' => 'Account ID is required.'));
            return;
        }

        $accounts = get_option('yoapsopo_accounts', array());
        $new_accounts = array();
        $found = false;

        foreach ($accounts as $acc) {
            // Check by account_id if available, or fallback to user_identifier matching
            if (
                (isset($acc['account_id']) && $acc['account_id'] === $account_id) ||
                (isset($acc['user_identifier']) && $acc['user_identifier'] === $account_id)
            ) {
                $found = true;
                continue; // Skip this one (delete)
            }
            $new_accounts[] = $acc;
        }

        if ($found) {
            update_option('yoapsopo_accounts', $new_accounts, false);
            wp_send_json_success(array(
                'message' => 'Account deleted successfully.',
                'accounts' => $new_accounts
            ));
        } else {
            wp_send_json_error(array('message' => 'Account not found.'));
        }
    }

    /**
     * AJAX handler to delete all accounts
     *
     * @since 1.6.31
     */
    public function ajax_delete_all_accounts()
    {
        check_ajax_referer('yoapsopo_settings_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permission denied.'));
            return;
        }

        update_option('yoapsopo_accounts', array(), false);
        wp_send_json_success(array(
            'message' => 'All accounts deleted successfully.',
            'accounts' => array()
        ));
    }

    /**
     * AJAX handler to save settings
     *
     * @since 1.6.32
     */
    public function ajax_save_settings()
    {
        check_ajax_referer('yoapsopo_settings_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permission denied.'));
            return;
        }

        $opt = get_option('yoapsopo_settings', array());
        $post = wp_unslash($_POST);

        $opt['base_url'] = isset($post['base_url']) ? esc_url_raw($post['base_url']) : 'https://api.yoapy.com';
        $opt['key_id'] = isset($post['key_id']) ? sanitize_text_field($post['key_id']) : '';

        // Secret: sanitize as hex
        $secret_raw = isset($post['secret']) ? (string) $post['secret'] : '';
        $opt['secret'] = preg_replace('/[^0-9a-f]/i', '', $secret_raw);

        update_option('yoapsopo_settings', $opt, false);

        wp_send_json_success(array(
            'message' => 'Settings saved successfully.',
            'hasKeys' => !empty($opt['key_id']) && !empty($opt['secret'])
        ));
    }

    /**
     * AJAX handler to parse post content
     *
     * @since 1.6.33
     */
    public function ajax_parse_post_content()
    {
        check_ajax_referer('yoapsopo_metabox', 'nonce');

        $content = isset($_POST['content']) ? wp_unslash($_POST['content']) : '';
        $title = isset($_POST['title']) ? wp_unslash($_POST['title']) : '';

        // Parse blocks if it looks like blocks
        if (has_blocks($content)) {
            $blocks = parse_blocks($content);
            $text_content = '';
            foreach ($blocks as $block) {
                $text_content .= render_block($block);
            }
            $content = $text_content;
        }

        // Strip tags
        $clean_content = wp_strip_all_tags($content);
        $clean_title = wp_strip_all_tags($title);

        // Combine title and content
        $final_text = $clean_title . "\n\n" . $clean_content;

        wp_send_json_success(array(
            'text' => trim($final_text)
        ));
    }

    /**
     * AJAX handler to test connection
     *
     * @since 1.6.32
     */
    public function ajax_test_connection()
    {
        check_ajax_referer('yoapsopo_settings_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permission denied.'));
            return;
        }

        $client = new YOAPSOPO_Client();
        $res = $client->ping();

        if (is_wp_error($res)) {
            wp_send_json_error(array('message' => $res->get_error_message()));
        } else {
            $http_code = (int) ($res['http_code'] ?? 0);
            if ($http_code === 200) {
                wp_send_json_success(array(
                    'message' => 'Connected successfully.',
                    'http_code' => $http_code
                ));
            } else {
                wp_send_json_error(array(
                    'message' => 'Connection failed (HTTP ' . $http_code . ')',
                    'http_code' => $http_code
                ));
            }
        }
    }
}