jQuery(document).ready(function ($) {
    // --- Account Import Logic ---

    var $importBtn = $('#yoapsopo-import-accounts-btn');
    var $modal = $('#yoapsopo-import-modal');
    var $modalClose = $('.yoapsopo-modal-close, .yoapsopo-modal-overlay');
    var $accountsList = $('#yoapsopo-import-list');
    var $saveImportBtn = $('#yoapsopo-save-import-btn');
    var $loadingState = $('#yoapsopo-import-loading');
    var $emptyState = $('#yoapsopo-import-empty');
    var $errorState = $('#yoapsopo-import-error');

    // Open Modal & Fetch Accounts
    $importBtn.on('click', function (e) {
        e.preventDefault();
        $modal.fadeIn(200);
        fetchAccounts();
    });

    // Close Modal
    $modalClose.on('click', function () {
        $modal.fadeOut(200);
    });

    // Fetch Accounts AJAX
    function fetchAccounts() {
        $accountsList.empty();
        $loadingState.show();
        $emptyState.hide();
        $errorState.hide();
        $saveImportBtn.prop('disabled', true);

        $.ajax({
            url: YoApySettings.ajax_url,
            type: 'POST',
            data: {
                action: 'yoapsopo_fetch_importable_accounts',
                nonce: YoApySettings.nonce
            },
            success: function (response) {
                $loadingState.hide();
                if (response.success && response.data && response.data.length > 0) {
                    renderAccounts(response.data);
                    $saveImportBtn.prop('disabled', false);
                } else if (response.success) {
                    $emptyState.show();
                } else {
                    $errorState.find('p').text(response.data.message || 'Unknown error');
                    $errorState.show();
                }
            },
            error: function () {
                $loadingState.hide();
                $errorState.show();
            }
        });
    }

    // Render Accounts List
    function renderAccounts(accounts) {
        var html = '';

        // Group by platform
        var grouped = {};
        $.each(accounts, function (i, acc) {
            if (!grouped[acc.platform]) grouped[acc.platform] = [];
            grouped[acc.platform].push(acc);
        });

        $.each(grouped, function (platform, accs) {
            html += '<div class="yoapsopo-import-group mb-4">';
            html += '<h4 class="font-bold text-slate-700 mb-2 capitalize">' + platform + '</h4>';
            html += '<div class="grid grid-cols-1 md:grid-cols-2 gap-3">';

            $.each(accs, function (i, acc) {
                var avatar = acc.picture_url || '';
                var name = acc.account_name || acc.user_identifier;
                var handle = acc.user_identifier;

                html += '<label class="yoapsopo-import-item flex items-center gap-3 p-3 border rounded-lg cursor-pointer hover:bg-slate-50 transition-colors">';
                html += '<input type="checkbox" name="import_accounts[]" value="' + acc.account_id + '" data-platform="' + acc.platform + '" data-handle="' + handle + '" data-name="' + name + '" data-avatar="' + avatar + '" class="rounded border-slate-300 text-blue-600 focus:ring-blue-500">';

                if (avatar) {
                    html += '<img src="' + avatar + '" class="w-10 h-10 rounded-full object-cover bg-slate-100">';
                } else {
                    html += '<span class="w-10 h-10 rounded-full bg-slate-100 flex items-center justify-center text-xl">👤</span>';
                }

                html += '<div class="overflow-hidden">';
                html += '<div class="font-medium truncate">' + name + '</div>';
                html += '<div class="text-xs text-slate-500 truncate">@' + handle + '</div>';
                html += '</div>';
                html += '</label>';
            });

            html += '</div></div>';
        });

        $accountsList.html(html);
    }

    // Save Selected Accounts
    $saveImportBtn.on('click', function () {
        var selected = [];
        $('input[name="import_accounts[]"]:checked').each(function () {
            var $el = $(this);
            selected.push({
                account_id: $el.val(),
                platform: $el.data('platform'),
                user_identifier: $el.data('handle'),
                account_name: $el.data('name'),
                picture_url: $el.data('avatar')
            });
        });

        if (selected.length === 0) {
            alert('Please select at least one account.');
            return;
        }

        var $btn = $(this);
        $btn.addClass('yoapsopo-btn-loading').prop('disabled', true);

        $.ajax({
            url: YoApySettings.ajax_url,
            type: 'POST',
            data: {
                action: 'yoapsopo_save_imported_accounts',
                nonce: YoApySettings.nonce,
                accounts: selected
            },
            success: function (response) {
                if (response.success) {
                    $modal.fadeOut(200);
                    renderConnectedAccounts(response.data.accounts);
                } else {
                    alert(response.data.message || 'Error saving accounts');
                    $btn.removeClass('yoapsopo-btn-loading').prop('disabled', false);
                }
            },
            error: function () {
                alert('Connection error');
                $btn.removeClass('yoapsopo-btn-loading').prop('disabled', false);
            }
        });
    });

    // --- Copy to Clipboard Utility ---
    $(document).on('click', '[data-clip]', function (e) {
        e.preventDefault();
        var target = $(this).data('clip');
        var $input = $(target);

        if ($input.length) {
            $input.select();
            document.execCommand('copy');

            var $btn = $(this);
            var originalText = $btn.html();
            $btn.html('<span class="dashicons dashicons-yes"></span> Copied!');
            setTimeout(function () {
                $btn.html(originalText);
            }, 2000);
        }
    });

    // --- Toggle Secret Visibility ---
    $('#yoapsopo_secret_toggle').on('click', function () {
        var $input = $('#yoapsopo_secret');
        var $icon = $(this).find('.dashicons');

        if ($input.attr('type') === 'password') {
            $input.attr('type', 'text');
            $icon.removeClass('dashicons-visibility').addClass('dashicons-hidden');
            $(this).contents().last().replaceWith(' Hide');
        } else {
            $input.attr('type', 'password');
            $icon.removeClass('dashicons-hidden').addClass('dashicons-visibility');
            $(this).contents().last().replaceWith(' Show');
        }
    });
    // --- Save Settings Logic ---
    $('#yoapsopo-credentials-form').on('submit', function (e) {
        e.preventDefault();
        var $form = $(this);
        var $btn = $form.find('button[type="submit"]');
        var $msg = $('#yoapsopo-settings-messages');

        $btn.addClass('yoapsopo-btn-loading').prop('disabled', true);
        $msg.empty();

        $.ajax({
            url: YoApySettings.ajax_url,
            type: 'POST',
            data: $form.serialize() + '&action=yoapsopo_save_settings&nonce=' + YoApySettings.nonce,
            success: function (response) {
                $btn.removeClass('yoapsopo-btn-loading').prop('disabled', false);
                if (response.success) {
                    $msg.html('<div class="notice notice-success inline-block w-full box-border mb-4"><p>' + (response.data.message || 'Settings saved.') + '</p></div>');
                    // Enable/disable test connection button based on keys
                    if (response.data.hasKeys) {
                        $('#yoapsopo-test-connection-form button').prop('disabled', false);
                        $('#yoapsopo-import-accounts-btn').show();
                    }
                } else {
                    $msg.html('<div class="notice notice-error inline-block w-full box-border mb-4"><p>' + (response.data.message || 'Error saving settings.') + '</p></div>');
                }
            },
            error: function () {
                $btn.removeClass('yoapsopo-btn-loading').prop('disabled', false);
                $msg.html('<div class="notice notice-error inline-block w-full box-border mb-4"><p>Connection error.</p></div>');
            }
        });
    });

    // --- Test Connection Logic ---
    $('#yoapsopo-test-connection-form').on('submit', function (e) {
        e.preventDefault();
        var $form = $(this);
        var $btn = $form.find('button[type="submit"]');
        var $msg = $('#yoapsopo-connection-status-message');

        $btn.addClass('yoapsopo-btn-loading').prop('disabled', true);
        $msg.empty();

        $.ajax({
            url: YoApySettings.ajax_url,
            type: 'POST',
            data: {
                action: 'yoapsopo_test_connection',
                nonce: YoApySettings.nonce
            },
            success: function (response) {
                $btn.removeClass('yoapsopo-btn-loading').prop('disabled', false);
                if (response.success) {
                    $msg.html('<div class="text-green-600 font-medium mb-2 text-center">✅ ' + response.data.message + '</div>');
                } else {
                    $msg.html('<div class="text-red-600 font-medium mb-2 text-center">❌ ' + response.data.message + '</div>');
                }
            },
            error: function () {
                $btn.removeClass('yoapsopo-btn-loading').prop('disabled', false);
                $msg.html('<div class="text-red-600 font-medium mb-2 text-center">❌ Connection error.</div>');
            }
        });
    });

    // --- Render Connected Accounts Grid ---
    function renderConnectedAccounts(accounts) {
        var $grid = $('#yoapsopo-accounts-grid');
        var $empty = $('#yoapsopo-accounts-empty');
        var $deleteAllBtn = $('#yoapsopo-delete-all-accounts-btn');

        if (!accounts || accounts.length === 0) {
            $grid.empty().addClass('hidden');
            $empty.removeClass('hidden');
            $deleteAllBtn.remove(); // Remove button if no accounts
            return;
        }

        $empty.addClass('hidden');
        $grid.removeClass('hidden');

        // Ensure Delete All button exists if not present
        if ($deleteAllBtn.length === 0) {
            $('#yoapsopo-import-accounts-btn').before(
                '<button type="button" id="yoapsopo-delete-all-accounts-btn" class="yoapsopo-settings-btn yoapsopo-settings-btn-danger yoapsopo-settings-btn-sm mr-2" onclick="if(confirm(\'Are you sure you want to delete ALL connected accounts?\')) { YoApySettings.deleteAllAccounts(); }"><span class="dashicons dashicons-trash"></span> Delete All</button>'
            );
        }

        var html = '';
        var platformColors = {
            'facebook': 'bg-blue-50 text-blue-600 border-blue-100',
            'instagram': 'bg-pink-50 text-pink-600 border-pink-100',
            'youtube': 'bg-red-50 text-red-600 border-red-100',
            'tiktok': 'bg-cyan-50 text-cyan-600 border-cyan-100'
        };

        $.each(accounts, function (i, acc) {
            var pColor = platformColors[acc.platform] || 'bg-slate-50 text-slate-600 border-slate-100';
            var firstLetter = (acc.user_identifier || '?').charAt(0).toUpperCase();
            var avatarHtml = '';

            // Fallback avatar
            avatarHtml += '<div class="yoapsopo-avatar-fallback w-12 h-12 rounded-full bg-slate-100 flex items-center justify-center text-xl font-bold text-slate-500 border border-slate-200" style="' + (acc.picture_url ? 'display: none;' : '') + '">' + firstLetter + '</div>';

            // Image avatar
            if (acc.picture_url) {
                avatarHtml += '<img src="' + acc.picture_url + '" class="w-12 h-12 rounded-full object-cover border border-slate-100 shadow-sm" onerror="this.style.display=\'none\'; this.previousElementSibling.style.display=\'flex\';">';
            }

            html += '<div class="relative group p-4 rounded-xl border border-slate-200 hover:border-blue-300 hover:shadow-md transition-all bg-white">';
            html += '<div class="flex items-start gap-3">';
            html += '<div class="shrink-0 relative">' + avatarHtml + '</div>';

            // Delete button
            html += '<button type="button" class="absolute top-2 right-2 p-1.5 rounded-full bg-white text-slate-400 hover:text-red-500 hover:bg-red-50 border border-slate-200 shadow-sm opacity-0 group-hover:opacity-100 transition-all z-10" onclick="if(confirm(\'Delete this account?\')) { YoApySettings.deleteAccount(\'' + (acc.account_id || acc.user_identifier) + '\'); }" title="Delete Account"><span class="dashicons dashicons-trash" style="font-size: 16px; width: 16px; height: 16px;"></span></button>';

            html += '<div class="min-w-0 flex-grow">';
            html += '<div class="flex items-center gap-2 mb-1"><span class="text-xs font-bold uppercase tracking-wider px-2 py-0.5 rounded-full border ' + pColor + '">' + acc.platform + '</span></div>';
            html += '<h3 class="font-semibold text-slate-800 truncate" title="' + (acc.account_name || '') + '">' + (acc.account_name || 'Unnamed Account') + '</h3>';
            html += '<p class="text-sm text-slate-500 truncate">@' + acc.user_identifier + '</p>';
            html += '</div></div></div>';
        });

        $grid.html(html);
    }

    // --- Delete Account Logic ---
    window.YoApySettings.deleteAccount = function (accountId) {
        $.ajax({
            url: YoApySettings.ajax_url,
            type: 'POST',
            data: {
                action: 'yoapsopo_delete_account',
                nonce: YoApySettings.nonce,
                account_id: accountId
            },
            success: function (response) {
                if (response.success) {
                    renderConnectedAccounts(response.data.accounts);
                } else {
                    alert(response.data.message || 'Error deleting account');
                }
            },
            error: function () {
                alert('Connection error');
            }
        });
    };

    window.YoApySettings.deleteAllAccounts = function () {
        $.ajax({
            url: YoApySettings.ajax_url,
            type: 'POST',
            data: {
                action: 'yoapsopo_delete_all_accounts',
                nonce: YoApySettings.nonce
            },
            success: function (response) {
                if (response.success) {
                    renderConnectedAccounts(response.data.accounts);
                } else {
                    alert(response.data.message || 'Error deleting accounts');
                }
            },
            error: function () {
                alert('Connection error');
            }
        });
    };

});
