document.addEventListener('DOMContentLoaded', function () {
    // Handle post type change to show/hide video field
    const typeSelect = document.getElementById('yoapsopo_type');
    const videoField = document.getElementById('yoapsopo_video_field');
    const videoRequired = document.getElementById('yoapsopo_video_required');

    // Network specific elements
    const networksCheckboxes = document.querySelectorAll('input[name="yoapsopo_networks[]"]');
    const accountSections = {
        'facebook': document.getElementById('yoapsopo_facebook_account_section'),
        'instagram': document.getElementById('yoapsopo_instagram_account_section'),
        'youtube': document.getElementById('yoapsopo_youtube_account_section'),
        'tiktok': document.getElementById('yoapsopo_tiktok_account_section')
    };

    // TikTok specific elements
    const tiktokFields = document.getElementById('yoapsopo_tiktok_fields');
    const commercialToggle = document.getElementById('yoapsopo_tiktok_commercial_toggle');
    const commercialOptions = document.getElementById('yoapsopo_tiktok_commercial_options');
    const commercialYourBrand = document.getElementById('yoapsopo_tiktok_commercial_your_brand');
    const commercialBranded = document.getElementById('yoapsopo_tiktok_commercial_branded');
    const privacySelect = document.getElementById('yoapsopo_tiktok_privacy');
    const consentText = document.getElementById('yoapsopo_tiktok_consent_text');
    const consentCheckbox = document.getElementById('yoapsopo_tiktok_consent');

    // Track elements for commercial toggle
    const commercialTrack = commercialToggle ? commercialToggle.previousElementSibling : null;
    const commercialThumb = commercialTrack ? commercialTrack.querySelector('.yoapsopo-thumb') : null;

    // Get localized strings
    const localizedStrings = typeof YOAPSOPO !== 'undefined' && YOAPSOPO.i18n ? YOAPSOPO.i18n : {
        tiktokFetchingCreatorInfo: 'Fetching creator info...',
        tiktokCreatorInfoError: 'Failed to fetch creator info'
    };

    if (typeSelect && videoField) {
        function toggleVideoField() {
            if (typeSelect.value === 'video') {
                videoField.style.display = 'block';
                // Check if video URL is required
                const videoInput = document.getElementById('yoapsopo_video');
                if (videoInput && !videoInput.value.trim()) {
                    videoRequired.style.display = 'block';
                }
            } else {
                videoField.style.display = 'none';
                videoRequired.style.display = 'none';
            }
        }

        // Initial check
        toggleVideoField();

        // Listen for changes
        typeSelect.addEventListener('change', toggleVideoField);

        // Check video input on change
        const videoInput = document.getElementById('yoapsopo_video');
        if (videoInput) {
            videoInput.addEventListener('input', function () {
                if (typeSelect.value === 'video' && !this.value.trim()) {
                    videoRequired.style.display = 'block';
                } else {
                    videoRequired.style.display = 'none';
                }
            });
        }
    }

    // Show/hide account info sections based on network selection
    function toggleAccountSections() {
        // Hide all account sections first
        Object.values(accountSections).forEach(section => {
            if (section) {
                section.style.display = 'none';
            }
        });

        // Also hide TikTok fields
        if (tiktokFields) {
            tiktokFields.style.display = 'none';
        }

        // Show account sections for selected networks
        networksCheckboxes.forEach(checkbox => {
            if (checkbox.checked) {
                const network = checkbox.value;
                if (accountSections[network]) {
                    accountSections[network].style.display = 'block';
                }

                // Special handling for TikTok
                if (network === 'tiktok') {
                    if (tiktokFields) {
                        tiktokFields.style.display = 'block';
                    }
                }
            }
        });
    }

    // Initialize network change listener
    networksCheckboxes.forEach(checkbox => {
        checkbox.addEventListener('change', toggleAccountSections);
    });

    // Initial check
    toggleAccountSections();

    // Handle commercial content toggle
    if (commercialToggle && commercialTrack && commercialThumb) {
        commercialToggle.addEventListener('change', function () {
            if (this.checked) {
                commercialOptions.style.display = 'block';
                commercialTrack.style.backgroundColor = '#3b82f6';
                commercialThumb.style.transform = 'translateX(1.25rem)';
            } else {
                commercialOptions.style.display = 'none';
                commercialTrack.style.backgroundColor = '#e5e7eb';
                commercialThumb.style.transform = 'translateX(0)';
                commercialYourBrand.checked = false;
                commercialBranded.checked = false;
            }
        });
    }

    // Handle commercial content options
    if (commercialYourBrand && commercialBranded) {
        function updateConsentText() {
            const yourBrandChecked = commercialYourBrand.checked;
            const brandedChecked = commercialBranded.checked;
            
            // We use YOAPSOPO object for localized strings if available, otherwise fallback (though PHP localization is preferred via wp_localize_script)
            // For now, we will rely on text content update logic, but the strings should ideally come from localization object.
            // Since we moved this to JS file, we must use localized object.
            
            // Assuming we will update wp_localize_script in class-admin.php to include these strings.
            const s_music = YOAPSOPO.i18n.tiktokMusicShort || "By posting, you agree to TikTok's Music Usage Confirmation";
            const s_branded = YOAPSOPO.i18n.tiktokBrandedShort || "By posting, you agree to TikTok's Branded Content Policy and Music Usage Confirmation";

            if (yourBrandChecked && !brandedChecked) {
                consentText.textContent = s_music;
            } else if (!yourBrandChecked && brandedChecked) {
                consentText.textContent = s_branded;
            } else if (yourBrandChecked && brandedChecked) {
                consentText.textContent = s_branded;
            } else {
                consentText.textContent = s_music;
            }
        }

        commercialYourBrand.addEventListener('change', updateConsentText);
        commercialBranded.addEventListener('change', updateConsentText);
    }

    // Handle privacy settings for branded content
    if (privacySelect && commercialBranded) {
        commercialBranded.addEventListener('change', function () {
            if (this.checked) {
                // If branded content is selected, show a warning if privacy is not public
                const currentPrivacy = privacySelect.value;
                if (currentPrivacy && currentPrivacy !== 'PUBLIC_TO_EVERYONE') {
                   const msg = YOAPSOPO.i18n.tiktokBrandedPrivacyWarning || 'Branded content visibility cannot be set to private. Privacy setting will be automatically changed to "Everyone".';
                   alert(msg);
                   privacySelect.value = 'PUBLIC_TO_EVERYONE';
                }
            }
        });

        privacySelect.addEventListener('change', function () {
            const brandedChecked = commercialBranded.checked;
            if (brandedChecked && this.value !== 'PUBLIC_TO_EVERYONE') {
                const msg = YOAPSOPO.i18n.tiktokBrandedPrivacyError || 'Branded content visibility cannot be set to private. Please select "Everyone" for branded content.';
                alert(msg);
                this.value = 'PUBLIC_TO_EVERYONE';
            }
        });
    }

    // Form submission validation
    const postForm = document.getElementById('post') || document.querySelector('form');
    if (postForm) {
        postForm.addEventListener('submit', function (e) {
            // Check if TikTok is selected
            let tiktokSelected = false;
            networksCheckboxes.forEach(checkbox => {
                if (checkbox.value === 'tiktok' && checkbox.checked) {
                    tiktokSelected = true;
                }
            });

            if (tiktokSelected) {
                // Validate privacy status
                if (!privacySelect.value) {
                    alert(YOAPSOPO.i18n.tiktokPrivacyRequired || 'Please select a privacy status for your TikTok post.');
                    e.preventDefault();
                    return false;
                }

                // Validate commercial content if toggle is on
                if (commercialToggle.checked) {
                    if (!commercialYourBrand.checked && !commercialBranded.checked) {
                        alert(YOAPSOPO.i18n.tiktokCommercialRequired || 'You need to indicate if your content promotes yourself, a third party, or both.');
                        e.preventDefault();
                        return false;
                    }
                }

                // Validate consent
                if (!consentCheckbox.checked) {
                    alert(YOAPSOPO.i18n.tiktokConsentRequired || "You must agree to TikTok's terms before posting.");
                    e.preventDefault();
                    return false;
                }
            }
        });
    }

    // AJAX Content Fetching Logic
    const captionTextarea = document.querySelector('textarea[name="yoapsopo_text"]');
    const accountCheckboxes = document.querySelectorAll('.yoapsopo-account-checkbox');

    function fetchAndPopulateContent() {
        // Get content from editor
        let title = '';
        let content = '';

        // Check for Gutenberg
        if (typeof wp !== 'undefined' && wp.data && wp.data.select('core/editor')) {
            title = wp.data.select('core/editor').getEditedPostAttribute('title');
            content = wp.data.select('core/editor').getEditedPostAttribute('content');
        }
        // Check for Classic Editor (TinyMCE)
        else if (typeof tinyMCE !== 'undefined' && tinyMCE.activeEditor && !tinyMCE.activeEditor.isHidden()) {
            title = document.getElementById('title').value;
            content = tinyMCE.activeEditor.getContent();
        }
        // Fallback to raw textarea
        else {
            const titleInput = document.getElementById('title');
            const contentInput = document.getElementById('content');
            if (titleInput) title = titleInput.value;
            if (contentInput) content = contentInput.value;
        }

        // Prepare AJAX data
        const data = new FormData();
        data.append('action', 'yoapsopo_parse_post_content');
        data.append('nonce', YOAPSOPO.nonce || '');
        data.append('title', title);
        data.append('content', content);
        // Also send post ID just in case
        const postId = document.getElementById('post_ID') ? document.getElementById('post_ID').value : 0;
        data.append('post_id', postId);

        // Show loading state in textarea
        const originalPlaceholder = captionTextarea.placeholder;
        captionTextarea.placeholder = 'Fetching content...';

        // Send AJAX request
        fetch(YOAPSOPO.ajax || ajaxurl, {
            method: 'POST',
            body: data
        })
            .then(response => response.json())
            .then(response => {
                if (response.success && response.data && response.data.text) {
                    const currentVal = captionTextarea.value.trim();
                    if (!currentVal || currentVal === 'Auto Draft') {
                        captionTextarea.value = response.data.text;
                    } else {
                        captionTextarea.value = response.data.text;
                    }
                }
            })
            .catch(error => {
                console.error('Error fetching content:', error);
            })
            .finally(() => {
                captionTextarea.placeholder = originalPlaceholder;
            });
    }

    // Add listener to all account checkboxes
    accountCheckboxes.forEach(cb => {
        cb.addEventListener('change', function () {
            if (this.checked) {
                // Trigger fetch when an account is selected
                fetchAndPopulateContent();
            }
        });
    });
});
