jQuery(document).ready(function($) {
    // Handle media selection for image field
    var mediaFrame = null; // Declare media frame variable to prevent duplicates
    
    // Ensure only one event handler is attached
    $(document).off('click', '.yoapsopo-pick').on('click', '.yoapsopo-pick', function(e) {
        e.preventDefault();
        e.stopImmediatePropagation(); // Prevent event bubbling
        
        var target = $(this).data('target');
        
        // If media frame already exists, close and destroy it first to prevent duplication
        if (mediaFrame) {
            try {
                mediaFrame.off('select');
                mediaFrame.off('open');
                mediaFrame.off('close');
                if (mediaFrame.isOpen) {
                    mediaFrame.close();
                }
            } catch (err) {
                console.log('Error closing media frame:', err);
            }
            mediaFrame = null;
        }
        
        // Create new media frame
        mediaFrame = wp.media({
            title: typeof YOAPSOPO !== 'undefined' && typeof YOAPSOPO.i18n !== 'undefined' ? YOAPSOPO.i18n.chooseMedia : 'Choose or Upload Media',
            multiple: false,
            library: {
                type: ['image', 'video']
            }
        });
        
        // Add select event listener
        mediaFrame.on('select', function() {
            var attachment = mediaFrame.state().get('selection').first().toJSON();
            var url = attachment.url;
            
            // For videos, we might want to use a different URL
            if (attachment.type === 'video' && attachment.sizes && attachment.sizes.full) {
                url = attachment.sizes.full.url;
            }
            
            $(target).val(url);
            
            // Update preview if exists
            if ($(target).siblings('.yoapsopo-preview-image').length > 0) {
                $(target).siblings('.yoapsopo-preview-image').find('img').attr('src', url);
            }
            
            if ($(target).siblings('.yoapsopo-preview-video').length > 0) {
                $(target).siblings('.yoapsopo-preview-video').find('source').attr('src', url);
                $(target).siblings('.yoapsopo-preview-video').find('video')[0].load();
            }
            
            // Close the media frame after selection
            try {
                mediaFrame.close();
            } catch (err) {
                console.log('Error closing media frame:', err);
            }
        });
        
        // Open media frame
        mediaFrame.open();
    });
    
    // Handle post type change to show/hide video field
    $('#yoapsopo_type').on('change', function() {
        if ($(this).val() === 'video') {
            $('#yoapsopo_video_field').show();
            // Check if video URL is required
            var videoInput = $('#yoapsopo_video');
            if (videoInput && !videoInput.val().trim()) {
                $('#yoapsopo_video_required').show();
            }
        } else {
            $('#yoapsopo_video_field').hide();
            $('#yoapsopo_video_required').hide();
        }
        
        // Update network restrictions when type changes
        updateNetworkRestrictions();
    }).trigger('change'); // Trigger on load to set initial state
    
    // Check video input on change
    $('#yoapsopo_video').on('input', function() {
        if ($('#yoapsopo_type').val() === 'video' && !$(this).val().trim()) {
            $('#yoapsopo_video_required').show();
        } else {
            $('#yoapsopo_video_required').hide();
        }
    });
    
    // Handle form submission validation
    $('form#post').on('submit', function(e) {
        var type = $('#yoapsopo_type').val();
        var videoUrl = $('#yoapsopo_video').val();
        var imageUrl = $('#yoapsopo_image').val();
        var enabled = $('input[name="yoapsopo_enabled"]').is(':checked');
        
        // If YoApy posting is not enabled, skip validation
        if (!enabled) {
            return;
        }
        
        // Check if any networks are selected
        var networks = [];
        $('input[name="yoapsopo_networks[]"]:checked').each(function() {
            networks.push($(this).val());
        });
        
        // If no networks selected, skip validation
        if (networks.length === 0) {
            return;
        }
        
        // Check if media is required for selected networks
        var requiresMedia = false;
        var mediaRequiredNetworks = ['instagram', 'tiktok'];
        for (var i = 0; i < networks.length; i++) {
            if (mediaRequiredNetworks.includes(networks[i])) {
                requiresMedia = true;
                break;
            }
        }
        
        // For video posts, require a video URL
        if (type === 'video' && !videoUrl.trim()) {
            alert(typeof YOAPSOPO !== 'undefined' && typeof YOAPSOPO.i18n !== 'undefined' ? YOAPSOPO.i18n.videoRequired || 'Video URL is required for video posts.' : 'Video URL is required for video posts.');
            e.preventDefault();
            return false;
        }
        
        // For networks that require media, require either image or video
        if (requiresMedia && !imageUrl.trim() && !videoUrl.trim()) {
            alert(typeof YOAPSOPO !== 'undefined' && typeof YOAPSOPO.i18n !== 'undefined' ? YOAPSOPO.i18n.mediaRequired || 'Media (image or video) is required for Instagram and TikTok posts.' : 'Media (image or video) is required for Instagram and TikTok posts.');
            e.preventDefault();
            return false;
        }
        
        // NEW: Additional validation for YouTube and TikTok to require video when selected
        var videoRequiredNetworks = ['youtube', 'tiktok'];
        var videoRequired = false;
        for (var i = 0; i < networks.length; i++) {
            if (videoRequiredNetworks.includes(networks[i])) {
                videoRequired = true;
                break;
            }
        }
        
        // If YouTube or TikTok is selected, require a video
        if (videoRequired && !videoUrl.trim()) {
            alert(typeof YOAPSOPO !== 'undefined' && typeof YOAPSOPO.i18n !== 'undefined' ? YOAPSOPO.i18n.videoRequiredForNetwork || 'Video is required when posting to YouTube or TikTok.' : 'Video is required when posting to YouTube or TikTok.');
            e.preventDefault();
            return false;
        }
    });
    
    // Handle network toggle to show/hide account info and fetch avatar
    $('input[name="yoapsopo_networks[]"]').on('change', function() {
        var network = $(this).val();
        var isChecked = $(this).is(':checked');
        var accountSection = $('#yoapsopo_' + network + '_account_section');
        
        if (isChecked && accountSection.length) {
            accountSection.show();
            
            // Fetch creator info to get avatar
            fetchCreatorInfoForAvatar(network, accountSection);
        } else if (accountSection.length) {
            accountSection.hide();
        }
        
        // Update network restrictions when networks change
        updateNetworkRestrictions();
    });
    
    // Fetch creator info specifically for avatar display
    function fetchCreatorInfoForAvatar(network, accountSection) {
        // Get the account handle from the section
        var usernameElement = accountSection.find('.yoapsopo-account-username');
        if (!usernameElement.length) {
            return;
        }
        
        // Extract account handle (removing @)
        var accountHandle = usernameElement.text().replace('@', '');
        if (!accountHandle) {
            return;
        }
        
        // Make AJAX request
        var formData = new FormData();
        formData.append('action', 'yoapsopo_fetch_creator_info');
        formData.append('network', network);
        formData.append('account', accountHandle);
        formData.append('nonce', typeof YOAPSOPO !== 'undefined' ? YOAPSOPO.nonce : '');
        
        $.ajax({
            url: typeof YOAPSOPO !== 'undefined' ? YOAPSOPO.ajax : ajaxurl,
            type: 'POST',
            data: formData,
            processData: false,
            contentType: false,
            success: function(response) {
                if (response.success && response.data) {
                    // Update avatar image
                    var avatarImg = accountSection.find('.yoapsopo-account-avatar-img');
                    var avatarPlaceholder = accountSection.find('.yoapsopo-account-avatar-placeholder');
                    
                    if (avatarImg.length && response.data.avatar_url) {
                        avatarImg.attr('src', response.data.avatar_url);
                        avatarImg.show();
                        if (avatarPlaceholder.length) {
                            avatarPlaceholder.hide();
                        }
                    } else if (avatarPlaceholder.length) {
                        avatarPlaceholder.show();
                        if (avatarImg.length) {
                            avatarImg.hide();
                        }
                    }
                }
            },
            error: function() {
                // On error, show placeholder
                var avatarPlaceholder = accountSection.find('.yoapsopo-account-avatar-placeholder');
                var avatarImg = accountSection.find('.yoapsopo-account-avatar-img');
                
                if (avatarPlaceholder.length) {
                    avatarPlaceholder.show();
                }
                if (avatarImg.length) {
                    avatarImg.hide();
                }
            }
        });
    }
    
    // Initialize account sections for already checked networks
    setTimeout(function() {
        $('input[name="yoapsopo_networks[]"]:checked').each(function() {
            var network = $(this).val();
            var accountSection = $('#yoapsopo_' + network + '_account_section');
            
            if (accountSection.length) {
                accountSection.show();
                
                // Fetch creator info to get avatar
                fetchCreatorInfoForAvatar(network, accountSection);
            }
        });
        
        // Initialize network restrictions
        updateNetworkRestrictions();
    }, 100);
    
    // Network restrictions implementation (similar to planner)
    function updateNetworkRestrictions() {
        const typeSelect = $('#yoapsopo_type');
        const networkCheckboxes = $('input[name="yoapsopo_networks[]"]');
        
        if (!typeSelect.length || !networkCheckboxes.length) {
            return;
        }
        
        const type = typeSelect.val();
        
        // Define allowed networks for each type
        const allowedNetworks = {
            'image': ['facebook', 'instagram'],
            'story': ['facebook', 'instagram'],
            'live_schedule': ['youtube'],
            'video': ['facebook', 'instagram', 'youtube', 'tiktok'] // All allowed
        };
        
        const allowed = allowedNetworks[type] || [];
        
        networkCheckboxes.each(function() {
            const checkbox = $(this);
            const networkValue = checkbox.val();
            const isAllowed = allowed.includes(networkValue);
            
            if (isAllowed) {
                checkbox.prop('disabled', false);
                // Auto-check if it's the only allowed network (like YouTube for live_schedule)
                if (allowed.length === 1 && !checkbox.is(':checked')) {
                    checkbox.prop('checked', true).trigger('change');
                }
            } else {
                // If network is checked but not allowed, uncheck it and show alert
                if (checkbox.is(':checked')) {
                    checkbox.prop('checked', false).trigger('change');
                    // Show alert if trying to select YouTube or TikTok with incompatible post type
                    if ((networkValue === 'youtube' || networkValue === 'tiktok') && 
                        (type === 'image' || type === 'story')) {
                        alert('To post to ' + networkValue.charAt(0).toUpperCase() + networkValue.slice(1) + 
                              ', please select "Video/YouTube/Reels" post type.');
                    }
                }
                checkbox.prop('disabled', true);
            }
        });
    }
});