/**
 * YoApy Planner - Task Management
 * Handles task listing, polling, and actions.
 */
(function ($) {
    'use strict';

    // Ensure YoApyPlanner object exists
    if (typeof window.YoApyPlanner === 'undefined') {
        window.YoApyPlanner = {};
    }

    // Polling function to check task status
    var pollingTimer = null;
    function pollTaskStatus() {
        if (typeof window.yoapsopo_ajax_object === 'undefined') {
            console.error('yoapsopo_ajax_object is not defined');
            return;
        }

        // Ensure YoApyPlanner object exists
        if (typeof window.YoApyPlanner === 'undefined') {
            window.YoApyPlanner = {};
        }

        $.post(window.yoapsopo_ajax_object.ajax_url, {
            action: 'yoapsopo_get_tasks_ajax',
            nonce: window.yoapsopo_ajax_object.nonce
        }).done(function (response) {
            if (response && response.success && typeof window.YoApyPlanner.renderTaskList === 'function') {
                // Ensure tasks is an array
                var tasks = response.data.tasks || [];
                if (!Array.isArray(tasks)) {
                    tasks = Array.isArray(tasks) ? tasks : [];
                }
                window.YoApyPlanner.renderTaskList(tasks);

                // Smart polling: only continue if there are processing/scheduled tasks
                var hasActiveTask = tasks.some(function (t) {
                    return t.status === 'processing' || t.status === 'pending';
                });
                if (!hasActiveTask && pollingTimer) {
                    clearInterval(pollingTimer);
                    pollingTimer = null;
                }
            }
        }).fail(function (jqXHR, textStatus, errorThrown) {
            console.error('Polling failed:', textStatus, errorThrown);
        });
    }

    // Start smart polling (only when needed)
    function startPolling() {
        if (!pollingTimer) {
            pollingTimer = setInterval(pollTaskStatus, 30000);
        }
    }

    // Initial task load when document is ready
    $(document).ready(function () {
        if (typeof window.yoapsopo_ajax_object !== 'undefined') {
            if (typeof window.YoApyPlanner === 'undefined') {
                window.YoApyPlanner = {};
            }
            setTimeout(function () {
                $.post(window.yoapsopo_ajax_object.ajax_url, {
                    action: 'yoapsopo_get_tasks_ajax',
                    nonce: window.yoapsopo_ajax_object.nonce
                }).done(function (response) {
                    if (response && response.success && typeof window.YoApyPlanner.renderTaskList === 'function') {
                        var tasks = response.data.tasks || [];
                        if (!Array.isArray(tasks)) { tasks = []; }
                        window.YoApyPlanner.renderTaskList(tasks);
                        // Start polling if any tasks are active
                        if (tasks.some(function (t) { return t.status === 'processing' || t.status === 'pending'; })) {
                            startPolling();
                        }
                    }
                }).fail(function (jqXHR, textStatus, errorThrown) {
                    console.error('Initial task load failed:', textStatus, errorThrown);
                });
            }, 100);
        }
    });

    // Expose polling functions
    window.YoApyPlanner.pollTaskStatus = pollTaskStatus;
    window.YoApyPlanner.schedulePolling = pollTaskStatus;
    window.YoApyPlanner.startPolling = startPolling;

    /**
     * Task Actions (Send/Delete)
     */
    $(document).off('click', '.yoapsopo-act, [data-act]').on('click', '.yoapsopo-act, [data-act]', function (e) {
        e.preventDefault();
        e.stopImmediatePropagation();

        const button = $(this);
        const taskId = button.closest('[data-id]').data('id');
        const action = button.data('act');

        if (!taskId || !action) return;

        window.yoapsopoCurrentAction = action;
        window.yoapsopoCurrentTaskId = taskId;
        window.yoapsopoCurrentButton = button;

        if (action === 'delete') {
            // Create or show delete modal
            if (typeof window.YoApyPlanner !== 'undefined' && typeof window.YoApyPlanner.createDeleteModal === 'function') {
                window.YoApyPlanner.createDeleteModal();
            }
            // Ensure modal is visible
            $('#yoapsopo-delete-modal').addClass('active').css({
                'opacity': '1',
                'pointer-events': 'auto'
            });
        } else if (action === 'send') {
            console.log('Send button clicked for task:', taskId);
            button.prop('disabled', true);

            // Add visual feedback
            const originalText = button.html();
            button.html('<span class="dashicons dashicons-update spin"></span> Sending...');

            // Use direct AJAX call for sending
            $.post(typeof window.yoapsopo_ajax_object !== 'undefined' ? window.yoapsopo_ajax_object.ajax_url : ajaxurl, {
                action: 'yoapsopo_task_action',
                nonce: typeof window.yoapsopo_ajax_object !== 'undefined' ? window.yoapsopo_ajax_object.nonce : '',
                act: action,
                id: taskId
            }).done(function (response) {
                console.log('Send task response:', response);
                if (response && response.success) {
                    // Update task list after sending
                    $.post(typeof window.yoapsopo_ajax_object !== 'undefined' ? window.yoapsopo_ajax_object.ajax_url : ajaxurl, {
                        action: 'yoapsopo_get_tasks_ajax',
                        nonce: typeof window.yoapsopo_ajax_object !== 'undefined' ? window.yoapsopo_ajax_object.nonce : ''
                    }).done(function (tasksResponse) {
                        // Ensure YoApyPlanner object exists
                        if (typeof window.YoApyPlanner === 'undefined') {
                            window.YoApyPlanner = {};
                        }

                        if (tasksResponse && tasksResponse.success && typeof window.YoApyPlanner.renderTaskList === 'function') {
                            // Ensure tasks is an array
                            var tasks = tasksResponse.data.tasks || [];
                            if (!Array.isArray(tasks)) {
                                console.warn('Tasks is not an array, converting:', tasks);
                                tasks = Array.isArray(tasks) ? tasks : [];
                            }
                            window.YoApyPlanner.renderTaskList(tasks);
                            // Start polling to track task progress
                            startPolling();
                        } else {
                            console.error('Failed to update task list after send:', tasksResponse);
                        }
                    });
                    if (typeof window.YoApyPlanner !== 'undefined' && typeof window.YoApyPlanner.toast === 'function') {
                        window.YoApyPlanner.toast(typeof window.yoapsopo_ajax_object !== 'undefined' && typeof window.yoapsopo_ajax_object.i18n !== 'undefined' ? window.yoapsopo_ajax_object.i18n.actionSuccess || 'Action successful!' : 'Action successful!', true);
                    } else {
                        // Action successful
                    }
                } else {
                    if (typeof window.YoApyPlanner !== 'undefined' && typeof window.YoApyPlanner.toast === 'function') {
                        window.YoApyPlanner.toast(typeof window.yoapsopo_ajax_object !== 'undefined' && typeof window.yoapsopo_ajax_object.i18n !== 'undefined' ? window.yoapsopo_ajax_object.i18n.actionFailed || 'Action failed.' : 'Action failed.', false);
                    } else {
                        // Action failed
                    }
                }
            }).fail(function (jqXHR, textStatus, errorThrown) {
                console.error('Send task AJAX error:', textStatus, errorThrown);
                if (typeof window.YoApyPlanner !== 'undefined' && typeof window.YoApyPlanner.toast === 'function') {
                    window.YoApyPlanner.toast(typeof window.yoapsopo_ajax_object !== 'undefined' && typeof window.yoapsopo_ajax_object.i18n !== 'undefined' ? window.yoapsopo_ajax_object.i18n.actionFailed || 'Action failed.' : 'Action failed.', false);
                } else {
                    // Action failed
                }
            }).always(function () {
                // Restore button text and enable it
                button.html(originalText).prop('disabled', false);
            });
        }
    });

    // Create delete modal function
    // REMOVED: This function is now handled by planner-modal.js to avoid conflicts

    // REMOVED: Modal initialization now handled by planner-modal.js to avoid conflicts

})(jQuery);