/**
 * YoApy Planner - Modal System
 * Centralized modal management for the planner.
 */
(function ($) {
    'use strict';

    // Create delete modal function
    function createDeleteModal() {
        // Check if modal already exists
        if ($('#yoapsopo-delete-modal').length > 0) {
            return;
        }

        // Create modal HTML
        const modalHTML = `
            <div id="yoapsopo-delete-modal" class="yoapsopo-modal-overlay flex items-center justify-center fixed inset-0 z-[100000] bg-slate-900/50 backdrop-blur-sm transition-opacity opacity-0 pointer-events-none" style="transition: opacity 0.2s ease-out;">
                <div class="yoapsopo-modal bg-white rounded-2xl shadow-2xl w-full max-w-md mx-4 transform scale-95 transition-transform duration-200" style="transform: scale(0.95);">
                    <div class="p-6">
                        <div class="flex items-start gap-4">
                            <div class="flex-shrink-0 w-12 h-12 rounded-full bg-red-100 flex items-center justify-center">
                                <span class="dashicons dashicons-warning text-red-500 text-2xl" style="font-size: 24px; width: 24px; height: 24px;"></span>
                            </div>
                            <div class="flex-1">
                                <h3 class="text-lg font-semibold text-slate-800 mb-2">${typeof window.yoapsopo_ajax_object !== 'undefined' && typeof window.yoapsopo_ajax_object.i18n !== 'undefined' ? window.yoapsopo_ajax_object.i18n.confirmDelete : 'Confirm Delete'}</h3>
                                <p class="text-slate-600 text-sm leading-relaxed">${typeof window.yoapsopo_ajax_object !== 'undefined' && typeof window.yoapsopo_ajax_object.i18n !== 'undefined' ? window.yoapsopo_ajax_object.i18n.deleteWarning : 'Note: This will only delete the task from this list. To delete posts from social networks, you must delete them directly on each platform.'}</p>
                            </div>
                        </div>
                    </div>
                    <div class="bg-slate-50 px-6 py-4 rounded-b-2xl flex gap-3 justify-end">
                        <button id="yoapsopo-modal-cancel" class="px-5 py-2.5 rounded-lg text-slate-700 font-medium text-sm shadow-sm transition-colors bg-white border border-slate-300 hover:bg-slate-50">${typeof window.yoapsopo_ajax_object !== 'undefined' && typeof window.yoapsopo_ajax_object.i18n !== 'undefined' ? window.yoapsopo_ajax_object.i18n.cancel : 'Cancel'}</button>
                        <button id="yoapsopo-modal-confirm" class="px-5 py-2.5 rounded-lg text-white font-medium text-sm shadow-sm transition-colors bg-red-500 hover:bg-red-600">${typeof window.yoapsopo_ajax_object !== 'undefined' && typeof window.yoapsopo_ajax_object.i18n !== 'undefined' ? window.yoapsopo_ajax_object.i18n.delete : 'Delete'}</button>
                    </div>
                </div>
            </div>
        `;
        $('body').append(modalHTML);

        // Attach event handlers
        attachModalEventHandlers();
    }

    // Attach event handlers for the modal
    function attachModalEventHandlers() {
        // Handle modal confirm action
        $(document).off('click', '#yoapsopo-modal-confirm').on('click', '#yoapsopo-modal-confirm', function (e) {
            e.preventDefault();
            e.stopImmediatePropagation();

            if (window.yoapsopoCurrentAction === 'delete' && window.yoapsopoCurrentTaskId && window.yoapsopoCurrentButton) {
                // Close modal immediately
                $('#yoapsopo-delete-modal').removeClass('active');

                // Disable the delete button to prevent double clicks
                window.yoapsopoCurrentButton.prop('disabled', true);

                // Add visual feedback that action is processing
                const $confirmBtn = $(this);
                const originalText = $confirmBtn.html();
                $confirmBtn.html('<span class="dashicons dashicons-update spin"></span> Deleting...').prop('disabled', true);

                // Use direct AJAX call since YoApyPlanner.api might not be reliable
                $.post(typeof window.yoapsopo_ajax_object !== 'undefined' ? window.yoapsopo_ajax_object.ajax_url : ajaxurl, {
                    action: 'yoapsopo_task_action',
                    nonce: typeof window.yoapsopo_ajax_object !== 'undefined' ? window.yoapsopo_ajax_object.nonce : '',
                    act: window.yoapsopoCurrentAction,
                    id: window.yoapsopoCurrentTaskId
                }).done(function (response) {
                    if (response && response.success) {
                        // Ensure the task list is updated after deletion
                        // Update task list immediately with data from response
                        if (response.data && response.data.tasks) {
                            if (typeof window.YoApyPlanner !== 'undefined' && typeof window.YoApyPlanner.renderTaskList === 'function') {
                                window.YoApyPlanner.renderTaskList(response.data.tasks);
                            }
                        } else {
                            // Fallback: Fetch tasks if not provided in response
                            $.post(typeof window.yoapsopo_ajax_object !== 'undefined' ? window.yoapsopo_ajax_object.ajax_url : ajaxurl, {
                                action: 'yoapsopo_get_tasks_ajax',
                                nonce: typeof window.yoapsopo_ajax_object !== 'undefined' ? window.yoapsopo_ajax_object.nonce : ''
                            }).done(function (tasksResponse) {
                                if (tasksResponse && tasksResponse.success) {
                                    if (typeof window.YoApyPlanner !== 'undefined' && typeof window.YoApyPlanner.renderTaskList === 'function') {
                                        window.YoApyPlanner.renderTaskList(tasksResponse.data.tasks || []);
                                    }
                                }
                            });
                        }

                        if (typeof window.YoApyPlanner !== 'undefined' && typeof window.YoApyPlanner.toast === 'function') {
                            window.YoApyPlanner.toast(typeof window.yoapsopo_ajax_object !== 'undefined' && typeof window.yoapsopo_ajax_object.i18n !== 'undefined' ? window.yoapsopo_ajax_object.i18n.actionSuccess || 'Action successful!' : 'Action successful!', true);
                        } else {
                            // Action successful
                        }
                    } else {
                        console.error('Delete task failed:', response);
                        if (typeof window.YoApyPlanner !== 'undefined' && typeof window.YoApyPlanner.toast === 'function') {
                            window.YoApyPlanner.toast(typeof window.yoapsopo_ajax_object !== 'undefined' && typeof window.yoapsopo_ajax_object.i18n !== 'undefined' ? window.yoapsopo_ajax_object.i18n.actionFailed || 'Action failed.' : 'Action failed.', false);
                        } else {
                            // Action failed
                        }
                    }
                }).fail(function (jqXHR, textStatus, errorThrown) {
                    console.error('Delete task AJAX error:', textStatus, errorThrown);
                    if (typeof window.YoApyPlanner !== 'undefined' && typeof window.YoApyPlanner.toast === 'function') {
                        window.YoApyPlanner.toast(typeof window.yoapsopo_ajax_object !== 'undefined' && typeof window.yoapsopo_ajax_object.i18n !== 'undefined' ? window.yoapsopo_ajax_object.i18n.actionFailed || 'Action failed.' : 'Action failed.', false);
                    } else {
                        // Action failed
                    }
                }).always(function () {
                    // Restore button text and enable it
                    $confirmBtn.html(originalText).prop('disabled', false);
                    // Reset variables
                    window.yoapsopoCurrentAction = null;
                    window.yoapsopoCurrentTaskId = null;
                    window.yoapsopoCurrentButton = null;
                });
            }
        });

        // Handle modal cancel/close actions
        $(document).off('click', '#yoapsopo-modal-cancel, .yoapsopo-modal-overlay').on('click', '#yoapsopo-modal-cancel, .yoapsopo-modal-overlay', function (e) {
            e.preventDefault();
            e.stopImmediatePropagation();

            // Close modal
            $('#yoapsopo-delete-modal').removeClass('active');

            // Reset variables
            window.yoapsopoCurrentAction = null;
            window.yoapsopoCurrentTaskId = null;
            window.yoapsopoCurrentButton = null;
        });

        // Prevent modal from closing when clicking inside it
        $(document).off('click', '.yoapsopo-modal').on('click', '.yoapsopo-modal', function (e) {
            e.stopPropagation();
        });

        // Handle delete button click (Delegate to document)
        $(document).off('click', '[data-act="delete"]').on('click', '[data-act="delete"]', function (e) {
            e.preventDefault();

            const btn = $(this);
            const row = btn.closest('tr');
            const card = btn.closest('.rounded-xl'); // For mobile view
            const id = row.length ? row.data('id') : (card.length ? card.data('id') : btn.data('id'));

            if (!id) {
                console.error('Task ID not found for delete action');
                return;
            }

            // Store context for the modal
            window.yoapsopoCurrentAction = 'delete';
            window.yoapsopoCurrentTaskId = id;
            window.yoapsopoCurrentButton = btn;

            // Show modal
            const modal = $('#yoapsopo-delete-modal');
            modal.removeClass('opacity-0 pointer-events-none').addClass('active');
            modal.css('opacity', '1');
        });
    }

    // Initialize modal system
    $(document).ready(function () {
        // Expose functions globally
        if (typeof window.YoApyPlanner === 'undefined') {
            window.YoApyPlanner = {};
        }

        window.YoApyPlanner.createDeleteModal = createDeleteModal;
        window.YoApyPlanner.attachModalEventHandlers = attachModalEventHandlers;

        // Create the delete modal
        createDeleteModal();
    });

})(jQuery);