/**
 * YoApy Planner - Media Picker
 * Handles the WordPress media library integration.
 */
document.addEventListener('DOMContentLoaded', function () {
    // Garante que a biblioteca de mídia do WordPress foi carregada
    if (typeof wp === 'undefined' || typeof wp.media === 'undefined') {
        console.error('YOAPSOPO: WordPress media library not loaded.');
        return;
    }

    // Usaremos uma única instância do seletor de mídia para melhor performance
    let mediaFrame;

    // Usa delegação de evento para ouvir cliques em qualquer botão .yoapsopo-planner-pick
    document.body.addEventListener('click', function (e) {
        const button = e.target.closest('.yoapsopo-planner-pick');
        if (!button) {
            return;
        }

        e.preventDefault();

        const targetSelector = button.dataset.target;
        const targetInput = document.querySelector(targetSelector);
        if (!targetInput) {
            console.error('YOAPSOPO Media Picker: Input de destino não encontrado para o seletor:', targetSelector);
            return;
        }

        // Cria o frame de mídia apenas na primeira vez que for necessário
        if (!mediaFrame) {
            mediaFrame = wp.media({
                title: (window.YoApyPlanner && YoApyPlanner.config && YoApyPlanner.config.i18n && YoApyPlanner.config.i18n.chooseMedia) || 'Choose or Upload Media',
                button: {
                    text: 'Use this media'
                },
                multiple: false
            });
        }

        // 1. Remove qualquer "ouvinte" do evento 'select' que possa existir de cliques anteriores.
        mediaFrame.off('select');

        // 2. Adiciona um novo "ouvinte" para o evento 'select'.
        mediaFrame.on('select', function () {
            const attachment = mediaFrame.state().get('selection').first().toJSON();
            const isVideo = attachment.type === 'video' || (attachment.mime && attachment.mime.startsWith('video/'));

            const imageInput = document.getElementById('yoapsopo_image_url');
            const videoInputEl = document.getElementById('yoapsopo_video_url');

            if (isVideo && videoInputEl) {
                // Route video to video field, clear image field
                videoInputEl.value = attachment.url;
                if (imageInput) {
                    imageInput.value = '';
                    imageInput.dispatchEvent(new Event('input', { bubbles: true }));
                }
                videoInputEl.dispatchEvent(new Event('input', { bubbles: true }));
            } else if (!isVideo && imageInput) {
                // Route image to image field, clear video field
                imageInput.value = attachment.url;
                if (videoInputEl) {
                    videoInputEl.value = '';
                    videoInputEl.dispatchEvent(new Event('input', { bubbles: true }));
                }
                imageInput.dispatchEvent(new Event('input', { bubbles: true }));
            } else {
                // Fallback: use original target
                targetInput.value = attachment.url;
                targetInput.dispatchEvent(new Event('input', { bubbles: true }));
            }
        });

        // 3. Abre o seletor de mídia.
        mediaFrame.open();
    });

    // Handle Drag & Drop Zones
    const dropzones = document.querySelectorAll('.yoapsopo-dz');
    dropzones.forEach(dz => {
        const input = dz.querySelector('.yoapsopo-dz-file');
        const targetSelector = dz.dataset.target;
        const targetInput = document.querySelector(targetSelector);

        // Click to browse
        dz.addEventListener('click', function (e) {
            if (input) input.click();
        });

        // Handle file selection
        if (input) {
            input.addEventListener('change', function () {
                if (this.files && this.files[0]) {
                    // Since we can't easily get a URL from a local file without uploading,
                    // we'll simulate the WordPress media upload flow or just show the name.
                    // Ideally, we should upload this file via AJAX to WP Media Library.
                    // For now, let's trigger the WP Media Picker instead as it handles upload best.

                    // Reset input
                    this.value = '';

                    // Open WP Media Picker
                    const mockEvent = {
                        target: { closest: () => ({ dataset: { target: targetSelector } }) },
                        preventDefault: () => { }
                    };
                    // Trigger the existing media picker logic
                    // We need to find a button that targets this input to simulate the click,
                    // or just call the logic directly.
                    // Let's just open the media frame directly.

                    if (typeof wp !== 'undefined' && wp.media) {
                        const frame = wp.media({
                            title: 'Upload Media',
                            button: { text: 'Use this media' },
                            multiple: false
                        });
                        frame.on('select', function () {
                            const attachment = frame.state().get('selection').first().toJSON();
                            if (targetInput) {
                                targetInput.value = attachment.url;
                                targetInput.dispatchEvent(new Event('input', { bubbles: true }));
                            }
                        });
                        frame.open();
                    }
                }
            });
        }

        // Drag events
        ['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
            dz.addEventListener(eventName, preventDefaults, false);
        });

        function preventDefaults(e) {
            e.preventDefault();
            e.stopPropagation();
        }

        ['dragenter', 'dragover'].forEach(eventName => {
            dz.addEventListener(eventName, highlight, false);
        });

        ['dragleave', 'drop'].forEach(eventName => {
            dz.addEventListener(eventName, unhighlight, false);
        });

        function highlight(e) {
            dz.classList.add('dragover');
        }

        function unhighlight(e) {
            dz.classList.remove('dragover');
        }

        dz.addEventListener('drop', handleDrop, false);

        function handleDrop(e) {
            const dt = e.dataTransfer;
            const files = dt.files;

            if (files && files.length > 0) {
                // Same logic: open media picker or handle upload.
                // Since we want to use WP Media Library, let's open it.
                // Or better, if the user dropped a file, we should probably upload it.
                // But implementing full AJAX upload here is complex.
                // Let's fallback to opening the media picker for now to ensure consistency,
                // or just alert the user to use the picker.

                // Actually, the best UX is to open the media picker with the dropped file ready to upload
                // but WP Media doesn't support that easily from external drop.

                // Let's just open the media picker.
                if (typeof wp !== 'undefined' && wp.media) {
                    const frame = wp.media({
                        title: 'Upload Media',
                        button: { text: 'Use this media' },
                        multiple: false
                    });
                    frame.on('select', function () {
                        const attachment = frame.state().get('selection').first().toJSON();
                        if (targetInput) {
                            targetInput.value = attachment.url;
                            targetInput.dispatchEvent(new Event('input', { bubbles: true }));
                        }
                    });
                    frame.open();
                }
            }
        }
    });
});
