/**
 * YoApy Planner - Form
 * Handles form submission and validation.
 */
(function ($) {
    'use strict';

    // Handle network selection to show/hide account sections and TikTok fields
    window.initNetworkFields = function () {

        const networkCheckboxes = document.querySelectorAll('input[name="networks[]"]');
        const tiktokFields = document.getElementById('yoapsopo_tiktok_fields');

        // Account sections for each network
        const accountSections = {
            'facebook': document.getElementById('yoapsopo_facebook_account_section'),
            'instagram': document.getElementById('yoapsopo_instagram_account_section'),
            'youtube': document.getElementById('yoapsopo_youtube_account_section'),
            'tiktok': document.getElementById('yoapsopo_tiktok_account_section')
        };

        if (networkCheckboxes.length) {
            function toggleNetworkFields() {

                // Hide all account sections first
                Object.values(accountSections).forEach(section => {
                    if (section) {
                        section.style.display = 'none';
                    }
                });

                let tiktokSelected = false;

                // Show account sections for selected networks
                for (let i = 0; i < networkCheckboxes.length; i++) {
                    if (networkCheckboxes[i].checked) {
                        const network = networkCheckboxes[i].value;

                        // Show account section for this network
                        if (accountSections[network]) {
                            accountSections[network].style.display = 'block';

                            // Update account information with actual handle
                            updateAccountInfo(network);
                        }

                        // Check if TikTok is selected
                        if (network === 'tiktok') {
                            tiktokSelected = true;
                        }
                    }
                }

                // Handle TikTok fields
                if (tiktokFields) {
                    tiktokFields.style.display = tiktokSelected ? 'block' : 'none';
                }

                // Toggle required attribute for consent checkbox
                const consentCheckbox = document.getElementById('yoapsopo_tiktok_consent');
                if (consentCheckbox) {
                    if (tiktokSelected) {
                        consentCheckbox.setAttribute('required', 'required');
                        // Fetch creator info when TikTok is selected
                        fetchTiktokCreatorInfo();
                    } else {
                        consentCheckbox.removeAttribute('required');
                    }
                }
            }

            for (let i = 0; i < networkCheckboxes.length; i++) {
                networkCheckboxes[i].addEventListener('change', toggleNetworkFields);
            }
            toggleNetworkFields();

            // Add listener for account checkboxes to update TikTok info dynamically
            const tiktokAccountSection = document.getElementById('yoapsopo_tiktok_account_section');
            if (tiktokAccountSection) {
                tiktokAccountSection.addEventListener('change', function (e) {
                    if (e.target && e.target.type === 'checkbox') {
                        fetchTiktokCreatorInfo();
                    }
                });
            }
        }

        // Handle commercial content toggle
        const commercialToggle = document.getElementById('yoapsopo_tiktok_commercial_toggle');
        const commercialOptions = document.getElementById('yoapsopo_tiktok_commercial_options');

        if (commercialToggle) {
            commercialToggle.addEventListener('change', function () {
                if (this.checked) {
                    commercialOptions.style.display = 'block';
                } else {
                    commercialOptions.style.display = 'none';
                    // Reset sub-options
                    const subOptions = commercialOptions.querySelectorAll('input[type="checkbox"]');
                    subOptions.forEach(opt => opt.checked = false);
                }
            });
        }
    }

    // Update account information with actual handle
    function updateAccountInfo(network) {

        // Get the account section
        const accountSection = document.getElementById(`yoapsopo_${network}_account_section`);
        if (!accountSection) {
            return;
        }

        // Get the username element
        const usernameElement = accountSection.querySelector('.yoapsopo-account-username');
        if (!usernameElement) {
            return;
        }

        // Get the account handle from meta element
        const metaElement = document.getElementById('yoapsopo_prev_meta');
        if (!metaElement) {
            return;
        }

        // Get the account handle for this network
        let accountHandle = '';
        switch (network) {
            case 'facebook':
                accountHandle = metaElement.dataset.facebook || metaElement.dataset.default || '';
                break;
            case 'instagram':
                accountHandle = metaElement.dataset.instagram || metaElement.dataset.default || '';
                break;
            case 'youtube':
                accountHandle = metaElement.dataset.youtube || metaElement.dataset.default || '';
                break;
            case 'tiktok':
                accountHandle = metaElement.dataset.tiktok || metaElement.dataset.default || '';
                break;
        }

        // Update the username display
        if (accountHandle) {
            usernameElement.textContent = '@' + accountHandle;
        } else {
            usernameElement.textContent = '@' + (metaElement.dataset.default || 'yourusername');
        }

        // Fetch creator info to get avatar for all networks
        if (accountHandle) {
            fetchCreatorInfoForAvatar(network, accountHandle, accountSection);
        }
    }

    // Fetch creator info specifically for avatar display
    function fetchCreatorInfoForAvatar(network, accountHandle, accountSection) {

        // Make AJAX request
        const xhr = new XMLHttpRequest();
        const formData = new FormData();
        formData.append('action', 'yoapsopo_fetch_creator_info');
        formData.append('network', network);
        formData.append('account', accountHandle);
        formData.append('nonce', typeof window.yoapsopo_ajax_object !== 'undefined' ? window.yoapsopo_ajax_object.nonce : '');

        xhr.open('POST', typeof window.yoapsopo_ajax_object !== 'undefined' ? window.yoapsopo_ajax_object.ajax_url : ajaxurl);
        xhr.onreadystatechange = function () {
            if (xhr.readyState === 4) {

                if (xhr.status === 200) {
                    try {
                        const response = JSON.parse(xhr.responseText);
                        if (response.success && response.data) {
                            // Update avatar image
                            const avatarImg = accountSection.querySelector('.yoapsopo-account-avatar-img');
                            const avatarPlaceholder = accountSection.querySelector('.yoapsopo-account-avatar-placeholder');

                            if (avatarImg && response.data.avatar_url) {
                                avatarImg.src = response.data.avatar_url;
                                avatarImg.style.display = 'block';
                                if (avatarPlaceholder) {
                                    avatarPlaceholder.style.display = 'none';
                                }
                            } else if (avatarPlaceholder) {
                                avatarPlaceholder.style.display = 'block';
                                if (avatarImg) {
                                    avatarImg.style.display = 'none';
                                }
                            }
                        }
                    } catch (e) {
                        console.error('JSON parse error for avatar:', e);
                    }
                }
            }
        };
        xhr.send(formData);
    }

    // Fetch TikTok creator info
    function fetchTiktokCreatorInfo() {
        const accountSection = document.getElementById('yoapsopo_tiktok_account_section');
        const selectedAccounts = [];

        if (accountSection) {
            const checkboxes = accountSection.querySelectorAll('input[type="checkbox"]:checked');
            checkboxes.forEach(cb => {
                selectedAccounts.push({
                    account: cb.dataset.handle || cb.value,
                    nickname: cb.dataset.name || cb.value,
                    avatar_url: cb.dataset.avatar || ''
                });
            });
        }

        // Fallback for legacy or if no accounts selected but TikTok is active (shouldn't happen with new UI)
        if (selectedAccounts.length === 0) {
            const metaElement = document.getElementById('yoapsopo_prev_meta');
            let tiktokAccount = '';

            if (metaElement) {
                tiktokAccount = metaElement.dataset.tiktok ||
                    metaElement.getAttribute('data-tiktok') ||
                    metaElement.dataset.default ||
                    metaElement.getAttribute('data-default') || '';
            }

            if (tiktokAccount) {
                selectedAccounts.push({
                    account: tiktokAccount,
                    nickname: tiktokAccount,
                    avatar_url: ''
                });
            }
        }

        if (selectedAccounts.length === 0) {
            showError('No TikTok account selected');
            return;
        }

        displayCreatorInfo(selectedAccounts);
    }

    function showLoading() {
        const creatorInfoDiv = document.getElementById('yoapsopo_tiktok_creator_info');
        const creatorErrorDiv = document.getElementById('yoapsopo_tiktok_creator_error');
        const creatorLoadingDiv = document.getElementById('yoapsopo_tiktok_creator_loading');

        if (creatorInfoDiv) creatorInfoDiv.style.display = 'none';
        if (creatorErrorDiv) creatorErrorDiv.style.display = 'none';
        if (creatorLoadingDiv) creatorLoadingDiv.style.display = 'block';
    }

    function showError(message) {
        const creatorInfoDiv = document.getElementById('yoapsopo_tiktok_creator_info');
        const creatorLoadingDiv = document.getElementById('yoapsopo_tiktok_creator_loading');
        const creatorErrorDiv = document.getElementById('yoapsopo_tiktok_creator_error');
        const errorText = creatorErrorDiv ? creatorErrorDiv.querySelector('.yoapsopo-tiktok-error-text') : null;

        if (creatorInfoDiv) creatorInfoDiv.style.display = 'none';
        if (creatorLoadingDiv) creatorLoadingDiv.style.display = 'none';
        if (creatorErrorDiv) {
            creatorErrorDiv.style.display = 'block';
            if (errorText) {
                errorText.textContent = message;
            }
        }
    }

    function displayCreatorInfo(accounts) {
        const creatorLoadingDiv = document.getElementById('yoapsopo_tiktok_creator_loading');
        const creatorErrorDiv = document.getElementById('yoapsopo_tiktok_creator_error');
        const creatorInfoDiv = document.getElementById('yoapsopo_tiktok_creator_info');

        if (creatorLoadingDiv) creatorLoadingDiv.style.display = 'none';
        if (creatorErrorDiv) creatorErrorDiv.style.display = 'none';

        if (accounts && accounts.length > 0 && creatorInfoDiv) {
            let html = '';

            // Handle single object passed (legacy compatibility)
            if (!Array.isArray(accounts)) {
                accounts = [accounts];
            }

            accounts.forEach(acc => {
                const avatarHtml = acc.avatar_url
                    ? `<img src="${acc.avatar_url}" class="w-full h-full object-cover">`
                    : `<span class="text-lg">🎵</span>`;

                html += `
                <div style="margin-bottom: 10px; padding: 10px; background: #fff; border-radius: 6px; border: 1px solid #e5e7eb; display: flex; align-items: center; gap: 10px;">
                    <div style="width: 40px; height: 40px; border-radius: 50%; background: #f3f4f6; display: flex; align-items: center; justify-content: center; overflow: hidden; flex-shrink: 0;">
                        ${avatarHtml}
                    </div>
                    <div>
                        <div style="font-weight: 600;">@${acc.account}</div>
                        <div style="font-size: 12px; color: #6b7280;">${acc.nickname}</div>
                    </div>
                </div>`;
            });

            creatorInfoDiv.innerHTML = html;
            creatorInfoDiv.style.display = 'block';
        } else {
            showError('Invalid creator info received');
        }
    }

    // Handle network restrictions based on post type
    function initNetworkRestrictions() {
        const typeSelect = document.querySelector('select[name="type"]');
        const networkCheckboxes = document.querySelectorAll('input[name="networks[]"]');
        const imageElement = document.getElementById('yoapsopo_image_url');
        const videoElement = document.getElementById('yoapsopo_video_url');
        const imageUploadContainer = imageElement ? imageElement.closest('div').parentElement : null;
        const videoUploadContainer = videoElement ? videoElement.closest('div').parentElement : null;

        if (!typeSelect || !networkCheckboxes.length) return;

        function updateNetworkRestrictions() {
            const type = typeSelect.value;
            let allowedNetworks = [];

            // Reset visibility
            if (imageUploadContainer) imageUploadContainer.style.display = 'none';
            if (videoUploadContainer) videoUploadContainer.style.display = 'none';

            if (type === 'live_schedule') {
                allowedNetworks = ['youtube'];
                if (imageUploadContainer) imageUploadContainer.style.display = 'block'; // Cover image
            } else if (type === 'story') {
                allowedNetworks = ['facebook', 'instagram'];
                if (imageUploadContainer) imageUploadContainer.style.display = 'block';
                if (videoUploadContainer) videoUploadContainer.style.display = 'block';
            } else if (type === 'image') {
                allowedNetworks = ['facebook', 'instagram'];
                if (imageUploadContainer) imageUploadContainer.style.display = 'block';
            } else {
                // Video/Reels - all allowed
                allowedNetworks = ['facebook', 'instagram', 'tiktok', 'youtube'];
                if (videoUploadContainer) videoUploadContainer.style.display = 'block';
            }

            networkCheckboxes.forEach(cb => {
                const isAllowed = allowedNetworks.includes(cb.value);
                cb.disabled = !isAllowed;
                if (!isAllowed) {
                    cb.checked = false;
                    cb.parentElement.classList.add('opacity-50', 'cursor-not-allowed');
                } else {
                    cb.parentElement.classList.remove('opacity-50', 'cursor-not-allowed');
                }
            });

            // Trigger change event to update TikTok fields visibility
            networkCheckboxes[0].dispatchEvent(new Event('change'));
        }

        typeSelect.addEventListener('change', updateNetworkRestrictions);
        updateNetworkRestrictions(); // Initial check
    }

    $(document).ready(function () {
        initNetworkFields();
        initNetworkRestrictions();

        // Form Submission
        $('#yoapsopo_form_task').off('submit').on('submit', function (e) {
            // Validation
            const networks = $('input[name="networks[]"]:checked');
            if (networks.length === 0) {
                if (typeof window.YoApyPlanner !== 'undefined' && typeof window.YoApyPlanner.modal === 'function') {
                    window.YoApyPlanner.modal('Validation Error', 'Please select at least one social network.', 'error');
                } else {
                    alert('Please select at least one social network.');
                }
                e.preventDefault();
                e.stopImmediatePropagation(); // Ensure no other handlers run
                return false;
            }

            // Media Validation
            const type = $('select[name="type"]').val();
            const imageUrl = $('#yoapsopo_image_url').val();
            const videoUrl = $('#yoapsopo_video_url').val();

            if (type === 'image' && !imageUrl) {
                if (typeof window.YoApyPlanner !== 'undefined' && typeof window.YoApyPlanner.modal === 'function') {
                    window.YoApyPlanner.modal('Media Required', 'Please select an image for your post.', 'error');
                } else {
                    alert('Please select an image for your post.');
                }
                e.preventDefault();
                e.stopImmediatePropagation();
                return false;
            } else if (type === 'video' && !videoUrl) {
                if (typeof window.YoApyPlanner !== 'undefined' && typeof window.YoApyPlanner.modal === 'function') {
                    window.YoApyPlanner.modal('Media Required', 'Please select a video for your post.', 'error');
                } else {
                    alert('Please select a video for your post.');
                }
                e.preventDefault();
                e.stopImmediatePropagation();
                return false;
            } else if (type === 'live_schedule' && !imageUrl) {
                if (typeof window.YoApyPlanner !== 'undefined' && typeof window.YoApyPlanner.modal === 'function') {
                    window.YoApyPlanner.modal('Media Required', 'Please select a cover image for your live schedule.', 'error');
                } else {
                    alert('Please select a cover image for your live schedule.');
                }
                e.preventDefault();
                e.stopImmediatePropagation();
                return false;
            } else if (type === 'story' && !imageUrl && !videoUrl) {
                if (typeof window.YoApyPlanner !== 'undefined' && typeof window.YoApyPlanner.modal === 'function') {
                    window.YoApyPlanner.modal('Media Required', 'Please select an image or video for your story.', 'error');
                } else {
                    alert('Please select an image or video for your story.');
                }
                e.preventDefault();
                e.stopImmediatePropagation();
                return false;
            }

            // Live Schedule Date Validation
            if (type === 'live_schedule') {
                const whenInput = $('input[name="when"]').val();
                if (!whenInput) {
                    if (typeof window.YoApyPlanner !== 'undefined' && typeof window.YoApyPlanner.modal === 'function') {
                        window.YoApyPlanner.modal('Date Required', 'You must select a future date and time for YouTube Live Schedule.', 'error');
                    } else {
                        alert('You must select a future date and time for YouTube Live Schedule.');
                    }
                    e.preventDefault();
                    e.stopImmediatePropagation();
                    return false;
                }

                const selectedDate = new Date(whenInput);
                const now = new Date();
                if (selectedDate <= now) {
                    if (typeof window.YoApyPlanner !== 'undefined' && typeof window.YoApyPlanner.modal === 'function') {
                        window.YoApyPlanner.modal('Invalid Date', 'The scheduled time must be in the future for YouTube Live Schedule.', 'error');
                    } else {
                        alert('The scheduled time must be in the future for YouTube Live Schedule.');
                    }
                    e.preventDefault();
                    e.stopImmediatePropagation();
                    return false;
                }
            }

            // TikTok specific validation
            let tiktokSelected = false;
            networks.each(function () {
                if ($(this).val() === 'tiktok') tiktokSelected = true;
            });

            if (tiktokSelected) {
                const commercialToggle = document.getElementById('yoapsopo_tiktok_commercial_toggle');
                const commercialYourBrand = document.getElementById('yoapsopo_tiktok_commercial_your_brand');
                const commercialBranded = document.getElementById('yoapsopo_tiktok_commercial_branded');
                const consentCheckbox = document.getElementById('yoapsopo_tiktok_consent');
                const privacySelect = document.getElementById('yoapsopo_tiktok_privacy');

                // Validate privacy status
                if (privacySelect && !privacySelect.value) {
                    if (typeof window.YoApyPlanner !== 'undefined' && typeof window.YoApyPlanner.modal === 'function') {
                        window.YoApyPlanner.modal('TikTok Validation', 'Please select a privacy status for your TikTok post.', 'error');
                    } else {
                        alert('Please select a privacy status for your TikTok post.');
                    }
                    e.preventDefault();
                    e.stopImmediatePropagation();
                    return false;
                }

                if (commercialToggle && commercialToggle.checked) {
                    if (commercialYourBrand && commercialBranded &&
                        !commercialYourBrand.checked && !commercialBranded.checked) {
                        if (typeof window.YoApyPlanner !== 'undefined' && typeof window.YoApyPlanner.modal === 'function') {
                            window.YoApyPlanner.modal('TikTok Validation', 'You need to indicate if your content promotes yourself, a third party, or both.', 'error');
                        } else {
                            alert('You need to indicate if your content promotes yourself, a third party, or both.');
                        }
                        e.preventDefault();
                        e.stopImmediatePropagation();
                        return false;
                    }
                }

                if (consentCheckbox && !consentCheckbox.checked) {
                    if (typeof window.YoApyPlanner !== 'undefined' && typeof window.YoApyPlanner.modal === 'function') {
                        window.YoApyPlanner.modal('TikTok Consent Required', 'You must agree to TikTok\'s terms before posting.', 'error');
                    } else {
                        alert('You must agree to TikTok\'s terms before posting.');
                    }
                    e.preventDefault();
                    e.stopImmediatePropagation();
                    return false;
                }
            }

            e.preventDefault();
            e.stopImmediatePropagation(); // Prevent other handlers

            const $form = $(this);
            const $btn = $form.find('button[type="submit"]');

            $btn.prop('disabled', true).html('<span class="dashicons dashicons-update spin"></span> ' + (typeof window.yoapsopo_ajax_object !== 'undefined' && typeof window.yoapsopo_ajax_object.i18n !== 'undefined' ? window.yoapsopo_ajax_object.i18n.saving : 'Saving...'));

            const formData = new FormData(this);
            formData.append('action', 'yoapsopo_save_task_ajax');
            formData.append('nonce', typeof window.yoapsopo_ajax_object !== 'undefined' ? window.yoapsopo_ajax_object.nonce : '');

            $.ajax({
                url: typeof window.yoapsopo_ajax_object !== 'undefined' ? window.yoapsopo_ajax_object.ajax_url : ajaxurl,
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                success: function (response) {
                    if (response.success) {
                        $form[0].reset();
                        if (window.YoApyPlanner && typeof window.YoApyPlanner.updatePreview === 'function') {
                            window.YoApyPlanner.updatePreview();
                        }

                        // Ensure the task list is updated after saving
                        // First try to use the tasks from the response
                        if (response.data && response.data.tasks) {
                            if (typeof window.YoApyPlanner !== 'undefined' && typeof window.YoApyPlanner.renderTaskList === 'function') {
                                window.YoApyPlanner.renderTaskList(response.data.tasks);
                            } else {
                                console.error('YoApyPlanner.renderTaskList function not available');
                            }
                        } else {
                            // Fallback: Fetch tasks if not provided in response
                            if (typeof window.yoapsopo_ajax_object !== 'undefined') {
                                // Get tasks using the standard AJAX method
                                $.post(window.yoapsopo_ajax_object.ajax_url, {
                                    action: 'yoapsopo_get_tasks_ajax',
                                    nonce: window.yoapsopo_ajax_object.nonce
                                }).done(function (tasksResponse) {
                                    if (tasksResponse && tasksResponse.success) {
                                        if (typeof window.YoApyPlanner !== 'undefined' && typeof window.YoApyPlanner.renderTaskList === 'function') {
                                            window.YoApyPlanner.renderTaskList(tasksResponse.data.tasks || []);
                                        } else {
                                            console.error('YoApyPlanner.renderTaskList function not available for fetched tasks');
                                        }
                                    } else {
                                        console.error('Failed to fetch tasks:', tasksResponse);
                                    }
                                }).fail(function (jqXHR, textStatus, errorThrown) {
                                    console.error('Failed to fetch tasks:', textStatus, errorThrown);
                                });
                            }
                        }

                        // Start polling immediately to track the new task
                        if (window.YoApyPlanner && typeof window.YoApyPlanner.pollTaskStatus === 'function') {
                            setTimeout(function () {
                                window.YoApyPlanner.pollTaskStatus();
                            }, 1000);
                        }

                        // Use toast if available, otherwise fallback to console log
                        if (typeof window.YoApyPlanner !== 'undefined' && typeof window.YoApyPlanner.toast === 'function') {
                            window.YoApyPlanner.toast(typeof window.yoapsopo_ajax_object !== 'undefined' && typeof window.yoapsopo_ajax_object.i18n !== 'undefined' ? window.yoapsopo_ajax_object.i18n.taskSaved : 'Task saved successfully!');
                        } else {
                            // Task saved successfully
                        }
                    } else {
                        if (typeof window.YoApyPlanner !== 'undefined' && typeof window.YoApyPlanner.modal === 'function') {
                            window.YoApyPlanner.modal(typeof window.yoapsopo_ajax_object !== 'undefined' && typeof window.yoapsopo_ajax_object.i18n !== 'undefined' ? window.yoapsopo_ajax_object.i18n.error || 'Error' : 'Error', response.data.message || (typeof window.yoapsopo_ajax_object !== 'undefined' && typeof window.yoapsopo_ajax_object.i18n !== 'undefined' ? window.yoapsopo_ajax_object.i18n.error : 'An error occurred.'), 'error');
                        } else {
                            alert(response.data.message || 'An error occurred.');
                        }
                    }
                },
                error: function (jqXHR, textStatus, errorThrown) {
                    console.error('Save task AJAX error:', textStatus, errorThrown);
                    if (typeof window.YoApyPlanner !== 'undefined' && typeof window.YoApyPlanner.modal === 'function') {
                        window.YoApyPlanner.modal('System Error', typeof window.yoapsopo_ajax_object !== 'undefined' && typeof window.yoapsopo_ajax_object.i18n !== 'undefined' ? window.yoapsopo_ajax_object.i18n.error || 'An unexpected error occurred.' : 'An unexpected error occurred.', 'error');
                    } else {
                        alert('An unexpected error occurred.');
                    }
                },
                complete: function () {
                    $btn.prop('disabled', false).html('<span class="dashicons dashicons-saved"></span> ' + (typeof window.yoapsopo_ajax_object !== 'undefined' && typeof window.yoapsopo_ajax_object.i18n !== 'undefined' ? window.yoapsopo_ajax_object.i18n.send || 'Post' : 'Post'));
                }
            });
        });
    });

    // Call the function when the DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', function () {
            window.initNetworkFields();


        });
    } else {
        window.initNetworkFields();


    }

})(jQuery);