/**
 * YoApy Planner - Core
 * Shared utilities and initialization.
 */

// Helper: normalize relative permalinks (e.g. Facebook returns "/reel/123/" without domain)
function yoapsopoNormalizePermalink(url, networkKey) {
    if (!url || typeof url !== 'string') return url;
    // If it starts with / but NOT // (protocol-relative), prepend the domain
    if (url.charAt(0) === '/' && url.charAt(1) !== '/') {
        var base = networkKey ? networkKey.split(':')[0].toLowerCase() : 'facebook';
        var domains = {
            facebook: 'https://www.facebook.com',
            instagram: 'https://www.instagram.com',
            youtube: 'https://www.youtube.com',
            tiktok: 'https://www.tiktok.com'
        };
        return (domains[base] || 'https://www.facebook.com') + url;
    }
    return url;
}
// Create YoApyPlanner namespace with safety mechanism
if (typeof window.YoApyPlanner === 'undefined') {
    // Use a backing variable
    var _yoApyPlanner = {};

    // Define the property with getter/setter to prevent overwrites
    Object.defineProperty(window, 'YoApyPlanner', {
        get: function () {
            return _yoApyPlanner;
        },
        set: function (val) {
            // Merge the new value into the existing one to prevent data loss
            if (val && typeof val === 'object') {
                for (var key in val) {
                    if (Object.prototype.hasOwnProperty.call(val, key)) {
                        _yoApyPlanner[key] = val[key];
                    }
                }
            } else {
                _yoApyPlanner = val;
            }
        },
        configurable: true
    });
}

// Create a function that will be exposed globally
var defineRenderTaskList = function ($) {
    'use strict';

    return function (tasks) {
        // Ensure tasks is an array
        if (!Array.isArray(tasks)) {
            tasks = Array.isArray(tasks) ? tasks : [];
        }

        // Ensure DOM is ready before accessing elements
        if (document.readyState !== 'complete') {
            setTimeout(() => {
                if (typeof window.YoApyPlanner !== 'undefined' && typeof window.YoApyPlanner.renderTaskList === 'function') {
                    window.YoApyPlanner.renderTaskList(tasks);
                }
            }, 50);
            return;
        }

        // Wait for DOM to be ready if elements are not found
        // Check for elements regardless of their visibility state
        let tableBody = $('#yoapsopo_tasks_tbody');
        let cardsContainer = $('#yoapsopo_tasks_cards');
        let placeholder = $('#yoapsopo-no-tasks-placeholder');

        // More aggressive fallback selectors - also look for elements that might be hidden
        if (!tableBody.length) {
            const tableBodyElement = document.getElementById('yoapsopo_tasks_tbody');
            if (tableBodyElement) {
                tableBody = $(tableBodyElement);
            }
        }
        if (!cardsContainer.length) {
            const cardsContainerElement = document.getElementById('yoapsopo_tasks_cards');
            if (cardsContainerElement) {
                cardsContainer = $(cardsContainerElement);
            }
        }
        if (!placeholder.length) {
            const placeholderElement = document.getElementById('yoapsopo-no-tasks-placeholder');
            if (placeholderElement) {
                placeholder = $(placeholderElement);
            }
        }

        // Check if essential elements exist
        // For task display, we need at least one container to exist (either table or cards)
        // The fact that both have length 0 suggests the elements are not in the DOM at all
        if (!tableBody.length && !cardsContainer.length && !placeholder.length) {
            // Check if we've already retried enough times
            if (typeof window.YoApyPlanner.retryCount === 'undefined') {
                window.YoApyPlanner.retryCount = 0;
            }

            if (window.YoApyPlanner.retryCount < 30) { // Increase retry limit
                window.YoApyPlanner.retryCount++;
                // Retry after a short delay
                setTimeout(() => {
                    if (typeof window.YoApyPlanner !== 'undefined' && typeof window.YoApyPlanner.renderTaskList === 'function') {
                        window.YoApyPlanner.renderTaskList(tasks);
                    }
                }, 200);
            } else {
                // Reset retry count for next time
                delete window.YoApyPlanner.retryCount;
            }
            return;
        }

        // If we have elements but they're not visible, we can still work with them
        // Reset retry count if we successfully found the elements
        if (typeof window.YoApyPlanner.retryCount !== 'undefined') {
            delete window.YoApyPlanner.retryCount;
        }

        // Ensure containers are available for manipulation (even if hidden)
        if (tableBody.length) {
            tableBody.empty();
        }
        if (cardsContainer.length) {
            cardsContainer.empty();
        }

        // Handle placeholder visibility
        if (!tasks || tasks.length === 0) {
            if (placeholder.length) {
                // Show placeholder - but make sure it's visible even if parent is hidden
                placeholder.show();
                placeholder.css('display', 'block');
            }
            return;
        }

        if (placeholder.length) {
            placeholder.hide();
        }

        // Force visibility of containers based on viewport
        // This ensures that even if CSS is messed up, we try to show the content
        if (tableBody.length) {
            tableBody.closest('table').show();
            // Ensure the parent wrapper is visible on desktop
            if (window.innerWidth >= 768) {
                tableBody.closest('.yoapsopo-table-wrap').removeClass('hidden');
            }
        }

        if (cardsContainer.length) {
            // Ensure cards are visible on mobile
            if (window.innerWidth < 768) {
                cardsContainer.removeClass('hidden');
            }
        }

        tasks.forEach(task => {
            const networksHtml = (task.networks || []).map(n => {
                const networkName = n.charAt(0).toUpperCase() + n.slice(1);
                return `<span class="yoapsopo-net-chip yoapsopo-net--${n}">${networkName}</span>`;
            }).join('');

            const resultsHtml = task.results ? Object.entries(task.results).map(([net, res]) => {
                const networkName = net.charAt(0).toUpperCase() + net.slice(1);
                if (res && res.permalink) return `<a href="${yoapsopoNormalizePermalink(res.permalink, net)}" target="_blank" class="yoapsopo-result-link">${networkName}</a>`;
                if (res && res.success === false) return `<div class="yoapsopo-result-error" title="${res.message || ''}">${networkName}: Error</div>`;
                return '';
            }).join('') : '';

            // Format date/time for display
            let dateTimeInfo = '';
            if (task.when) {
                const date = new Date(task.when * 1000);
                // For completed tasks, show "Posted on" instead of scheduled time
                if (task.status === 'complete') {
                    dateTimeInfo = `
                    <div class="text-xs text-slate-500 mt-1">
                        ✅ Posted on ${date.toLocaleDateString()} ${date.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' })}
                    </div>
                `;
                } else if (task.status === 'scheduled') {
                    dateTimeInfo = `
                    <div class="text-xs text-slate-500 mt-1">
                        🕒 Scheduled for ${date.toLocaleDateString()} ${date.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' })}
                    </div>
                `;
                } else {
                    dateTimeInfo = `
                    <div class="text-xs text-slate-500 mt-1">
                        📅 Created on ${date.toLocaleDateString()} ${date.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' })}
                    </div>
                `;
                }
            } else if (task.status === 'complete') {
                // For immediate tasks that don't have a scheduled time
                const now = new Date();
                dateTimeInfo = `
                <div class="text-xs text-slate-500 mt-1">
                    ✅ Posted on ${now.toLocaleDateString()} ${now.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' })}
                </div>
            `;
            }

            // Determine the appropriate text for the send button based on task status
            let sendButtonText = window.YoApyPlanner.config.i18n.send || 'Post';
            if (task.status === 'complete') {
                sendButtonText = 'Post Again';
            } else if (task.status === 'error') {
                sendButtonText = 'Try Again';
            } else if (task.status === 'scheduled') {
                sendButtonText = 'Post Now';
            }

            // Renderiza a linha da tabela (Desktop)
            const tableRow = `
            <tr data-id="${task.id}" class="hover:bg-slate-50/50 transition-colors duration-150">
                <td class="px-4 py-3 align-top">
                    <div class="font-medium text-slate-800">${task.title || '—'}</div>
                    <div class="text-xs text-slate-500">${task.type}</div>
                    ${dateTimeInfo}
                </td>
                <td class="px-4 py-3 align-top"><div class="flex flex-wrap gap-1">${networksHtml}</div></td>
                <td class="px-4 py-3 align-top">
                    <span class="yoapsopo-status-badge yoapsopo-status--${task.status}">${task.status}</span>
                    <div class="mt-1.5 space-y-1 text-xs">${resultsHtml}</div>
                    <div class="mt-3">
                        <a href="${window.YoApyPlanner.config.admin_url || ''}admin.php?page=yoapsopo_logs" class="yoapsopo-btn yoapsopo-btn-secondary yoapsopo-btn-xs">
                            <span class="dashicons dashicons-external"></span> View Logs
                        </a>
                    </div>
                </td>
                <td class="px-4 py-3 align-top">
                    <div class="flex flex-col gap-2">
                        <button class="yoapsopo-btn yoapsopo-btn-primary yoapsopo-btn-sm yoapsopo-act py-2" data-act="send"><span class="dashicons dashicons-migrate"></span> ${sendButtonText}</button>
                        <button class="yoapsopo-btn yoapsopo-btn-danger yoapsopo-btn-sm yoapsopo-act py-2" data-act="delete"><span class="dashicons dashicons-trash"></span> ${window.YoApyPlanner.config.i18n.delete || 'Delete'}</button>
                    </div>
                </td>
            </tr>`;

            if (tableBody.length) {
                tableBody.append(tableRow);
            }

            // Renderiza o card para mobile
            const cardHtml = `
            <div data-id="${task.id}" class="rounded-xl border border-slate-200 bg-white p-4 shadow-sm">
                <div class="flex flex-wrap items-start justify-between gap-2">
                    <div>
                        <div class="font-medium text-slate-800">${task.title || '—'}</div>
                        <div class="text-xs text-slate-500">${task.type}</div>
                        ${dateTimeInfo}
                    </div>
                    <span class="yoapsopo-status-badge yoapsopo-status--${task.status}">${task.status}</span>
                </div>
                <div class="mt-3 flex flex-wrap gap-1">
                    ${networksHtml}
                </div>
                ${resultsHtml ? `<div class="mt-2 space-y-1 text-xs">${resultsHtml}</div>` : ''}
                <div class="mt-3 flex flex-wrap gap-2">
                    <button class="yoapsopo-btn yoapsopo-btn-primary yoapsopo-btn-sm yoapsopo-act py-2" data-act="send"><span class="dashicons dashicons-migrate"></span> ${sendButtonText}</button>
                    <button class="yoapsopo-btn yoapsopo-btn-danger yoapsopo-btn-sm yoapsopo-act py-2" data-act="delete"><span class="dashicons dashicons-trash"></span> ${window.YoApyPlanner.config.i18n.delete || 'Delete'}</button>
                </div>
            </div>`;

            if (cardsContainer.length) {
                cardsContainer.append(cardHtml);
            }
        });
    };
};

// Define the renderTaskList function immediately, before jQuery wrapper
// Use a self-assigning approach to ensure it's available immediately
window.YoApyPlanner.renderTaskList = (function () {
    // Return a function that will handle the case where jQuery isn't ready yet
    return function (tasks) {
        if (typeof jQuery !== 'undefined') {
            // If jQuery is available, use the jQuery-optimized version
            var jQueryFunc = defineRenderTaskList(jQuery);
            if (typeof jQueryFunc === 'function') {
                jQueryFunc(tasks);
            }
        } else {
            // If jQuery isn't available, wait for it
            var waitCount = 0;
            var maxWaits = 50; // Maximum number of waits (5 seconds at 100ms intervals)

            var waitForjQuery = function () {
                waitCount++;
                if (typeof jQuery !== 'undefined') {
                    var jQueryFunc = defineRenderTaskList(jQuery);
                    if (typeof jQueryFunc === 'function') {
                        jQueryFunc(tasks);
                    }
                } else if (waitCount < maxWaits) {
                    setTimeout(waitForjQuery, 100);
                }
            };

            setTimeout(waitForjQuery, 100);
        }
    };
})();

(function ($) {
    'use strict';

    // Log when DOM is ready
    $(document).ready(function () {
        // Initialized
    });

    try {
        // Check if yoapsopo_ajax_object exists
        if (typeof yoapsopo_ajax_object === 'undefined') {
            console.error('YOAPSOPO: yoapsopo_ajax_object is not defined. AJAX functionality will be disabled.');
        }

        const ajax_url = yoapsopo_ajax_object.ajax_url;
        const nonce = yoapsopo_ajax_object.nonce;
        const i18n = yoapsopo_ajax_object.i18n;

        // Expose config
        window.YoApyPlanner.config = {
            ajax_url: ajax_url,
            nonce: nonce,
            i18n: i18n,
            admin_url: typeof yoapsopo_ajax_object !== 'undefined' ? yoapsopo_ajax_object.admin_url : ''
        };
    } catch (error) {
        console.error('Error in planner-core.js initialization:', error);
    }

    /**
     * Toast Notification
     */
    window.YoApyPlanner.toast = function (msg, isSuccess = true) {
        const toastId = 'yoapsopo-toast-' + Date.now();
        const toastHTML = `<div id="${toastId}" class="fixed z-[99999] top-5 right-5 rounded-xl px-4 py-3 shadow-lg text-sm text-white transition-all animate-slideIn" style="background: ${isSuccess ? 'linear-gradient(135deg, #22c55e, #16a34a)' : 'linear-gradient(135deg, #ef4444, #dc2626)'};"><span class="dashicons ${isSuccess ? 'dashicons-yes-alt' : 'dashicons-warning'}" style="vertical-align: middle; margin-right: 8px;"></span>${msg}</div>`;
        $('body').append(toastHTML);
        setTimeout(() => {
            const el = $('#' + toastId);
            el.css({ opacity: 0, transform: 'translateY(-20px)' });
            setTimeout(() => el.remove(), 500);
        }, 3000);
    };

    /**
     * API Wrapper
     */
    window.YoApyPlanner.api = function (action, data) {
        data = data || {};
        data.action = action;
        data.nonce = window.YoApyPlanner.config.nonce;
        return $.post(window.YoApyPlanner.config.ajax_url, data);
    };

    /**
     * Render Task List
     */
    // Make sure the function is properly set with jQuery context
    // But don't overwrite the already defined function to preserve the self-assigning approach
    var jQueryEnhancedRenderTaskList = defineRenderTaskList($);

})(jQuery);