<?php
if (! defined('ABSPATH')) exit; // Exit if accessed directly

class Yeekit_Dynamic_Discounts_Simple_Discount
{
    public static function handle($cart, $rule, $apply_mode = 'biggest_discount')
    {
        $discount_settings = $rule['discount'] ?? [];
        $type = $discount_settings['type'] ?? 'percentage';
        $value = floatval($discount_settings['value'] ?? 0);

        if ($type !== 'fixed_price' && $value <= 0) {
            return;
        }

        foreach ($cart->get_cart() as $cart_item_key => $cart_item) {
            // Validate Logic
            if (Yeekit_Dynamic_Discounts_Filter::validate_item($cart_item, $rule)) {
                $product = $cart_item['data'];

                // CRITICAL: Always use original price to prevent double discount
                // Store original price in cart item meta if not exists
                if (!isset($cart_item['yeekit_original_price'])) {
                    $cart_item['yeekit_original_price'] = floatval($product->get_price());
                    // Update cart item with original price stored
                    WC()->cart->cart_contents[$cart_item_key]['yeekit_original_price'] = $cart_item['yeekit_original_price'];
                }

                $original_price = $cart_item['yeekit_original_price'];
                // Get current price (which might be modified by other rules if we iterate all)
                $current_price = floatval($product->get_price());

                $new_price = $original_price;

                if ($type === 'percentage') {
                    $new_price = $original_price * (1 - ($value / 100));
                } elseif ($type === 'fixed') {
                    $new_price = max(0, $original_price - $value);
                } elseif ($type === 'fixed_price') {
                    $new_price = max(0, $value);
                }

                // Decision Logic based on apply_mode
                $should_apply = false;

                if ($apply_mode === 'all') {
                    // For 'all', we apply regardless? Or stack?
                    // Currently 'Yeekit' logic bases on Original Price, so it replaces.
                    // If we want TRUE stacking, we should use current_price as base.
                    // But typically 'All' means apply rule 1, then rule 2.
                    // If Rule 2 is better, it wins in replacement logic.
                    // If Rule 2 is worse, it loses.
                    // So 'All' effectively becomes 'Last Wins' or 'Best Wins' unless we change base.
                    // Let's assume 'All' behaves like 'Biggest Discount' for replacement logic to be safe?
                    // OR maybe user WANTS bad discount to overwrite good one? Unlikely.
                    // Let's stick to "Best Price Wins" for Biggest/Lowest modes.
                    $should_apply = true;
                } elseif ($apply_mode === 'biggest_discount') {
                    // We want the LOWEST price
                    if ($new_price < $current_price) {
                        $should_apply = true;
                    }
                } elseif ($apply_mode === 'lowest_discount') {
                    // We want the HIGHEST price (but still discounted? or just worst deal?)
                    // If current price is original (no discount), then any discount is better.
                    // If current price is $80 (20% off), and new price is $90 (10% off).
                    // We want $90.
                    // Note: If current price == original price, then we apply.
                    if ($current_price == $original_price || $new_price > $current_price) {
                        $should_apply = true;
                    }
                } elseif ($apply_mode === 'first') {
                    $should_apply = true;
                }

                if ($should_apply) {
                    // Calculate actual discount amount
                    $discount_amount = $original_price - $new_price;

                    // Set new price
                    $cart_item['data']->set_price($new_price);

                    // Store metadata for order tracking and reports
                    $cart->cart_contents[$cart_item_key]['_yeekit_free_item'] = [
                        'rule_id' => $rule['id'] ?? 0,
                        'rule_type' => 'yeekit_simple_discount', // Standardized Key
                        'discount_type' => $type, // Standardized Key
                        'discount_value' => $value, // Standardized Key
                        'type' => $type,
                        'value' => $value, // Discount rate/value (10%, $5, etc)
                        'original_price' => $original_price,
                        'discount_amount' => $discount_amount, // Actual $ amount discounted
                        'final_price' => $new_price,
                        'allow_qty_change' => true
                    ];
                }
            }
        }
    }
}
