<?php
if (! defined('ABSPATH')) exit; // Exit if accessed directly

class Yeekit_Dynamic_Discounts_Set_Table
{
    public function __construct()
    {
        add_action('wp', [$this, 'init']);
    }

    public function init()
    {
        // 1. Check global settings (Use shared Bulk table setting or add new one if needed?
        // Currently use shared Bulk table setting or assume default display if Set rule exists)
        $show_table = get_option('yk_dd_show_bulk_discount_table', 'yes');
        if ($show_table !== 'yes') {
            return;
        }

        // 2. Get display position (Shared setting)
        $position = get_option('yk_dd_position_to_show_bulk_table', 'woocommerce_before_add_to_cart_form');
        if ($position) {
            add_action($position, [$this, 'render_table']);
        }
    }

    public function render_table()
    {
        if (! is_product()) {
            return;
        }

        global $product;
        if (! $product instanceof WC_Product) {
            return;
        }

        // 3. Find matching Set Discount rule for this product
        $matched_rule = null;
        $rules = Yeekit_Dynamic_Discounts_Rules_Manager::get_active_rules();

        foreach ($rules as $rule) {
            if (empty($rule['type']) || $rule['type'] !== 'yeekit_set_discount') {
                continue;
            }

            // Create fake cart item to validate
            $cart_item = [
                'product_id' => $product->get_id(),
                'variation_id' => 0,
                'data' => $product,
                'quantity' => 1
            ];

            if (Yeekit_Dynamic_Discounts_Filter::validate_item($cart_item, $rule)) {
                $matched_rule = $rule;
                break;
            }
        }

        if (! $matched_rule || empty($matched_rule['set_adjustments']['ranges'])) {
            return;
        }

        // 4. Display table
        $ranges = $matched_rule['set_adjustments']['ranges'];

        // Sort ascending by quantity
        usort($ranges, function ($a, $b) {
            return ($a['qty'] ?? 0) <=> ($b['qty'] ?? 0);
        });

        $product_price = floatval($product->get_price());
?>

        <div class="yeekit-bulk-table-wrapper yeekit-set-table-wrapper">
            <h3><?php esc_html_e('Bundle Discount', 'yeediscounts'); ?></h3>

            <table id="yeekit-set-table" class="yeekit-bulk-table-msg yeekit-table">
                <thead class="yeekit-bulk-table-thead">
                    <tr class="yeekit-bulk-table-tr yeekit-bulk-table-thead">
                        <th class="yeekit-bulk-table-td">
                            <span><?php esc_html_e('Title', 'yeediscounts'); ?></span>
                        </th>
                        <th class="yeekit-bulk-table-td">
                            <span><?php esc_html_e('Quantity', 'yeediscounts'); ?></span>
                        </th>
                        <th class="yeekit-bulk-table-td">
                            <span><?php esc_html_e('Discount', 'yeediscounts'); ?></span>
                        </th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($ranges as $index => $range) :
                        $qty = $range['qty'] ?? 1;
                        $range_display = $qty; // Display set quantity (e.g. 5)

                        $type = $range['type'] ?? 'fixed_set_price';
                        $value = floatval($range['value'] ?? 0);
                        $label = $range['label'] ?? '';

                        // Auto generate label
                        if (empty($label)) {
                            $label = __('Discount', 'yeediscounts') . ' ' . ($index + 1);
                        }

                        // Calculate display price
                        // Note: Bundle price is usually package price per item or reduced per item
                        $discount_display_text = '';
                        $price_per_item = $product_price;

                        if ($type === 'fixed_set_price') {
                            // Package price 1 set
                            $set_price = $value;
                            $price_per_item = $set_price / $qty;
                            $discount_display_text = wc_price($set_price) . ' / ' . sprintf(
                                /* translators: %s: quantity */
                                __('%s items', 'yeediscounts'),
                                $qty
                            );
                        } elseif ($type === 'percentage') {
                            $discount_display_text = $value . '%';
                            $price_per_item = $product_price * (1 - ($value / 100));
                        } elseif ($type === 'flat') {
                            $discount_display_text = wc_price($value) . ' ' . esc_html__('flat', 'yeediscounts');
                            $price_per_item = max(0, $product_price - $value);
                        } elseif ($type === 'fixed_price') {
                            // If fixed price per item
                            $discount_display_text = wc_price($value) . ' ' . esc_html__('flat', 'yeediscounts');
                            $price_per_item = $value;
                        }
                    ?>
                        <tr class="yeekit-bulk-table-tr yeekit-bulk-table-row">
                            <!-- Title -->
                            <td class="yeekit-bulk-table-td yeekit-bulk-title col_index_1" data-colindex="1">
                                <?php echo esc_html($label); ?>
                            </td>

                            <!-- Quantity -->
                            <td class="yeekit-bulk-table-td yeekit-bulk-range col_index_2" data-colindex="2">
                                <?php echo esc_html($range_display); ?>
                            </td>

                            <!-- Discount -->
                            <td class="yeekit-bulk-table-td yeekit-bulk-table-discount col_index_3" data-colindex="3">
                                <span class="yeekit-table-discounted-value"><?php echo wp_kses_post($discount_display_text); ?></span>
                                <span class="yeekit-table-discounted-price" style="display: none"><?php echo wp_kses_post(wc_price($price_per_item)); ?></span>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
<?php
    }
}

new Yeekit_Dynamic_Discounts_Set_Table();
