<?php
if (! defined('ABSPATH')) exit; // Exit if accessed directly

class Yeekit_Dynamic_Discounts_Set_Discounts
{
    private static $calculating = false;
    private static $original_prices = [];

    /**
     * Main entry point for handling Set/Bundle Discount
     */
    public static function handle($cart, $rule)
    {
        // Prevent recursive loop
        if (self::$calculating) return;
        self::$calculating = true;

        $ranges = $rule['set_adjustments']['ranges'] ?? [];
        if (empty($ranges)) {
            self::$calculating = false;
            return;
        }

        // Sort levels by quantity descending to prioritize highest tier
        usort($ranges, function ($a, $b) {
            return ($b['qty'] ?? 0) <=> ($a['qty'] ?? 0);
        });

        // Determine counting mode
        $operator = $rule['set_adjustments']['operator'] ?? 'product';

        // Pre-calculate totals if needed
        $cumulative_qty = 0;
        $product_qtys = []; // For 'variation' mode: parent_id => total_qty

        if ($operator === 'product_cumulative') {
            foreach ($cart->get_cart() as $cart_item) {
                if (Yeekit_Dynamic_Discounts_Filter::validate_item($cart_item, $rule)) {
                    $cumulative_qty += $cart_item['quantity'];
                }
            }
        } elseif ($operator === 'variation') {
            foreach ($cart->get_cart() as $cart_item) {
                if (Yeekit_Dynamic_Discounts_Filter::validate_item($cart_item, $rule)) {
                    $p_id = $cart_item['product_id']; // Parent ID for variations usually
                    if (!isset($product_qtys[$p_id])) $product_qtys[$p_id] = 0;
                    $product_qtys[$p_id] += $cart_item['quantity'];
                }
            }
        }

        foreach ($cart->get_cart() as $cart_item_key => $cart_item) {
            // Check eligibility
            if (! Yeekit_Dynamic_Discounts_Filter::validate_item($cart_item, $rule)) {
                continue;
            }

            // Determine Qty to check against ranges
            $check_qty = $cart_item['quantity']; // Default 'product' mode

            if ($operator === 'product_cumulative') {
                $check_qty = $cumulative_qty; // Use total sum
            } elseif ($operator === 'variation') {
                $check_qty = $product_qtys[$cart_item['product_id']] ?? $cart_item['quantity'];
            }

            // Restore: Capture Original Price
            $product = $cart_item['data'];
            if (! isset(self::$original_prices[$cart_item_key])) {
                self::$original_prices[$cart_item_key] = floatval($product->get_price());
            }
            $original_price = self::$original_prices[$cart_item_key];

            // Use item qty for Recursive math (not total check qty)
            $qty = $cart_item['quantity']; // Actual item quantity for math

            // Find matching range based on check_qty
            $matched_range = null;
            foreach ($ranges as $range) {
                $set_qty = intval($range['qty'] ?? 1);
                if ($check_qty >= $set_qty) {
                    $matched_range = $range;
                    break;
                }
            }

            if ($matched_range) {
                $set_qty = intval($matched_range['qty'] ?? 1);
                $type = $matched_range['type'] ?? 'fixed_set_price';
                $value = floatval($matched_range['value'] ?? 0);
                $is_recursive = !empty($matched_range['is_recursive']);

                $new_price = $original_price;

                if ($is_recursive) {
                    // Recursive Logic: Every X products get the discount price
                    // Use check_qty (Total Valid Qty) to calculate sets, not just individual item qty
                    $num_sets = floor($check_qty / $set_qty);
                    $remainder = $check_qty % $set_qty;

                    // Calculate Ratios for Weighted Average Price
                    // How much of this item contributes to a Set vs Remainder?
                    // Assuming distribution is uniform across all items.
                    if ($check_qty > 0) {
                        $ratio_discounted = ($num_sets * $set_qty) / $check_qty;
                        $ratio_remainder = $remainder / $check_qty;
                    } else {
                        $ratio_discounted = 0;
                        $ratio_remainder = 1;
                    }

                    // Tính giá cho 1 set item (Discounted)
                    $price_per_set_item = $original_price;

                    if ($type === 'fixed_set_price') {
                        // Giá trọn gói cho set -> chia đều cho số lượng trong set
                        $price_per_set_item = $value / $set_qty;
                    } elseif ($type === 'percentage') {
                        $price_per_set_item = $original_price * (1 - ($value / 100));
                    } elseif ($type === 'flat') {
                        $price_per_set_item = max(0, $original_price - $value);
                    } elseif ($type === 'fixed_price') {
                        $price_per_set_item = max(0, $value);
                    }

                    // Weighted Average Price
                    // New Price = (DiscountedPrice * %PartOfSet) + (OriginalPrice * %PartOfRemainder)
                    $new_price = ($price_per_set_item * $ratio_discounted) + ($original_price * $ratio_remainder);
                } else {
                    // Logic Không đệ quy: Áp dụng all nếu đủ mức

                    // Ở đây implement theo style Bulk nếu không recursive (áp dụng all)
                    if ($type === 'fixed_set_price') {
                        $new_price = $value / $set_qty;
                    } elseif ($type === 'percentage') {
                        $new_price = $original_price * (1 - ($value / 100));
                    } elseif ($type === 'flat') {
                        $new_price = max(0, $original_price - $value);
                    }
                }

                // Calculate actual discount amount
                $discount_amount = $original_price - $new_price;

                // Áp dụng giá mới
                $cart_item['data']->set_price($new_price);

                // Store metadata for order tracking and reports
                $cart->cart_contents[$cart_item_key]['_yeekit_free_item'] = [
                    'rule_id' => $rule['id'] ?? 0,
                    'rule_type' => 'yeekit_set_discount', // Standardized Key
                    'discount_type' => $type, // Standardized Key
                    'discount_value' => $value, // Standardized Key
                    'type' => $type,
                    'value' => $value,
                    'original_price' => $original_price,
                    'discount_amount' => $discount_amount, // Actual $ amount discounted
                    'final_price' => $new_price,
                    'allow_qty_change' => true
                ];
            }
        }

        self::$calculating = false;
    }
}
