<?php
if (! defined('ABSPATH')) exit; // Exit if accessed directly
class Yeekit_Dynamic_Discounts_Sale_Badge
{
    public function __construct()
    {
        // 1. Force WooCommerce to think product is on sale so it renders the flash template
        add_filter('woocommerce_product_is_on_sale', [$this, 'force_on_sale'], 10, 2);
        // 2. Customize the flash content
        add_filter('woocommerce_sale_flash', [$this, 'handle_sale_flash'], 10, 3);
    }
    public function force_on_sale($is_on_sale, $product)
    {
        if (is_admin() && ! defined('DOING_AJAX')) {
            return $is_on_sale;
        }
        // $backtrace = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS, 10); //
        // foreach ($backtrace as $trace) {
        //     if (isset($trace['function']) && $trace['function'] === 'get_price_html') {
        //         return $is_on_sale;
        //     }
        // }
        if ($is_on_sale) {
            return true;
        }
        $show_badge_setting = get_option('yk_dd_show_on_sale_badge', 'at_least_has_any_rules');
        if ($show_badge_setting === 'disabled') {
            return false;
        }
        // Check if product has active rules
        if ($this->is_product_discounted($product)) {
            return true;
        }
        return $is_on_sale;
    }
    // Unified helper directly checking rule match
    private function is_product_discounted($product)
    {
        $show_badge_setting = get_option('yk_dd_show_on_sale_badge', 'at_least_has_any_rules');
        if ($show_badge_setting === 'at_least_has_any_rules') {
            return $this->product_has_any_rule($product);
        } elseif ($show_badge_setting === 'when_condition_matches') {
            return $this->product_matches_conditions($product);
        }
        return false;
    }
    public function handle_sale_flash($sales_html, $post, $product)
    {
        // Safety check if parameters are missing (sometimes called with fewer args)
        if (!$product && $post && $post instanceof WC_Product) {
            $product = $post;
        }
        if (!$product) return $sales_html;
        $show_badge_setting = get_option('yk_dd_show_on_sale_badge', 'at_least_has_any_rules');
        // Debug

        // If explicitly disabled, return original HTML or nothing?
        // Default "disabled" option likely means "Do not show OUR badge", but preserve WC's?
        // OR "Do not show ANY badge"? 
        // Based on "Choose when to display", if User selects "Do not show", they prob imply hiding it for discounted items if not natively on sale.
        // But if item IS natively on sale, we should prob keep it.
        // Let's assume this setting controls visibility for DYNAMICALLY discounted items.
        $should_show = false;
        if ($show_badge_setting === 'disabled') {
            return $sales_html;
        }
        // Native sale check
        if ($product->is_on_sale()) {
            $should_show = true;
        } else {
            // Check dynamic discounts
            if ($show_badge_setting === 'at_least_has_any_rules') {
                $should_show = $this->product_has_any_rule($product);
            } elseif ($show_badge_setting === 'when_condition_matches') {
                $should_show = $this->product_matches_conditions($product);
            }
        }
        // Match found?

        if (!$should_show) {
            return $sales_html;
        }
        // Customization
        $customize = get_option('yk_dd_customize_sale_badge', 'no');
        $force_override = get_option('yk_dd_force_override_sale_badge', 'no');
        // If not customizing and not natively on sale, we still need to show something if dynamic discount applies
        // But if not customizing, we fallback to standard WC badge.
        // If product is NOT natively on sale, $sales_html is empty. We need to generate it.
        if (empty($sales_html) && $should_show && $customize !== 'yes') {
            $sales_html = '<span class="onsale">' . esc_html__('Sale!', 'yeediscounts') . '</span>';
        }
        if ($customize === 'yes') {
            $text = get_option('yk_dd_sale_badge_text', '');
            $content = 'Sale'; // Default fallback
            $show_percentage = get_option('yk_dd_show_percentage_in_sale_badge', 'no');
            // Calculate percentage if needed
            $percentage_text = '';
            if ($show_percentage === 'yes') {
                $percent = $this->get_max_discount_percentage($product);
                if ($percent > 0) {
                    $percentage_text = '-' . round($percent) . '%';
                }
            }
            $final_text = $text;
            if (!empty($percentage_text)) {
                $final_text = $percentage_text; // Override content with percentage
                // Or maybe append? The description says "Displays only when rule matches else displays default sale badge content"
                // Usually percentage replaces "Sale"
            }
            // Replace content in existing HTML or create new
            if ($force_override === 'yes' || empty($sales_html)) {
                return '<span class="onsale">' . esc_html($final_text) . '</span>';
            } else {
                // Try to replace text inside existing HTML
                // This is tricky as HTML structure varies by theme. 
                // Simple regex or string replace might be safer implementation for now.
                return preg_replace('/(>)([^<]+)(<)/', '$1' . esc_html($final_text) . '$3', $sales_html, 1);
            }
        }
        return $sales_html;
    }
    private function get_matching_rules_for_product($product)
    {
        $active_rules = Yeekit_Dynamic_Discounts_Rules_Manager::get_active_rules();
        $matched_rules = [];
        foreach ($active_rules as $rule) {
            $item = [
                'product_id' => $product->get_id(),
                'variation_id' => $product->is_type('variation') ? $product->get_id() : 0,
                'data' => $product,
                'quantity' => 1,
            ];
            if (Yeekit_Dynamic_Discounts_Filter::validate_item($item, $rule)) {
                $matched_rules[] = $rule;
            }
        }
        // Handle Exclusivity
        foreach ($matched_rules as $rule) {
            if (!empty($rule['ignore_other_rules'])) {
                // If we find an exclusive rule that matches, return ONLY matching exclusive rules? 
                // Or just this one? Typically "Ignore all OTHER rules".
                // If multiple exclusive rules match, arguably we can return all exclusive ones (and let priority decide) 
                // or just the first one. Let's return only exclusive ones found.
                // Filter to keep ONLY exclusive rules
                $exclusive_matches = array_filter($matched_rules, function ($r) {
                    return !empty($r['ignore_other_rules']);
                });
                // If we assume strict "First exclusive wins", just return the first one found (if we iterated in priority order).
                // But generally "Ignore non-exclusive rules".
                return array_values($exclusive_matches);
            }
        }
        return $matched_rules;
    }
    private function product_has_any_rule($product)
    {
        $matches = $this->get_matching_rules_for_product($product);
        return !empty($matches);
    }
    private function product_matches_conditions($product)
    {
        // Currently same as "Apply Any Rule" because validate_item checks conditions too?
        // Actually validate_item checks Filter (Product/Cat/Author).
        // Conditions (Cart Value, etc.) are in Yeekit_Dynamic_Discounts_Rules_Manager::match_conditions
        // BUT match_conditions checks CART state usually (subtotal, etc.)
        // If we are on Product Page (catalog), we can't check Cart conditions accurately.
        // So standard practice: Assume conditions *might* be met, check Filters.
        // So for now, alias to product_has_any_rule
        return $this->product_has_any_rule($product);
    }
    private function get_max_discount_percentage($product)
    {
        $max_percent = 0;

        $rules = $this->get_matching_rules_for_product($product);

        foreach ($rules as $rule) {

            // Estimate discount
            $discount = $rule['discount'] ?? [];
            $type = $discount['type'] ?? 'percentage';
            $value = floatval($discount['value'] ?? 0);
            if ($type === 'percentage') {
                if ($value > $max_percent) $max_percent = $value;
            } elseif ($type === 'fixed' || $type === 'fixed_price') {
                // Calculate percentage equivalent
                $price = $product->get_price();
                if ($price > 0) {
                    $diff = 0;
                    if ($type === 'fixed') $diff = $value;
                    if ($type === 'fixed_price') $diff = $price - $value;
                    $p = ($diff / $price) * 100;
                    if ($p > $max_percent) $max_percent = $p;
                }
            }
            // check BOGO

            if (isset($rule['type']) && $rule['type'] === 'yeekit_buy_x_get_x') {
                $ranges = $rule['buyx_getx_adjustments']['ranges'] ?? [];
                foreach ($ranges as $range) {
                    $free_type = $range['free_type'] ?? 'free_product';
                    $amount = floatval($range['amount'] ?? 0);

                    if ($free_type === 'percentage') {
                        if ($amount > $max_percent) $max_percent = $amount;
                    } elseif ($free_type === 'free_product') {
                        $max_percent = 100;
                    } elseif ($free_type === 'flat') {
                        // Similar calculation for fixed amount on the 'Get' item
                        $price = $product->get_price();
                        if ($price > 0) {
                            $p = ($amount / $price) * 100;
                            if ($p > $max_percent) $max_percent = $p;
                        }
                    }
                }
            }
        }
        return $max_percent;
    }
}
new Yeekit_Dynamic_Discounts_Sale_Badge();
