<?php
if (! defined('ABSPATH')) exit; // Exit if accessed directly

class Yeekit_Dynamic_Discounts_Price_Display
{
    public function __construct()
    {
        add_filter('woocommerce_get_price_html', [$this, 'handle_price_html'], 10, 2);
        add_filter('woocommerce_cart_item_price', [$this, 'handle_cart_item_price'], 10, 3);
    }

    public function handle_price_html($price_html, $product)
    {
        if (is_admin()) {
            return $price_html;
        }

        // Check setting
        $show_price_setting = get_option('yk_dd_show_discounted_price', 'at_least_has_any_rules');

        if ($show_price_setting === 'disabled') {
            return $price_html;
        }

        // Check Page Context
        // Use default only if option doesn't exist (false), otherwise respect saved value (even if empty)
        $pages = get_option('yk_dd_show_discounted_price_pages');
        if ($pages === false) {
            $pages = ['shop', 'category', 'product'];
        }
        if (!is_array($pages)) $pages = (array) $pages;

        $show_on_page = false;

        if (is_product() && in_array('product', $pages)) {
            $show_on_page = true;
        }
        // Check Category/Shop
        // is_product_category() checks for category archives
        // is_shop() checks for the shop page
        elseif ((is_product_category() || is_product_tag()) && in_array('category', $pages)) {
            $show_on_page = true;
        } elseif (is_shop() && !is_product_category() && in_array('shop', $pages)) {
            $show_on_page = true;
        }

        if (!$show_on_page) {
            return $price_html;
        }

        // Also support other archives/home if user considers them "shop" or "category"? 
        // Typically "shop" implies main shop loop. "Category" implies archive loops.
        // What about Home page with product shortcodes?
        // Usually `is_front_page()` or `is_home()`.
        // Let's stick to standard WC conditionals for now.

        // Avoid recursion if likely? safe enough.

        $rules = Yeekit_Dynamic_Discounts_Rules_Manager::get_active_rules();
        if (empty($rules)) {
            return $price_html;
        }

        // Determine best discount for this product context
        $best_discount_amount = 0;
        $original_price = floatval($product->get_price());

        // Safety: If price is 0 or empty (e.g. variable product root sometimes), handle careful
        if (!$original_price && !$product->is_type('variable')) {
            // Can't discount 0
            return $price_html;
        }

        // Check Variable Product: WC usually calls this for root, and then for variations.
        // If variable product, we might want to show range?
        // Let's stick to handling the specific product passed (Simple or Variation).
        // If it's the Parent Variable product, $product->get_price() usually returns min_price.

        $matched = false;
        $final_price = $original_price;

        // Identify matching rules first
        $matching_rules = [];
        foreach ($rules as $rule) {
            // ... filter type ...
            if (!in_array($rule['type'], ['yeekit_simple_discount'])) {
                continue;
            }

            $item = [
                'product_id' => $product->get_id(),
                'variation_id' => $product->is_type('variation') ? $product->get_id() : 0,
                'data' => $product,
                'quantity' => 1,
            ];

            if (Yeekit_Dynamic_Discounts_Filter::validate_item($item, $rule)) {
                $matching_rules[] = $rule;
            }
        }

        // Handle Exclusivity
        $rules_to_process = $matching_rules;
        foreach ($matching_rules as $rule) {
            if (!empty($rule['ignore_other_rules'])) {
                $rules_to_process = array_filter($matching_rules, function ($r) {
                    return !empty($r['ignore_other_rules']);
                });
                break; // Found exclusive, stop looking
            }
        }

        // Process final set of rules
        foreach ($rules_to_process as $rule) {
            $matched = true;

            // Calculate Discount
            $discount_settings = $rule['discount'] ?? [];
            $d_type = $discount_settings['type'] ?? 'percentage';
            $d_value = floatval($discount_settings['value'] ?? 0);

            $calculated_price = $final_price;

            if ($d_type === 'percentage') {
                $calculated_price = $original_price * (1 - ($d_value / 100));
            } elseif ($d_type === 'fixed') {
                $calculated_price = max(0, $original_price - $d_value);
            } elseif ($d_type === 'fixed_price') {
                $calculated_price = max(0, $d_value);
            }

            // Logic: Keep the lowest price so far (best discount)
            if ($calculated_price < $final_price) {
                $final_price = $calculated_price;
            }
        }

        if ($matched && $final_price < $original_price) {
            // Render new price HTML
            // Using WC function to handle currency, formatting, and the <del> structure
            return wc_format_sale_price($original_price, $final_price);
        }

        return $price_html;
    }
    public function handle_cart_item_price($price_html, $cart_item, $cart_item_key)
    {
        if (is_admin() && !defined('DOING_AJAX')) {
            return $price_html;
        }

        // Check setting
        $show_strikeout = get_option('yk_dd_show_strikeout_on_cart_page', 'yes');
        if ($show_strikeout !== 'yes') {
            return $price_html;
        }

        $product = $cart_item['data'];
        $regular_price = $product->get_regular_price();
        $current_price = $product->get_price();

        // Fix: Use stored original price if available (from rule meta)
        // This ensures consistency if the rule applied based on a specific price
        if (!empty($cart_item['_yeekit_free_item']) && isset($cart_item['_yeekit_free_item']['original_price'])) {
            $regular_price = floatval($cart_item['_yeekit_free_item']['original_price']);
        }

        // If price is reduced from regular price
        if ($regular_price > $current_price) {
            return wc_format_sale_price($regular_price, $current_price);
        }

        return $price_html;
    }
}
new Yeekit_Dynamic_Discounts_Price_Display();
