<?php
if (! defined('ABSPATH')) exit; // Exit if accessed directly

class Yeekit_Dynamic_Discounts_Free_Shipping
{
    public function __construct()
    {
        // Hook into shipping calc
        add_filter('woocommerce_package_rates', [$this, 'handle_shipping'], 100, 2);
    }

    public function handle_shipping($rates, $package)
    {
        if (is_admin() && ! defined('DOING_AJAX')) {
            return $rates;
        }

        // Get ACTIVE rules for Free Shipping
        $rules = Yeekit_Dynamic_Discounts_Rules_Manager::get_active_rules();
        $fs_rules = [];
        foreach ($rules as $rule) {
            if (isset($rule['type']) && $rule['type'] === 'yeekit_free_shipping') {
                $fs_rules[] = $rule;
            }
        }

        if (empty($fs_rules)) return $rates;

        // Check if ANY rule matches
        $is_valid = false;
        foreach ($fs_rules as $rule) {
            // Validate against CART items in the package
            // Shipping package contains 'contents' which mimics cart
            // But Filter::validate_item expects cart item structure.
            // We can iterate package contents.

            // Logic: Typically Free Shipping is "If Cart meets conditions X, Y, Z". 
            // Conditions are usually Subtotal, Quantity, or Specific Products in Cart.
            // Our Rules Manager Filter is Item-based? Or Cart-based?
            // "Yeekit_Dynamic_Discounts_Filter::validate_item" validates a SINGLE item.
            // BUT for Free Shipping, we usually want "Cart has Item X" effectively.

            // Let's assume if ANY item in the cart satisfies the rule's conditions, Free Shipping applies?
            // OR do we need a special "Cart Validation" logic?
            // Current Filter.php supports: Subtotal (of item?), Quantity (of item?), Category, Product.
            // If the rule has "Subtotal greater than 100", currently it checks ITEM subtotal. 
            // This might be a limitation for "Order Subtotal > 100".
            // However, usually "Filter" in this plugin context seems to select ITEMS.
            // If the filter selects at least one item, does the rule apply?
            // For now, YES. If any item matches the rule, Free Shipping is granted. 

            foreach ($package['contents'] as $key => $item) {
                if (Yeekit_Dynamic_Discounts_Filter::validate_item($item, $rule)) {
                    $is_valid = true;
                    break;
                }
            }
            if ($is_valid) break;
        }
        if ($is_valid) {
            // User requested to use the Global Setting for the Title
            $method_title = get_option('yk_dd_free_shipping_method_title', 'Free shipping');

            // Create Single Rate
            // Even if multiple rules match, we offer one "Global Free Shipping" method.
            $method_id = 'yeekit_free_shipping';

            $new_rate = new WC_Shipping_Rate(
                $method_id,
                $method_title,
                0, // Cost
                [], // Taxes
                'yeekit_free_shipping' // Method ID
            );

            $rates[$method_id] = $new_rate;
        }

        return $rates;
    }
}
