<?php
if (! defined('ABSPATH')) exit; // Exit if accessed directly

class Yeekit_Dynamic_Discounts_Filter
{
    /**
     * Helper to retrieve IDs eligible for this rule based on filters
     * Simplified: checks for 'cart_item_products' filter. 
     * If other filters used (cat/tag), would need to resolve those to IDs or check item terms.
     */
    /**
     * Validate if a cart item matches the rule filters
     */
    public static function validate_item($cart_item, $rule)
    {
        $filters = $rule['filters'] ?? [];
        if (empty($filters)) {
            return true;
        }

        $product_id = $cart_item['product_id'];
        $variation_id = $cart_item['variation_id'];
        $product = $cart_item['data'];

        $has_inclusion = false;
        $matched_inclusion = false;

        foreach ($filters as $filter) {
            $type = $filter['type'];
            $operator = $filter['operator'];
            $value = $filter['value'];

            $match = false;

            switch ($type) {
                case 'cart_item_products':
                    $check_ids = array_map('absint', (array)$value);
                    if (in_array($product_id, $check_ids) || ($variation_id && in_array($variation_id, $check_ids))) {
                        $match = true;
                    }
                    break;
                case 'cart_item_product_category':
                    $check_id = $variation_id ? wp_get_post_parent_id($variation_id) : $product_id;
                    if (has_term($value, 'product_cat', $check_id)) {
                        $match = true;
                    }
                    break;
                case 'cart_item_product_tags':
                    $check_id = $variation_id ? wp_get_post_parent_id($variation_id) : $product_id;
                    if (has_term($value, 'product_tag', $check_id)) {
                        $match = true;
                    }
                    break;
                case 'cart_item_product_attributes':
                    // format taxonomy:term_slug
                    $check_id = $variation_id ? $variation_id : $product_id;
                    foreach ((array)$value as $attr_str) {
                        if (strpos($attr_str, ':') !== false) {
                            [$tax, $slug] = explode(':', $attr_str, 2);
                            if (has_term($slug, $tax, $check_id)) {
                                $match = true;
                                break;
                            }
                            // Also check parent if variation?
                            if ($variation_id) {
                                $parent_id = wp_get_post_parent_id($variation_id);
                                if (has_term($slug, $tax, $parent_id)) {
                                    $match = true;
                                    break;
                                }
                            }
                        }
                    }
                    break;
                case 'cart_item_product_sku':
                    $skus = is_array($value) ? $value : preg_split('/[,\s]+/', $value);
                    if (in_array($product->get_sku(), $skus)) {
                        $match = true;
                    }
                    break;
                case 'cart_item_product_onsale':
                    if ($product->is_on_sale()) {
                        $match = true;
                    }
                    break;
            }

            if ($operator === 'not_in') {
                if ($match) {
                    return false; // Excluded
                }
            } else {
                $has_inclusion = true;
                if ($match) {
                    $matched_inclusion = true;
                }
            }
        }

        return $has_inclusion ? $matched_inclusion : true;
    }
}
