<?php
if (! defined('ABSPATH')) exit; // Exit if accessed directly
// Include helper classes
require_once __DIR__ . '/filter.php';
require_once __DIR__ . '/buy_x_get_x.php';
require_once __DIR__ . '/buy_x_get_y.php';
require_once __DIR__ . '/bulk_discounts.php';
require_once __DIR__ . '/set_discounts.php';
require_once __DIR__ . '/free_shipping.php';
require_once __DIR__ . '/simple_discount.php';
require_once __DIR__ . '/cart_discount.php';
class Yeekit_Dynamic_Discounts_Woocommerce_Frontend
{
    protected $is_applying_discounts = false;
    public function __construct()
    {
        // Init Free Shipping
        new Yeekit_Dynamic_Discounts_Free_Shipping();
        add_action('woocommerce_before_calculate_totals', [$this, 'handle_discounts'], 9999);
        add_action('template_redirect', [$this, 'sync_cart_items']);
        add_action('woocommerce_check_cart_items', [$this, 'sync_cart_items']);
        add_action('woocommerce_after_cart_item_quantity_update', [$this, 'sync_cart_items'], 10, 0);
        //remove cart quantity + block
        add_filter('woocommerce_cart_item_quantity', [$this, 'restrict_quantity_input'], 10, 3);
        add_filter('woocommerce_store_api_product_quantity_editable', [$this, 'restrict_quantity_input_block'], 10, 3);
        //remove cart link + block
        add_filter('woocommerce_cart_item_remove_link', [$this, 'restrict_remove_link'], 10, 2);
        //error no remove wc_add_notice
        add_action('woocommerce_remove_cart_item', [$this, 'restrict_remove_link_block'], 10, 2);
        // Display discount info in cart item data
        add_filter('woocommerce_get_item_data', [$this, 'display_discount_meta'], 10, 2);
        // Show strikeout price in cart - handled by Price_Display class now
        // add_filter('woocommerce_cart_item_price', [$this, 'display_strikeout_price'], 10, 3);
        // Handle Fees (Cart Discount)
        add_action('woocommerce_cart_calculate_fees', [$this, 'handle_fees'], 10, 1);
        // Show "You saved: $X" in subtotal
        add_filter('woocommerce_cart_subtotal', [$this, 'display_you_saved_amount'], 999, 3);
        // Show "You saved: $X" in item subtotal
        add_filter('woocommerce_cart_item_subtotal', [$this, 'display_item_saved_amount'], 10, 3);
    }
    public function display_item_saved_amount($product_subtotal, $cart_item, $cart_item_key)
    {
        $product = $cart_item['data'];
        if (!$product) return $product_subtotal;
        $qty = $cart_item['quantity'];
        $price = floatval($product->get_price());
        $regular_price = floatval($product->get_regular_price());
        if ($regular_price > $price) {
            $saved = ($regular_price - $price) * $qty;
            if ($saved > 0) {
                /* translators: %s: Saved amount */
                $product_subtotal .= '<br><small class="woocommerce-cart-item-saved">' . sprintf(__('You saved: %s', 'yeediscounts'), wc_price($saved)) . '</small>';
            }
        }
        return $product_subtotal;
    }
    public function display_you_saved_amount($cart_subtotal, $compound, $cart)
    {
        $total_savings = 0;
        foreach ($cart->get_cart() as $cart_item) {
            $product = $cart_item['data'];
            if (! $product) continue;
            $qty = $cart_item['quantity'];
            $price = floatval($product->get_price());
            $regular_price = floatval($product->get_regular_price());
            // Check if regular price is greater than price (discounted)
            // Sometimes regular price is empty or same as price
            if ($regular_price > $price) {
                $total_savings += ($regular_price - $price) * $qty;
            }
        }
        if ($total_savings > 0) {
            /* translators: %s: Saved amount */
            $saved_html = '<br><small class="woocommerce-cart-subtotal-saved">' . sprintf(__('You saved: %s', 'yeediscounts'), wc_price($total_savings)) . '</small>';
            $cart_subtotal .= $saved_html;
        }
        return $cart_subtotal;
    }
    // ... (rest)
    public function handle_fees($cart)
    {
        if (is_admin() && ! defined('DOING_AJAX')) {
            return;
        }
        // Get matched rule
        // NOTE: get_matched_rule depends on current cart state.
        // During calculate_fees, cart state (subtotal) should be populated.
        $rule = Yeekit_Dynamic_Discounts_Rules_Manager::get_matched_rule();
        if (!$rule) {
            // Clear session if no rule matches
            if (WC()->session) WC()->session->__unset('yeekit_current_cart_rule_id');
            return;
        }
        if ($rule['type'] === 'yeekit_cart_discount') {
            // Store ID for Order creation tracking
            if (WC()->session) WC()->session->set('yeekit_current_cart_rule_id', $rule['id']);
            Yeekit_Dynamic_Discounts_Cart_Discount::handle($cart, $rule);
        } else {
            if (WC()->session) WC()->session->__unset('yeekit_current_cart_rule_id');
        }
    }
    public function display_discount_meta($item_data, $cart_item)
    {
        if (empty($cart_item['_yeekit_free_item'])) {
            return $item_data;
        }
        $meta = $cart_item['_yeekit_free_item'];
        $type = is_array($meta) ? ($meta['type'] ?? 'free_product') : 'free_product';
        if (is_array($meta)) {
            // Use standardized keys if available, fallback to legacy
            $amount = floatval($meta['discount_value'] ?? $meta['value'] ?? $meta['amount'] ?? 0);
            $type   = $meta['discount_type'] ?? $type; // Override type if standardized key exists
        }
        // 2. Discount Type Label
        // Use logic similar to backend/orders.php
        $val_type_labels = [];
        $main_rule_type = $meta['rule_type'] ?? ''; // Assuming rule_type is stored in meta
        if ($main_rule_type === 'yeekit_buy_x_get_x') {
            $val_type_labels = [
                'free_product' => __('Free', 'yeediscounts'),
                'percentage'   => __('Percentage discount', 'yeediscounts'),
                'flat'         => __('Fixed discount', 'yeediscounts'),
            ];
        } elseif ($main_rule_type === 'yeekit_buy_x_get_y') {
            $val_type_labels = [
                'free_product' => __('Free', 'yeediscounts'),
                'percentage'   => __('Percentage discount', 'yeediscounts'),
                'flat'         => __('Fixed discount', 'yeediscounts'),
                'fixed_price'  => __('Fixed Price per item', 'yeediscounts'),
            ];
        } else {
            $val_type_labels = [
                'percentage'      => __('Percentage discount', 'yeediscounts'),
                'fixed'           => __('Fixed discount', 'yeediscounts'),
                'fixed_price'     => __('Fixed price per item', 'yeediscounts'),
                // Fallbacks if rule ID lookup fails or specific BOGO types leak through
                'flat'            => __('Fixed discount', 'yeediscounts'),
                'free_product'    => __('Free', 'yeediscounts'),
            ];
        }
        // $type_label = isset($val_type_labels[$type]) ? $val_type_labels[$type] : $type;
        // $item_data[] = [
        //     'key'   => __('Discount Type', 'yeediscounts'),
        //     'value' => $type_label,
        // ];
        $display_value = '';
        if ($type === 'free_product') {
            $display_value = __('Free', 'yeediscounts');
        } elseif ($type === 'percentage') {
            $display_value = $amount . '%';
        } elseif ($type === 'fixed' || $type === 'flat') {
            $display_value = wc_price($amount);
        } elseif ($type === 'fixed_price') {
            // For fixed_price, show the actual discount amount saved
            $discount_amount = floatval($meta['discount_amount'] ?? 0);
            $display_value = wc_price($discount_amount);
        }
        if ($display_value) {
            $item_data[] = [
                'key'     => __('Discount', 'yeediscounts'),
                'value'   => $display_value,
                'display' => sprintf(
                    '<span class="yeekit-discount-badge">%s</span>',
                    $display_value
                ),
            ];
        }
        return $item_data;
    }
    public function restrict_remove_link_block($cart_item_key, $cart)
    {
        wc_clear_notices();
        if ($this->is_applying_discounts) {
            return $cart_item_key;
        }
        $cart_item = isset($cart->cart_contents[$cart_item_key]) ? $cart->cart_contents[$cart_item_key] : null;
        if (!empty($cart_item['_yeekit_free_item'])) {
            if (!empty($cart_item['_yeekit_free_item']['allow_qty_change'])) {
                return $cart_item_key;
            }
            wc_add_notice(__('This is a bundled product, you cannot remove it from the cart.', 'yeediscounts'), 'error');
        }
    }
    public function restrict_remove_link($link, $cart_item_key)
    {
        $cart_item = WC()->cart->get_cart_item($cart_item_key);
        if (!empty($cart_item['_yeekit_free_item'])) {
            if (!empty($cart_item['_yeekit_free_item']['allow_qty_change'])) {
                return $link;
            }
            return '';
        }
        return $link;
    }
    public function restrict_quantity_input($product_quantity, $cart_item_key, $cart_item)
    {
        if (!empty($cart_item['_yeekit_free_item'])) {
            // Check if allowed to change
            if (!empty($cart_item['_yeekit_free_item']['allow_qty_change'])) {
                return $product_quantity;
            }
            return sprintf('<span class="yeekit-free-qty">%s</span>', $cart_item['quantity']);
        }
        return $product_quantity;
    }
    public function restrict_quantity_input_block($value, $product, $cart_item)
    {
        if (!empty($cart_item['_yeekit_free_item'])) {
            if (!empty($cart_item['_yeekit_free_item']['allow_qty_change'])) {
                return $value;
            }
            return false;
        }
        return $value;
    }
    public function sync_cart_items()
    {
        if (is_admin() && ! wp_doing_ajax()) {
            return;
        }
        if (!WC()->cart) {
            return;
        }
        // Get matched rule
        $rule = Yeekit_Dynamic_Discounts_Rules_Manager::get_matched_rule();
        if (!$rule) {
            return;
        }
        // Only handle specific types that need strict sync (BXGY Auto Add)
        if ($rule['type'] === 'yeekit_buy_x_get_y') {
            Yeekit_Dynamic_Discounts_Buy_X_Get_Y::handle_sync(WC()->cart, $rule);
        }
    }
    public function handle_discounts($cart)
    {
        if (is_admin() && ! defined('DOING_AJAX')) {
            return;
        }
        // 0. Clean stale meta from previous calculations
        // Only clean items that are "Price Adjustments" (allow_qty_change).
        // Do NOT clean "Free Products" (BOGO/BXGY locked items), otherwise scan_cart fails to identify them.
        foreach ($cart->get_cart() as $key => $val) {
            if (isset($val['_yeekit_free_item']) && !empty($val['_yeekit_free_item']['allow_qty_change'])) {
                unset($cart->cart_contents[$key]['_yeekit_free_item']);
            }
        }
        // 1. Check Coupon Behavior Setting
        $coupon_behavior = get_option('yk_dd_disable_coupon_when_rule_applied', 'run_both');
        $has_coupons = !empty($cart->get_applied_coupons());
        // Case: Disable Rules if Coupons are present
        if ($coupon_behavior === 'disable_rules' && $has_coupons) {
            // Clean up any existing Yeekit metadata from cart items to prevent badges/notices
            foreach ($cart->get_cart() as $key => $val) {
                if (isset($val['_yeekit_free_item'])) {
                    unset($cart->cart_contents[$key]['_yeekit_free_item']);
                }
                // Also clean original price backup if exists
                if (isset($val['yeekit_original_price'])) {
                    unset($cart->cart_contents[$key]['yeekit_original_price']);
                }
            }
            return;
        }
        // Get ALL matched rules (not just one)
        $matched_rules = Yeekit_Dynamic_Discounts_Rules_Manager::get_all_matched_rules();
        //var_dump($matched_rules);
        if (empty($matched_rules)) {
            return;
        }
        // Case: Disable Coupons if Rules are Matched
        if ($coupon_behavior === 'disable_coupon' && $has_coupons) {
            // Remove all coupons
            $cart->remove_coupons();
            // Clear "Coupon applied successfully" messages
            wc_clear_notices();
            // Note: remove_coupons() triggers calculate_totals(), which calls this function again.
            // We return here to let the next pass handle the rule application (now with no coupons).
            return;
        }
        // Group rules by type
        $rules_by_type = [];
        foreach ($matched_rules as $rule) {
            $type = $rule['type'] ?? 'unknown';
            if (!isset($rules_by_type[$type])) {
                $rules_by_type[$type] = [];
            }
            $rules_by_type[$type][] = $rule;
        }
        // Get setting for how to apply discount (only for rules of same type)
        $apply_mode = get_option('yk_dd_apply_product_discount_to', 'biggest_discount');
        // For each rule type, apply appropriate rule(s)
        //var_dump($rules_by_type);
        foreach ($rules_by_type as $type => $rules) {
            $rule_to_apply = null;
            // If only one rule of this type, apply it
            if (count($rules) === 1) {
                $rule_to_apply = $rules[0];
            } else {
                // Multiple rules of same type - use setting to decide
                if ($apply_mode === 'first') {
                    $rule_to_apply = $rules[0];
                } else {
                    // For 'all', 'biggest_discount', 'lowest_discount'
                    // We iterate ALL rules and let the individual Rule Handler decide whether to apply/overwrite
                    // based on the $apply_mode passed to it.
                    foreach ($rules as $each_rule) {
                        $this->apply_rule($cart, $each_rule, $apply_mode);
                    }
                    continue;
                }
            }
            // Apply the selected rule for this type
            if ($rule_to_apply) {
                $this->apply_rule($cart, $rule_to_apply, $apply_mode);
            }
        }
    }
    /**
     * Apply a single discount rule
     */
    private function apply_rule($cart, $rule, $apply_mode = 'biggest_discount')
    {
        $this->is_applying_discounts = true;
        // Route to specific discount handler
        switch ($rule['type']) {
            case 'yeekit_buy_x_get_x':
                Yeekit_Dynamic_Discounts_Buy_X_Get_X::handle($cart, $rule);
                break;
            case 'yeekit_buy_x_get_y':
                Yeekit_Dynamic_Discounts_Buy_X_Get_Y::handle($cart, $rule);
                break;
            case 'yeekit_bulk_discount':
                Yeekit_Dynamic_Discounts_Bulk_Discounts::handle($cart, $rule);
                break;
            case 'yeekit_set_discount':
                Yeekit_Dynamic_Discounts_Set_Discounts::handle($cart, $rule);
                break;
            case 'yeekit_simple_discount':
                Yeekit_Dynamic_Discounts_Simple_Discount::handle($cart, $rule, $apply_mode);
                break;
            case 'yeekit_cart_discount':
                // Do nothing here, handled in handle_fees
                break;
        }
        $this->is_applying_discounts = false;
    }
}
new Yeekit_Dynamic_Discounts_Woocommerce_Frontend();
