<?php
if (! defined('ABSPATH')) {
    exit; // Exit if accessed directly
}
class Yeekit_Dynamic_Discounts_Discount_Bar
{
    public function __construct()
    {
        add_action('wp', [$this, 'init']);
    }

    public function init()
    {
        add_action('wp_enqueue_scripts', [$this, 'enqueue_styles']);

        // 1. Configured Position (Targeting Product Page)
        $position = get_option('yk_dd_position_to_show_discount_bar', 'woocommerce_before_add_to_cart_form');
        if ($position) {
            add_action($position, [$this, 'render']);
        }

        // 2. Cart & Checkout Positions (For BOGO as requested)
        add_action('woocommerce_before_cart', [$this, 'render']);
        add_action('woocommerce_before_checkout_form', [$this, 'render']);
    }

    public function enqueue_styles()
    {
        wp_enqueue_style(
            'yeekit-dd-frontend',
            YEEKIT_DYNAMIC_DISCOUNTS_WOOCOMMERCE_URL . 'assets/frontend.css',
            [],
            YEEKIT_DYNAMIC_DISCOUNTS_VERSION
        );
        wp_enqueue_script(
            'yeekit-dd-frontend',
            YEEKIT_DYNAMIC_DISCOUNTS_WOOCOMMERCE_URL . 'assets/frontend.js',
            ['jquery'],
            YEEKIT_DYNAMIC_DISCOUNTS_VERSION,
            true
        );
    }

    public function render()
    {
        // Check Context
        $is_product_page = is_product() && ($product = wc_get_product());
        $is_cart_checkout = is_cart() || is_checkout();

        if (! $is_product_page && ! $is_cart_checkout) {
            return;
        }

        // Check Cart Notification Setting (Global Toggle)
        if ($is_cart_checkout) {
            $show_notifications = get_option('yk_dd_show_cart_notifications', 'yes');
            if ($show_notifications !== 'yes') {
                return;
            }
        }

        // Check Coupon Behavior Setting
        $coupon_behavior = get_option('yk_dd_disable_coupon_when_rule_applied', 'run_both');
        $has_coupons = (WC()->cart && !empty(WC()->cart->get_applied_coupons()));

        if ($coupon_behavior === 'disable_rules' && $has_coupons) {
            return;
        }

        // 1. Try to get a fully matched rule
        $matched_rule = Yeekit_Dynamic_Discounts_Rules_Manager::get_matched_rule();
        $potential_match = null;

        // 2. If no full match, check for potential match (Progress)
        // Only check progress if we are NOT on product page (context matters)
        // Actually, valid to show "Buy more" on product page too if the user is logged in or cart has items?
        // Logic in rules.php checks CART total/quantity.
        if (!$matched_rule) {
            $potential_match = Yeekit_Dynamic_Discounts_Rules_Manager::get_potential_matches();
        }

        $rule_to_display = $matched_rule ?: $potential_match;

        if (!$rule_to_display) {
            return;
        }

        // Determine State
        $is_success = !empty($matched_rule);
        $is_progress = !empty($potential_match);

        if ($is_product_page && empty($rule_to_display['discount_bar_enabled'])) {
            return;
        }

        // Prepare Message & Variables
        $message = $rule_to_display['discount_bar_message'] ?? '';

        $title = $rule_to_display['title'] ?? '';
        $replacements = [
            '{discount_title}' => $title,
        ];

        if ($is_progress && isset($rule_to_display['progress'])) {
            $progress = $rule_to_display['progress']; // ['type', 'missing', 'current', 'required']
            if ($progress['type'] === 'amount') {
                $replacements['{remaining_amount}'] = wc_price($progress['missing']);
                $replacements['{remaining_qty}'] = '';
            } elseif ($progress['type'] === 'qty') {
                $replacements['{remaining_amount}'] = '';
                $replacements['{remaining_qty}'] = $progress['missing'];
            }
        } else {
            $replacements['{remaining_amount}'] = '';
            $replacements['{remaining_qty}'] = '';
        }

        $message = str_replace(array_keys($replacements), array_values($replacements), $message);

        // Scenario A: Product Page
        if ($is_product_page) {
            // Validate if this product is even eligible for this rule (Basic Filter Check)
            // For potential match, we already did basic filter check in get_potential_matches
            // For matched rule, we need to check if THIS product is part of it?
            // matched_rule is "Whole Cart" rule usually.

            // Re-use existing filter check for Product Page context relevance
            $cart_item = [
                'product_id'   => $product->get_id(),
                'variation_id' => 0,
                'data'         => $product,
                'quantity'     => 1
            ];

            // Only show if the product itself is relevant to the rule
            if (Yeekit_Dynamic_Discounts_Filter::validate_item($cart_item, $rule_to_display)) {
                $bg_color = !empty($rule_to_display['discount_bar_bg_color']) ? $rule_to_display['discount_bar_bg_color'] : ($is_progress ? '#f39c12' : 'transparent'); // Orange for progress
                $text_color = !empty($rule_to_display['discount_bar_text_color']) ? $rule_to_display['discount_bar_text_color'] : '#333333';

                printf(
                    '<div class="yeekit-discount-bar %4$s" style="--yk-bar-bg: %1$s; --yk-bar-text: %2$s;">
                        <span class="yeekit-bar-icon">%5$s</span>
                        <span class="yeekit-bar-message">%3$s</span>
                    </div>',
                    esc_attr($bg_color),
                    esc_attr($text_color),
                    wp_kses_post($message),
                    $is_progress ? 'yeekit-bar-progress' : 'yeekit-bar-success',
                    $is_progress ? '&#9888;' : '&#10003;' // Warning icon vs Checkmark
                );
            }
        }
        // Scenario B: Cart or Checkout Page
        elseif ($is_cart_checkout) {

            if ($is_success) {
                // Fallback for empty message if user didn't config rule message
                if (empty($message) || $message == __('Discount {discount_title}', 'yeediscounts')) {
                    $custom_text = get_option('yk_dd_cart_notification_text', 'Discount {discount_title} has been applied to your cart.');
                    if (!empty($custom_text)) {
                        $message = str_replace('{discount_title}', $title, $custom_text);
                    }
                }
            }

            if (!empty($message)) {
                wc_print_notice($message, $is_progress ? 'notice' : 'success');
            }
        }
    }
}
new Yeekit_Dynamic_Discounts_Discount_Bar();
