<?php
if (! defined('ABSPATH')) exit; // Exit if accessed directly

class Yeekit_Dynamic_Discounts_Cart_Discount
{
    public static function handle($cart, $rule)
    {
        $discount_settings = $rule['discount'] ?? [];
        $type = $discount_settings['type'] ?? 'percentage';
        $value = floatval($discount_settings['value'] ?? 0);

        if ($value <= 0) {
            return;
        }

        // Calculate discount amount
        $discount_amount = 0;
        $subtotal = $cart->get_subtotal(); // This hook runs in calculate_fees, so subtotal should be available (excl tax depending on settings)

        // If subtotal is 0 or not set yet (unlikely in calculate_fees unless empty cart), check cart contents
        if (empty($subtotal)) {
            $subtotal = 0;
            foreach ($cart->get_cart() as $item) {
                $subtotal += $item['line_subtotal'];
            }
        }

        if ($type === 'percentage') {
            $discount_amount = $subtotal * ($value / 100);
        } elseif ($type === 'fixed') {
            $discount_amount = $value;
        } elseif ($type === 'fixed_per_item') {
            // Calculate value * quantity for each matched item
            foreach ($cart->get_cart() as $cart_item) {
                // Must validate if item matches filters (if any)
                // If no filters set, it usually applies to all. 
                // Yeekit_Dynamic_Discounts_Filter::validate_item checks exclude/include logic.
                if (Yeekit_Dynamic_Discounts_Filter::validate_item($cart_item, $rule)) {
                    $discount_amount += $value * $cart_item['quantity'];
                }
            }
        } elseif ($type === 'fixed_price') {
            // Fixed Price Per Item:
            // Calculate difference between Current Price and Target Price for each matched item.
            foreach ($cart->get_cart() as $cart_item) {
                if (Yeekit_Dynamic_Discounts_Filter::validate_item($cart_item, $rule)) {
                    $current_price = floatval($cart_item['data']->get_price());
                    // Discount is Difference (Current - Target). If Target > Current, no discount (or negative? No, usually just 0).
                    $diff = $current_price - $value;
                    if ($diff > 0) {
                        $discount_amount += $diff * $cart_item['quantity'];
                    }
                }
            }
        }

        if ($discount_amount > 0) {
            // Apply as Fee (Negative Fee)
            // Use Custom Title if set, otherwise Rule Title
            $custom_title = $rule['discount']['title'] ?? '';
            $label = !empty($custom_title) ? $custom_title : (!empty($rule['title']) ? $rule['title'] : __('Discount', 'yeediscounts'));

            // Ensure we don't discount more than the subtotal?
            // add_fee handles negative values. 
            // WC usually limits total to 0. 

            $cart->add_fee($label, -$discount_amount, false, '');
            // 3rd arg: tax status (false = no tax on fee? or fee is untaxed?)
            // Usually discount reduces taxable layout. 
            // If the cart is taxable, the discount should probably remove tax proportionally?
            // For simplicity, let's treat it as a non-taxable negative fee for now, or match standard behavior.
            // If I set tax=false, it just reduces total.
        }
    }
}
