<?php
if (! defined('ABSPATH')) exit; // Exit if accessed directly

class Yeekit_Dynamic_Discounts_Bxgy_Table
{
    public function __construct()
    {
        add_action('wp', [$this, 'init']);
    }

    public function init()
    {
        // 1. Check display settings (Use shared Bulk setting)
        $show_table = get_option('yk_dd_show_bulk_discount_table', 'yes');
        if ($show_table !== 'yes') {
            return;
        }

        // 2. Get display position
        $position = get_option('yk_dd_position_to_show_bulk_table', 'woocommerce_before_add_to_cart_form');
        if ($position) {
            add_action($position, [$this, 'render_table']);
        }
    }

    public function render_table()
    {
        if (! is_product()) {
            return;
        }

        global $product;
        if (! $product instanceof WC_Product) {
            return;
        }

        // 3. Find BXGY rule where this product acts as "BUY ITEM"
        $matched_rule = null;
        $rules = Yeekit_Dynamic_Discounts_Rules_Manager::get_active_rules();

        foreach ($rules as $rule) {
            if (empty($rule['type']) || $rule['type'] !== 'yeekit_buy_x_get_y') {
                continue;
            }

            // Create fake item to check if this product activates the rule
            $cart_item = [
                'product_id' => $product->get_id(),
                'variation_id' => 0,
                'data' => $product,
                'quantity' => 1 // Quantity not important when validating filter, just check filter
            ];

            if (Yeekit_Dynamic_Discounts_Filter::validate_item($cart_item, $rule)) {
                $matched_rule = $rule;
                break; // Get first rule
            }
        }

        if (! $matched_rule || empty($matched_rule['buyx_gety_adjustments']['ranges'])) {
            return;
        }

        // 4. Display table
        $ranges = $matched_rule['buyx_gety_adjustments']['ranges'];
        $bxgy_type = $matched_rule['buyx_gety_adjustments']['type'] ?? 'bxgy_product';

        // Sort ascending by min_qty
        usort($ranges, function ($a, $b) {
            return ($a['min_qty'] ?? 0) <=> ($b['min_qty'] ?? 0);
        });

?>

        <div class="yeekit-bulk-table-wrapper yeekit-bxgy-table-wrapper">
            <h3><?php esc_html_e('Buy X Get Y', 'yeediscounts'); ?></h3>

            <table id="yeekit-bxgy-table" class="yeekit-bulk-table-msg yeekit-table">
                <thead class="yeekit-bulk-table-thead">
                    <tr class="yeekit-bulk-table-tr yeekit-bulk-table-thead">
                        <th class="yeekit-bulk-table-td">
                            <span><?php esc_html_e('Buy Quantity', 'yeediscounts'); ?></span>
                        </th>
                        <th class="yeekit-bulk-table-td">
                            <span><?php esc_html_e('Get Quantity', 'yeediscounts'); ?></span>
                        </th>
                        <th class="yeekit-bulk-table-td">
                            <span><?php esc_html_e('Reward', 'yeediscounts'); ?></span>
                        </th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($ranges as $index => $range) :
                        $min = $range['min_qty'] ?? 1;
                        $max = $range['max_qty'] ?? '';
                        $buy_display = $min . ($max ? ' - ' . $max : '+');

                        $get_qty = intval($range['get_qty'] ?? 1);

                        $discount_type = $range['discount_type'] ?? 'free_product';
                        $discount_val = floatval($range['discount_value'] ?? 0);

                        // Build "Reward" text
                        $reward_text = '';
                        $reward_desc = '';

                        // 1. DISCOUNT Part
                        if ($discount_type === 'free_product') {
                            $reward_text = __('Free', 'yeediscounts');
                        } elseif ($discount_type === 'percentage') {
                            $reward_text = $discount_val . '% ' . __('Off', 'yeediscounts');
                        } elseif ($discount_type === 'flat') {
                            $reward_text = wc_price($discount_val) . ' ' . __('Off', 'yeediscounts');
                        }

                        // 2. PRODUCT RECEIVED Part
                        $items_list = [];
                        if ($bxgy_type === 'bxgy_all') {
                            $items_list[] = __('Any Product', 'yeediscounts');
                        } elseif ($bxgy_type === 'bxgy_category') {
                            $cat_ids = $range['get_product'] ?? []; // For category type, this is category ID
                            foreach ((array)$cat_ids as $cid) {
                                $term = get_term($cid, 'product_cat');
                                if ($term && !is_wp_error($term)) {
                                    $link = get_term_link($term);
                                    if (!is_wp_error($link)) {
                                        $items_list[] = '<a href="' . esc_url($link) . '" target="_blank">' . esc_html($term->name) . '</a>';
                                    } else {
                                        $items_list[] = esc_html($term->name);
                                    }
                                }
                            }
                        } else {
                            // Product
                            $p_ids = $range['get_product'] ?? [];
                            // If product count is too large, implies "Selected Products"
                            if (count((array)$p_ids) > 3) {
                                $items_list[] = __('Selected Products', 'yeediscounts');
                            } else {
                                foreach ((array)$p_ids as $pid) {
                                    $p = wc_get_product($pid);
                                    if ($p) {
                                        $items_list[] = '<a href="' . esc_url($p->get_permalink()) . '" target="_blank">' . esc_html($p->get_name()) . '</a>';
                                    }
                                }
                            }
                        }

                        if (!empty($items_list)) {
                            $reward_desc = implode(', ', $items_list);
                        }

                    ?>
                        <tr class="yeekit-bulk-table-tr yeekit-bulk-table-row">
                            <!-- Buy Qty -->
                            <td class="yeekit-bulk-table-td col_index_1">
                                <?php echo esc_html($buy_display); ?>
                            </td>

                            <!-- Get Qty -->
                            <td class="yeekit-bulk-table-td col_index_2">
                                <?php echo esc_html($get_qty); ?>
                            </td>

                            <!-- Reward -->
                            <td class="yeekit-bulk-table-td col_index_3">
                                <span class="yeekit-bxgy-reward-main"><?php echo wp_kses_post($reward_text); ?></span>
                                <?php if ($reward_desc): ?>
                                    <div class="yeekit-bxgy-reward-desc" style="font-size: 0.85em; opacity: 0.8;">
                                        (<?php echo wp_kses_post($reward_desc); ?>)
                                    </div>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
<?php
    }
}

new Yeekit_Dynamic_Discounts_Bxgy_Table();
