<?php
if (! defined('ABSPATH')) exit; // Exit if accessed directly

class Yeekit_Dynamic_Discounts_Buy_X_Get_Y
{
    private static $is_syncing = false;

    /**
     * Handle sync (Add/Remove product) before calculation
     */
    public static function handle_sync($cart, $rule)
    {
        if (self::$is_syncing) {
            return;
        }

        $settings = $rule['buyx_gety_adjustments'] ?? [];
        $apply_mode = $settings['apply_mode'] ?? 'auto_add';

        // Only auto_add mode needs strict sync
        if ($apply_mode !== 'auto_add') {
            return;
        }

        try {
            self::$is_syncing = true;

            // Auto-Add logic only works when type is Specific Product (bxgy_product)
            // Because cannot auto-add "Category" or "All" to cart.
            $bxgy_type = $rule['buyx_gety_adjustments']['type'] ?? 'bxgy_product';
            if ($bxgy_type !== 'bxgy_product') {
                return;
            }

            // 2. Tính toán quyền lợi (Unified Logic)
            $benefit = self::calculate_global_benefit($cart, $rule);

            $total_get_qty = 0;
            $allowed_products = [];
            $discount_type = 'free_product';
            $discount_value = 0;

            if ($benefit['entitlement'] > 0 && !empty($benefit['range'])) {
                $total_get_qty = $benefit['entitlement'];
                $range = $benefit['range'];
                $allowed_products = !empty($range['get_product']) ? (array) $range['get_product'] : [];
                $discount_type = $range['discount_type'] ?? 'free_product';
                $discount_value = $range['discount_value'] ?? 0;
            }

            // 3. Scan ALL existing free items for this specific rule
            // This is critical to clean up items if the rule no longer matches (total_get_qty becomes 0)
            $existing_free_items = [];
            foreach ($cart->get_cart() as $key => $item) {
                if (!empty($item['_yeekit_free_item']['rule_id']) && $item['_yeekit_free_item']['rule_id'] == ($rule['id'] ?? 0)) {
                    $existing_free_items[] = ['key' => $key, 'item' => $item, 'product_id' => $item['product_id']];
                }
            }

            // 4. CLEANUP: If total entitlement is 0 (Rule not matched), remove all free items for this rule
            if ($total_get_qty <= 0) {
                foreach ($existing_free_items as $entry) {
                    $cart->remove_cart_item($entry['key']);
                }
                return;
            }

            if (empty($allowed_products)) {
                return; // Matched but no products configured?
            }

            // 5. SYNC: Loop allowed products to Add/Update
            // Note: Logic here is complex if multiple allowed products are selected (Receive A AND B? Or Choose?)
            // The original logic looped all allowed products and tried to maintain 'total_get_qty' for EACH?
            // Let's stick to original logic: "Loop through each allowed product to support Combo behavior".
            // Warning: This implies if I have [A, B] and entitlement 2, I get 2 A AND 2 B?
            // Or is 'total_get_qty' shared? Original logic used 'total_get_qty' independently for each p_id loop.
            // We will preserve that behavior unless user complains.

            foreach ($allowed_products as $p_id) {
                $p_id = intval($p_id);
                if ($p_id <= 0) continue;

                // Count existing free qty for THIS product
                $current_free_qty = 0;
                $free_item_keys = [];
                foreach ($existing_free_items as $entry) {
                    if ($entry['product_id'] == $p_id) {
                        $current_free_qty += $entry['item']['quantity'];
                        $free_item_keys[] = $entry['key'];
                    }
                }

                // ACTION: DELETE EXCESS (or if product not allowed anymore? we only loop allowed)
                if ($current_free_qty > $total_get_qty) {
                    $remove_qty = $current_free_qty - $total_get_qty;
                    foreach ($free_item_keys as $key) {
                        if ($remove_qty <= 0) break;
                        $item_qty = $cart->cart_contents[$key]['quantity'];
                        if ($item_qty > $remove_qty) {
                            $cart->set_quantity($key, $item_qty - $remove_qty);
                            $remove_qty = 0;
                        } else {
                            $cart->remove_cart_item($key);
                            $remove_qty -= $item_qty;
                        }
                    }
                }
                // ACTION: ADD MISSING
                elseif ($current_free_qty < $total_get_qty) {
                    // Check if we already have this product as Paid item?
                    // Original logic checked global quantity of p_id to prevent double-dipping or over-gifting?
                    // Line 86 original: loops cart for p_id.

                    $existing_total_qty = 0;
                    foreach ($cart->get_cart() as $key => $item) {
                        if ($item['product_id'] == $p_id) {
                            $existing_total_qty += $item['quantity'];
                        }
                    }

                    // If total (Paid + Free) >= Entitlement, do nothing?
                    // This logic implies "Make sure user has X items of B".
                    if ($existing_total_qty >= $total_get_qty) {
                        continue;
                    }

                    $needed = $total_get_qty - $existing_total_qty;

                    if ($needed > 0) {
                        $original_price = 0;
                        // Try to get product price to store original
                        $_product = wc_get_product($p_id);
                        if ($_product) {
                            $original_price = floatval($_product->get_price());
                        }

                        $cart->add_to_cart($p_id, $needed, 0, [], [
                            '_yeekit_free_item' => [
                                'rule_id' => $rule['id'] ?? 0,
                                'rule_type' => 'yeekit_buy_x_get_y',
                                'discount_type' => $discount_type,
                                'discount_value' => $discount_value,
                                'type' => $discount_type,
                                'amount' => $discount_value,
                                'original_price' => $original_price,
                                'allow_qty_change' => false // Revert qty lock
                            ]
                        ]);
                    }
                }
            }
        } finally {
            self::$is_syncing = false;
        }
    }

    /**
     * Điểm vào chính để xử lý BXGY (Giai đoạn định giá)
     */
    public static function handle($cart, $rule)
    {
        $settings = $rule['buyx_gety_adjustments'] ?? [];

        // 1, 2, 3. Tính quyền lợi Unified
        $benefit = self::calculate_global_benefit($cart, $rule);

        if ($benefit['entitlement'] <= 0 || empty($benefit['range'])) {
            return;
        }

        $range = $benefit['range'];
        $entitlement = $benefit['entitlement'];
        $buy_qty = $benefit['buy_qty']; // Total buy qty or ref

        // 3a. Clean up invalid gifts (e.g. range switch)
        // If we switched from Range A (Get Product X) to Range B (Get Product Y),
        // we must remove Product X if it's no longer allowed.
        self::cleanup_invalid_gifts($cart, $rule, $range);

        // 4. Cập nhật giỏ hàng (Tự động thêm / Đồng bộ)
        self::process_cart_benefits($cart, $rule, $range, $entitlement, $settings, $buy_qty);

        // 5. Enforce Pricing Adjustment (Robustness)
        // Ensure ALL free items get their price set, regardless of how they were added.
        self::apply_pricing($cart);
    }

    protected static function cleanup_invalid_gifts($cart, $rule, $range)
    {
        $rule_id = $rule['id'] ?? 0;
        $allowed_ids = !empty($range['get_product']) ? (array) $range['get_product'] : [];
        $allowed_ids = array_map('intval', $allowed_ids);

        // If 'get_product' is empty (e.g. Percentage off ANYTHING?), we skip cleanup?
        // No, if get_product is empty, maybe it allows ALL? 
        // Need to check 'bxgy_type'.
        // Assuming 'bxgy_product' uses 'get_product'. 
        // If type is category, we should check category. 
        // For now, let's strictly handle the "Product List" case which is the reported bug.

        $bxgy_type = $rule['buyx_gety_adjustments']['type'] ?? 'bxgy_product';
        if ($bxgy_type !== 'bxgy_product') return;

        // Scan cart
        foreach ($cart->get_cart() as $key => $item) {
            // Check if this item belongs to OUR rule
            if (empty($item['_yeekit_free_item'])) continue;
            if (($item['_yeekit_free_item']['rule_id'] ?? 0) != $rule_id) continue;

            // Check if current product is still allowed
            $p_id = $item['product_id'];
            $v_id = $item['variation_id'];

            $is_allowed = false;
            if (in_array($p_id, $allowed_ids) || ($v_id && in_array($v_id, $allowed_ids))) {
                $is_allowed = true;
            }

            if (!$is_allowed) {
                // Remove stale gift
                $cart->remove_cart_item($key);
            }
        }
    }

    protected static function count_buy_items($cart, $rule)
    {
        $operator = $rule['buyx_gety_adjustments']['operator'] ?? 'product_cumulative';
        $total_qty = 0;

        foreach ($cart->get_cart() as $cart_item_key => $cart_item) {
            // Bỏ qua hàng tặng
            if (! empty($cart_item['_yeekit_free_item'])) {
                continue;
            }
            // Kiểm tra tính đủ điều kiện
            if (Yeekit_Dynamic_Discounts_Filter::validate_item($cart_item, $rule)) {
                $total_qty += $cart_item['quantity'];
            }
        }

        return ['qty' => $total_qty];
    }

    protected static function get_matched_range($ranges, $buy_qty)
    {
        if (empty($ranges)) return null;

        // Sắp xếp giảm dần theo min_qty
        usort($ranges, function ($a, $b) {
            return ($b['min_qty'] ?? 0) <=> ($a['min_qty'] ?? 0);
        });

        foreach ($ranges as $range) {
            $min = intval($range['min_qty'] ?? 1);
            $max = ! empty($range['max_qty']) ? intval($range['max_qty']) : null;

            // Kiểm tra xem số lượng có đủ không
            if ($buy_qty >= $min) {
                // Nếu max được đặt, hãy kiểm tra nó, TRỪ KHI nó đệ quy
                $is_recursive = ! empty($range['is_recursive']);

                if (! $is_recursive && $max !== null && $buy_qty > $max) {
                    continue;
                }

                return $range; // Tìm thấy kết quả phù hợp nhất
            }
        }
        return null; // Không tìm thấy kết quả phù hợp
    }

    protected static function calculate_entitlement($buy_qty, $range)
    {
        $min_qty = max(1, intval($range['min_qty'] ?? 1));
        $get_qty = intval($range['get_qty'] ?? 1);
        $is_recursive = ! empty($range['is_recursive']);

        if ($is_recursive) {
            $sets = floor($buy_qty / $min_qty);
            return $sets * $get_qty;
        }

        return $get_qty;
    }

    protected static function process_cart_benefits($cart, $rule, $range, $total_get_qty, $settings, $buy_qty = 0)
    {

        $apply_mode = $settings['apply_mode'] ?? 'cheapest';
        $bxgy_type = $rule['buyx_gety_adjustments']['type'] ?? 'bxgy_product';
        $allowed_ids = ! empty($range['get_product']) ? (array) $range['get_product'] : [];
        // Cast IDs to int for robust comparison (especially for has_term)
        $allowed_ids = array_map('intval', $allowed_ids);

        $discount_type  = $range['discount_type'] ?? 'free_product';
        $raw_value = $range['discount_value'] ?? 0;
        $discount_value = floatval(str_replace(',', '.', $raw_value));

        // 1. Thu thập các ứng viên (Targets)
        $targets = [];

        foreach ($cart->get_cart() as $key => $item) {
            // Safety Check
            if (empty($item) || !isset($item['product_id'])) continue;

            // Nếu đã được tặng bởi rule khác -> Bỏ qua
            if (!empty($item['_yeekit_free_item']) && ($item['_yeekit_free_item']['rule_id'] != ($rule['id'] ?? 0))) {
                continue;
            }

            // Kiểm tra khớp (Match)
            $is_match = false;
            if ($bxgy_type === 'bxgy_product') {
                if (in_array($item['product_id'], $allowed_ids) || ($item['variation_id'] && in_array($item['variation_id'], $allowed_ids))) {
                    $is_match = true;
                }
            } elseif ($bxgy_type === 'bxgy_category') {
                $check_id = $item['variation_id'] ? wp_get_post_parent_id($item['variation_id']) : $item['product_id'];
                if (has_term($allowed_ids, 'product_cat', $check_id)) {
                    $is_match = true;
                }
            } elseif ($bxgy_type === 'bxgy_all') {
                // All luôn match
                $is_match = true;
            }

            if ($is_match) {
                // Kiểm tra overlap: Sản phẩm này có phải tính là "Buy Item" không?
                // Nếu có, nó chịu hạn mức overlap.
                $is_overlap = Yeekit_Dynamic_Discounts_Filter::validate_item($item, $rule);

                $targets[] = [
                    'key' => $key,
                    'item' => $item,
                    'price' => floatval($item['data']->get_price()),
                    'quantity' => $item['quantity'],
                    'is_overlap' => $is_overlap
                ];
            }
        }

        if (empty($targets)) return;

        // 2. Sắp xếp Targets theo chế độ (Cheapest/Highest)
        // Mặc định Auto Add cũng dùng logic Cheapest cho việc định giá
        usort($targets, function ($a, $b) use ($apply_mode) {
            if ($apply_mode === 'highest') {
                return $b['price'] <=> $a['price'];
            }
            // Cheapest
            return $a['price'] <=> $b['price'];
        });

        // 3. Tính toán Hạn mức (Quotas)
        $remaining_total_quota = $total_get_qty;

        // Tính Overlap Quota (Giới hạn cho các sp vừa mua vừa tặng)
        $remaining_overlap_quota = $remaining_total_quota;

        // Logic tính Overlap Cap:
        // Nếu sp là overlap, nó bị giới hạn bởi công thức tự tài trợ.
        $min_buy_qty = max(1, intval($range['min_qty'] ?? 1));
        $get_unit_qty = intval($range['get_qty'] ?? 1);
        $is_recursive = ! empty($range['is_recursive']);

        if ($is_recursive) {
            // Ratio = Get / (Buy + Get)
            $ratio = $get_unit_qty / ($min_buy_qty + $get_unit_qty);
            $overlap_cap = floor($buy_qty * $ratio);
        } else {
            // Non-recursive: Max = Buy - Min
            $overlap_cap = max(0, $buy_qty - $min_buy_qty);
            // Cap không được vượt quá Get Unit (vì non-recursive chỉ tặng 1 lần set)
            $overlap_cap = min($overlap_cap, $get_unit_qty);
        }
        $remaining_overlap_quota = $overlap_cap;

        // 4. Áp dụng giảm giá
        foreach ($targets as $target) {

            if ($remaining_total_quota <= 0) break;

            $key = $target['key'];
            $item = $target['item'];
            $qty_available = $target['quantity'];
            $is_overlap = $target['is_overlap'];
            // Xác định số lượng được free cho item này
            $qty_can_free = $qty_available;

            // Giới hạn bởi Total Quota
            $qty_can_free = min($qty_can_free, $remaining_total_quota);
            // Nếu là overlap, giới hạn bởi Overlap Quota

            if ($is_overlap) {
                if ($remaining_overlap_quota <= 0) {
                    // Nếu hết quota overlap, item này không được free (dù còn total quota cho items non-overlap)
                    // Tuy nhiên cần cẩn thận: Nếu item này ĐÃ có meta free, cần xóa?
                    // Logic hiện tại chỉ thêm/update. Việc xóa meta dư thừa nên xử lý riêng hoặc giả định session clear.
                    continue;
                }
                $qty_can_free = min($qty_can_free, $remaining_overlap_quota);
            }

            if ($qty_can_free <= 0) continue;


            // Xử lý áp dụng
            if ($qty_can_free < $qty_available) {
                // Cần tách (Split)
                $cart->set_quantity($key, $qty_available - $qty_can_free);

                $variation_id = $item['variation_id'] ?? 0;
                $variation = $item['variation'] ?? [];
                $cart_item_data = $item; // Copy data
                unset($cart_item_data['data']); // Remove dynamic objects

                $original_price = floatval($item['data']->get_price());

                // Set meta
                $cart_item_data['_yeekit_free_item'] = [
                    'rule_id' => $rule['id'] ?? 0,
                    'rule_type' => 'yeekit_buy_x_get_y',
                    'discount_type' => $discount_type,
                    'discount_value' => $discount_value,
                    'type' => $discount_type,
                    'amount' => $discount_value,
                    'original_price' => $original_price, // Store Original Price
                    'allow_qty_change' => true
                ];

                // Add new item
                $product_id = $item['product_id'];
                $new_key = $cart->add_to_cart($product_id, $qty_can_free, $variation_id, $variation, $cart_item_data);

                if ($new_key && isset($cart->cart_contents[$new_key])) {
                    // Update meta for new item with fresh original price if needed, but handled above
                    self::apply_price_adjustment($cart->cart_contents[$new_key], $discount_type, $discount_value);
                }
            } else {
                $original_price = floatval($item['data']->get_price());

                // Free toàn bộ item này
                $cart->cart_contents[$key]['_yeekit_free_item'] = [
                    'rule_id' => $rule['id'] ?? 0,
                    'rule_type' => 'yeekit_buy_x_get_y',
                    'discount_type' => $discount_type,
                    'discount_value' => $discount_value,
                    'type' => $discount_type,
                    'amount' => $discount_value,
                    'original_price' => $original_price, // Store Original Price
                    'allow_qty_change' => false
                ];
                self::apply_price_adjustment($cart->cart_contents[$key], $discount_type, $discount_value);
            }

            // Trừ quota
            $remaining_total_quota -= $qty_can_free;
            if ($is_overlap) {
                $remaining_overlap_quota -= $qty_can_free;
            }
        }
    }

    protected static function apply_price_adjustment(&$cart_item, $discount_type, $discount_value)
    {
        $product = $cart_item['data'];
        // Use stored original price if available to avoid compounding
        $meta = $cart_item['_yeekit_free_item'] ?? [];
        $original_price = isset($meta['original_price']) ? floatval($meta['original_price']) : floatval($product->get_price());

        $new_price = $original_price;

        if ($discount_type === 'free_product') {
            $new_price = 0;
        } elseif ($discount_type === 'percentage') {
            $percent = floatval($discount_value);
            if ($percent > 100) $percent = 100;
            if ($percent < 0) $percent = 0;

            $new_price = $original_price * (1 - ($percent / 100));
        } elseif ($discount_type === 'flat') {
            $flat = floatval($discount_value);
            $new_price = max(0, $original_price - $flat);
        } elseif ($discount_type === 'fixed_price') {
            // Fixed Price per item
            $fixed = floatval($discount_value);
            $new_price = max(0, $fixed);
        }

        $cart_item['data']->set_price($new_price);
    }

    protected static function apply_pricing($cart)
    {
        foreach ($cart->get_cart() as $cart_item) {
            if (!empty($cart_item['_yeekit_free_item'])) {
                $meta = $cart_item['_yeekit_free_item'];
                $type = $meta['discount_type'] ?? 'free_product';
                $value = floatval($meta['discount_value'] ?? 0);

                self::apply_price_adjustment($cart_item, $type, $value);
            }
        }
    }

    protected static function calculate_global_benefit($cart, $rule)
    {
        $settings = $rule['buyx_gety_adjustments'] ?? [];
        $operator = $settings['operator'] ?? 'product_cumulative';
        $ranges = $settings['ranges'] ?? [];

        $total_entitlement = 0;
        $best_range = null;
        $total_buy_qty = 0;

        if ($operator === 'product') {
            // Individual Mode: Calc per item
            foreach ($cart->get_cart() as $cart_item) {
                // Skip free items
                if (!empty($cart_item['_yeekit_free_item'])) continue;

                if (Yeekit_Dynamic_Discounts_Filter::validate_item($cart_item, $rule)) {
                    $qty = $cart_item['quantity'];
                    $total_buy_qty += $qty;

                    // Match range for THIS item specific qty
                    $range = self::get_matched_range($ranges, $qty);
                    if ($range) {
                        $entitlement = self::calculate_entitlement($qty, $range);
                        $total_entitlement += $entitlement;
                        // Keep reference to a valid range (e.g., for discount type/value)
                        // If multiple ranges differ in reward, this is a limitation, we pick the last valid one.
                        $best_range = $range;
                    }
                }
            }
        } else {
            // Cumulative Mode (Default & Variations): Sum first, then check
            $buy_data = self::count_buy_items($cart, $rule);
            $total_buy_qty = $buy_data['qty'];

            if ($total_buy_qty > 0) {
                $best_range = self::get_matched_range($ranges, $total_buy_qty);
                if ($best_range) {
                    $total_entitlement = self::calculate_entitlement($total_buy_qty, $best_range);
                }
            }
        }

        return [
            'entitlement' => $total_entitlement,
            'range' => $best_range,
            'buy_qty' => $total_buy_qty
        ];
    }
}
